"""Hyperbolic AI action provider.

This module provides actions for interacting with Hyperbolic AI services.
It includes functionality for text, image and audio generation.
"""

import uuid
from typing import Any

from ...action_decorator import create_action
from ..action_provider import ActionProvider
from .schemas import (
    GenerateAudioSchema,
    GenerateImageSchema,
    GenerateTextSchema,
)
from .service import AIService
from .types import (
    AudioGenerationRequest,
    ChatCompletionRequest,
    ChatMessage,
    ImageGenerationRequest,
)
from .utils import save_base64_data, save_text


class AIActionProvider(ActionProvider):
    """Action provider for generating text, images and audio via AI."""

    description = "Action provider for generating text, images and audio via AI."

    def __init__(
        self,
        api_key: str | None = None,
    ):
        """Initialize the AI action provider.

        Args:
            api_key: The API key for authentication.

        """
        super().__init__("hyperbolic_ai", [], api_key=api_key)
        self.ai_service = AIService(api_key=self.api_key)

    @create_action(
        name="generate_text",
        description="""
This tool generates text using specified language model.

Required inputs:
- prompt: Text prompt for generation.
- model: (Optional) Model to use for text generation.
    Default: "meta-llama/Meta-Llama-3-70B-Instruct"

Example successful response:
    Text generation successful:
    - Saved to: /path/to/generated_text_{uuid}.txt

    Preview (first 500 chars):

Example error response:
    Error: Invalid model specified
    Error: API request failed

Important notes:
- The prompt should be clear and specific
- Response length depends on the prompt and model
- The generated text is saved to a file with a UUID in the name
""",
        schema=GenerateTextSchema,
    )
    def generate_text(self, args: dict[str, Any]) -> str:
        """Generate text using specified language model.

        Args:
            args (dict[str, Any]): Input arguments for the action.

        Returns:
            str: A message containing the action response or error details.

        """
        try:
            validated_args = GenerateTextSchema(**args)

            messages = []
            if validated_args.system_prompt:
                messages.append(ChatMessage(role="system", content=validated_args.system_prompt))
            messages.append(ChatMessage(role="user", content=validated_args.prompt))

            request = ChatCompletionRequest(
                messages=messages,
                model=validated_args.model,
            )

            response = self.ai_service.generate_text(request)

            if not response.choices or len(response.choices) == 0:
                return "Error: Generation failed: No text was generated by the model."

            generated_text = response.choices[0].message.content

            filename = f"./tmp/generated_text_{uuid.uuid4()}.txt"
            file_path = save_text(generated_text, filename)

            preview = generated_text[:500] + "..." if len(generated_text) > 500 else generated_text

            output = [
                "Text generation successful:",
                f"- Saved to: {file_path}",
                "",
                "Preview (first 500 chars):",
                preview,
            ]

            return "\n".join(output)
        except Exception as e:
            return f"Error: Text generation: {e!s}"

    @create_action(
        name="generate_image",
        description="""
This tool generates images using specified model.

Required inputs:
- prompt: The image prompt to generate from
- model_name: (Optional) The model to use (default: "SDXL1.0-base")
- height: (Optional) Image height in pixels (default: 1024)
- width: (Optional) Image width in pixels (default: 1024)
- steps: (Optional) Number of inference steps (default: 30)
- num_images: (Optional) Number of images to generate (default: 1)
- negative_prompt: (Optional) What to avoid in the image

Example successful response:
    Image generation successful:
    - Saved to: /path/to/generated_image_{uuid}.png

Example error response:
    Error: Invalid model specified
    Error: API request failed

Important notes:
- The prompt should be descriptive and specific
- Images are saved as PNG files
- The image files include a UUID in the filename
""",
        schema=GenerateImageSchema,
    )
    def generate_image(self, args: dict[str, Any]) -> str:
        """Generate images using specified model.

        Args:
            args (dict[str, Any]): Input arguments for the action.

        Returns:
            str: A message containing the action response or error details.

        """
        try:
            validated_args = GenerateImageSchema(**args)

            request = ImageGenerationRequest(
                prompt=validated_args.prompt,
                model_name=validated_args.model_name,
                height=validated_args.height,
                width=validated_args.width,
                steps=validated_args.steps,
                num_images=validated_args.num_images,
                negative_prompt=validated_args.negative_prompt,
            )

            response = self.ai_service.generate_image(request)

            file_paths = []
            for img in response.images:
                filename = f"./tmp/generated_image_{uuid.uuid4()}.png"
                file_path = save_base64_data(img.image, filename)
                file_paths.append(file_path)

            if not file_paths:
                return "Error: Generation failed: No images were generated."

            output = ["Image generation successful:"]
            for path in file_paths:
                output.append(f"- Saved to: {path}")

            return "\n".join(output)

        except Exception as e:
            return f"Error: Image generation: {e!s}"

    @create_action(
        name="generate_audio",
        description="""
This tool generates audio from text using specified language and speaker.

Required inputs:
- text: The text to convert to speech
- language: (Optional) The language code (default: "EN")
- speaker: (Optional) The speaker voice (default: "EN-US")
- speed: (Optional) Speaking speed multiplier (0.1-5.0)

Example successful response:
    Audio generation successful:
    - Saved to: /path/to/generated_audio_{uuid}.mp3

Example error response:
    Error: Invalid language specified
    Error: API request failed

Important notes:
- The text should be in the specified language
- Audio is saved as MP3 file
- The audio files include a UUID in the filename
""",
        schema=GenerateAudioSchema,
    )
    def generate_audio(self, args: dict[str, Any]) -> str:
        """Generate audio from text using specified language and speaker.

        Args:
            args (dict[str, Any]): Input arguments for the action.

        Returns:
            str: A message containing the action response or error details.

        """
        try:
            if isinstance(args, str):
                args = {"text": args}

            validated_args = GenerateAudioSchema(**args)

            request = AudioGenerationRequest(
                text=validated_args.text,
                language=validated_args.language,
                speaker=validated_args.speaker,
                speed=validated_args.speed,
            )

            response = self.ai_service.generate_audio(request)

            filename = f"./tmp/generated_audio_{uuid.uuid4()}.mp3"
            file_path = save_base64_data(response.audio, filename)

            output = [
                "Audio generation successful:",
                f"- Saved to: {file_path}",
            ]

            return "\n".join(output)

        except Exception as e:
            return f"Error: Audio generation: {e!s}"


def ai_action_provider(
    api_key: str | None = None,
) -> AIActionProvider:
    """Create a new instance of the AIActionProvider.

    Args:
        api_key: Optional API key for authentication. If not provided,
                will attempt to read from HYPERBOLIC_API_KEY environment variable.

    Returns:
        A new AI action provider instance.

    Raises:
        ValueError: If API key is not provided and not found in environment.

    """
    return AIActionProvider(api_key=api_key)
