<div align="center">

```
██╗███╗   ██╗████████╗███████╗ ██████╗ ██████╗ ██╗██╗   ██╗███╗   ███╗
██║████╗  ██║╚══██╔══╝██╔════╝██╔════╝ ██╔══██╗██║██║   ██║████╗ ████║
██║██╔██╗ ██║   ██║   █████╗  ██║  ███╗██████╔╝██║██║   ██║██╔████╔██║
██║██║╚██╗██║   ██║   ██╔══╝  ██║   ██║██╔══██╗██║██║   ██║██║╚██╔╝██║
██║██║ ╚████║   ██║   ███████╗╚██████╔╝██║  ██║██║╚██████╔╝██║ ╚═╝ ██║
╚═╝╚═╝  ╚═══╝   ╚═╝   ╚══════╝ ╚═════╝ ╚═╝  ╚═╝╚═╝ ╚═════╝ ╚═╝     ╚═╝
```

### 🔐 *Advanced Data Validation Framework* 🔐

<p align="center">
  <strong>The Integrity Element - Where Data Meets Trust</strong>
</p>

[![PyPI version](https://img.shields.io/pypi/v/integrium?color=purple&style=for-the-badge)](https://pypi.org/project/integrium/)
[![Python Versions](https://img.shields.io/pypi/pyversions/integrium?style=for-the-badge)](https://pypi.org/project/integrium/)
[![License](https://img.shields.io/badge/License-MIT-green.svg?style=for-the-badge)](https://opensource.org/licenses/MIT)
[![Downloads](https://img.shields.io/pypi/dm/integrium?style=for-the-badge&color=orange)](https://pepy.tech/project/integrium)

[![Type Checked](https://img.shields.io/badge/type%20checked-mypy-blue?style=for-the-badge)](http://mypy-lang.org/)
[![Code style: black](https://img.shields.io/badge/code%20style-black-000000.svg?style=for-the-badge)](https://github.com/psf/black)

<p align="center">
  <a href="#-quick-start">Quick Start</a> •
  <a href="#-features">Features</a> •
  <a href="#-examples">Examples</a> •
  <a href="#-documentation">Documentation</a>
</p>

![Separator](https://user-images.githubusercontent.com/74038190/212284100-561aa473-3905-4a80-b561-0d28506553ee.gif)

</div>

## 🌟 What is INTEGRIUM?

**INTEGRIUM** is a modern, powerful data validation framework that goes **beyond** simple schema validation. It's built on **Pydantic** but adds intelligent sanitization, custom validators, and a developer-friendly API that makes data integrity **effortless**.

```python
from integrium import Validator, Field, sanitize

class User(Validator):
    name: str = Field(min_length=2, max_length=50)
    email: str = Field(pattern=r'^[\w\.-]+@[\w\.-]+\.\w+$')
    age: int = Field(ge=0, le=150)
    
user = User(name="  John Doe  ", email="john@example.com", age=30)
# ✅ Automatically validated and sanitized!
```

---

## ✨ Key Features

<table>
<tr>
<td width="50%">

### 🛡️ Validation Power
- ✅ **Pydantic Integration** - All Pydantic features included
- 📧 **Email Validation** - RFC-compliant email checking
- 🔢 **Numeric Constraints** - Min, max, ranges
- 📝 **String Validation** - Length, patterns, formats
- 📅 **Date/Time** - ISO formats, timezones

</td>
<td width="50%">

### 🧹 Smart Sanitization
- 🔤 **String Cleaning** - Trim, normalize whitespace
- 🔡 **Case Normalization** - Upper, lower, title case
- 🚫 **XSS Protection** - HTML/script stripping
- 🌐 **URL Validation** - Format checking
- 💾 **Data Coercion** - Type conversion

</td>
</tr>
<tr>
<td width="50%">

### 🎯 Developer Experience
- 💡 **Full Type Hints** - IDE autocomplete
- 📚 **Rich Error Messages** - Know exactly what failed
- 🔌 **Extensible** - Custom validators easy to add
- ⚡ **Fast** - Built on Pydantic's Rust core

</td>
<td width="50%">

### 🏗️ Production Ready
- ✅ **Tested** - Comprehensive test suite
- 📖 **Documented** - Every feature explained
- 🔒 **Secure** - Input sanitization built-in
- 🌍 **i18n Ready** - Multi-language errors

</td>
</tr>
</table>

---

## 📦 Installation

```bash
# Basic installation
pip install integrium

# With email validation
pip install integrium[email]

# With all features
pip install integrium[all]
```

---

## 🎯 Quick Start

### Basic Validation

```python
from integrium import Validator

class Product(Validator):
    name: str
    price: float
    quantity: int
    
# Valid data
product = Product(name="Laptop", price=999.99, quantity=10)
print(product.name)  # "Laptop"

# Invalid data raises ValidationError
try:
    Product(name="", price=-10, quantity="invalid")
except ValidationError as e:
    print(e.errors())
```

### With Advanced Constraints

```python
from integrium import Validator, Field
from pydantic import EmailStr

class UserProfile(Validator):
    username: str = Field(
        min_length=3,
        max_length=20,
        pattern=r'^[a-zA-Z0-9_]+$'
    )
    email: EmailStr
    age: int = Field(ge=13, le=120)
    bio: str = Field(max_length=500, default="")
    
profile = UserProfile(
    username="john_doe",
    email="john@example.com",
    age=25,
    bio="Python developer"
)
```

### Smart Sanitization

```python
from integrium import sanitize

# String cleaning
text = sanitize("  Hello   World  ")  # "hello world"

# Email normalization  
email = sanitize("John.Doe@EXAMPLE.COM", mode="email")  # "john.doe@example.com"

# URL cleaning
url = sanitize("  https://example.com/page  ", mode="url")  # "https://example.com/page"
```

---

## 🏗️ Architecture

```mermaid
graph LR
    A[Raw Data] --> B[INTEGRIUM]
    B --> C{Validation}
    B --> D{Sanitization}
    
    C --> E[Pydantic Core]
    D --> F[Custom Rules]
    
    E --> G[Type Checking]
    E --> H[Constraints]
    F --> I[String Clean]
    F --> J[XSS Filter]
    
    G --> K[Validated Data]
    H --> K
    I --> K
    J --> K
    
    K --> L[Your Application]
    
    style B fill:#9C27B0
    style K fill:#4CAF50
```

---

## 🔥 Advanced Features

### Custom Validators

```python
from integrium import Validator, field_validator

class Account(Validator):
    username: str
    password: str
    
    @field_validator('password')
    @classmethod
    def validate_password(cls, v):
        if len(v) < 8:
            raise ValueError('Password must be at least 8 characters')
        if not any(c.isupper() for c in v):
            raise ValueError('Password must contain uppercase letter')
        if not any(c.isdigit() for c in v):
            raise ValueError('Password must contain digit')
        return v
```

### Nested Validation

```python
from integrium import Validator
from typing import List

class Address(Validator):
    street: str
    city: str
    country: str
    postal_code: str

class Company(Validator):
    name: str
    employees: int
    addresses: List[Address]

company = Company(
    name="Tech Corp",
    employees=100,
    addresses=[
        {"street": "123 Main St", "city": "NYC", "country": "USA", "postal_code": "10001"},
        {"street": "456 Park Ave", "city": "LA", "country": "USA", "postal_code": "90001"}
    ]
)
```

### Schema Decorator

```python
from integrium import schema

@schema
class APIResponse:
    status: str
    data: dict
    message: str = "Success"
    
# Use as a validator
response = APIResponse(status="ok", data={"user_id": 123})
```

---

## 📊 Comparison with Other Libraries

| Feature | INTEGRIUM | Pydantic | Cerberus | Marshmallow | Voluptuous |
|---------|-----------|----------|----------|-------------|------------|
| Type Hints | ✅ Full | ✅ Full | ❌ No | ⚠️ Partial | ❌ No |
| Sanitization | ✅ Built-in | ❌ Manual | ⚠️ Limited | ⚠️ Limited | ❌ No |
| Async Support | ✅ Yes | ✅ Yes | ❌ No | ❌ No | ❌ No |
| Error Messages | ✅ Rich | ✅ Good | ⚠️ Basic | ✅ Good | ⚠️ Basic |
| Performance | ⚡⚡⚡⚡ | ⚡⚡⚡⚡⚡ | ⚡⚡⚡ | ⚡⚡ | ⚡⚡⚡ |
| Learning Curve | 🟢 Easy | 🟢 Easy | 🟡 Medium | 🟡 Medium | 🟢 Easy |

---

## 🎨 Real-World Examples

### API Request Validation

```python
from integrium import Validator, Field
from fastapi import FastAPI, HTTPException

app = FastAPI()

class CreateUserRequest(Validator):
    username: str = Field(min_length=3, max_length=20)
    email: EmailStr
    password: str = Field(min_length=8)
    
@app.post("/users")
async def create_user(request: CreateUserRequest):
    # Data is already validated!
    return {"message": "User created", "username": request.username}
```

### Form Data Cleaning

```python
from integrium import sanitize

def clean_form_data(form):
    return {
        "name": sanitize(form.get("name", "")),
        "email": sanitize(form.get("email", ""), mode="email"),
        "message": sanitize(form.get("message", ""), mode="xss")
    }
```

### Database Model Validation

```python
from integrium import Validator
from datetime import datetime

class DBUser(Validator):
    id: int
    username: str
    email: str
    created_at: datetime
    is_active: bool = True
    
    class Config:
        from_attributes = True  # Allow ORM models

# Use with SQLAlchemy, Django, etc.
user = DBUser.from_orm(db_user_object)
```

---

## 📚 Documentation

- 📖 [Full API Reference](https://integrium.readthedocs.io)
- 🚀 [Quick Start Guide](https://integrium.readthedocs.io/quickstart)
- 🎓 [Tutorial](https://integrium.readthedocs.io/tutorial)
- 🎯 [Best Practices](https://integrium.readthedocs.io/best-practices)

---

## 🗺️ Roadmap

### ✅ Version 0.1.0 (Current)
- [x] Pydantic integration
- [x] Basic sanitization
- [x] Field validation
- [x] Custom validators

### 🚧 Version 0.2.0 (Coming Soon)
- [ ] Async validation
- [ ] i18n error messages
- [ ] JSON Schema export
- [ ] More sanitization modes

### 🔮 Version 0.3.0 (Planned)
- [ ] GraphQL schema generation
- [ ] OpenAPI integration
- [ ] Validation caching
- [ ] Performance optimizations

---

## 🤝 Contributing

Contributions welcome! Please see [CONTRIBUTING.md](CONTRIBUTING.md) for guidelines.

---

## 📜 License

MIT License - see [LICENSE](LICENSE) file for details.

---

## 👤 Author

<div align="center">

### **Juste Elysée MALANDILA**

[![LinkedIn](https://img.shields.io/badge/LinkedIn-0077B5?style=for-the-badge&logo=linkedin&logoColor=white)](https://linkedin.com/in/juste-elysee-malandila)
[![Email](https://img.shields.io/badge/Email-D14836?style=for-the-badge&logo=gmail&logoColor=white)](mailto:justech4dev@gmail.com)
[![GitHub](https://img.shields.io/badge/GitHub-100000?style=for-the-badge&logo=github&logoColor=white)](https://github.com/jdevsky)

*"Building trust through data integrity."* 🔐

</div>

---

<div align="center">

### Made with ❤️ by [Juste Elysée MALANDILA](https://linkedin.com/in/juste-elysee-malandila)

**INTEGRIUM** - *The Integrity Element* 🔐

![Footer](https://capsule-render.vercel.app/api?type=waving&color=gradient&height=100&section=footer)

</div>
