"""Tests for INTEGRIUM sanitizer. Author: Juste Elysée MALANDILA"""
import pytest
from integrium import sanitize, strip_html, normalize_whitespace, truncate


def test_sanitize_default():
    """Test default sanitization."""
    assert sanitize("  HELLO WORLD  ") == "hello world"
    assert sanitize("MixedCase") == "mixedcase"


def test_sanitize_email():
    """Test email sanitization."""
    result = sanitize("John.Doe@EXAMPLE.COM  ", mode="email")
    assert result == "john.doe@example.com"
    
    # Remove comments
    result = sanitize("user@example.com (Comment)", mode="email")
    assert result == "user@example.com"


def test_sanitize_url():
    """Test URL sanitization."""
    result = sanitize("  example.com/page  ", mode="url")
    assert result == "https://example.com/page"
    
    result = sanitize("http://example.com", mode="url")
    assert result == "http://example.com"


def test_sanitize_xss():
    """Test XSS protection."""
    # HTML escape
    result = sanitize("<script>alert('XSS')</script>", mode="xss")
    assert "script" not in result or "&lt;" in result
    
    # Event handlers removed
    result = sanitize('<div onclick="alert()">Click</div>', mode="xss")
    assert 'onclick=' not in result
    
    # JavaScript protocol removed
    result = sanitize('javascript:alert(1)', mode="xss")
    assert "javascript:" not in result or result == "alert(1)"


def test_sanitize_sql():
    """Test SQL injection prevention."""
    result = sanitize("user'; DROP TABLE users;--", mode="sql")
    assert "DROP" not in result.upper() or "''" in result


def test_sanitize_phone():
    """Test phone number sanitization."""
    result = sanitize("+1 (555) 123-4567", mode="phone")
    assert result == "+15551234567"


def test_sanitize_alphanumeric():
    """Test alphanumeric only."""
    result = sanitize("Hello@World#123!", mode="alphanumeric")
    assert result == "HelloWorld123"


def test_sanitize_alpha():
    """Test letters only."""
    result = sanitize("Hello123World!", mode="alpha")
    assert result == "HelloWorld"


def test_sanitize_numeric():
    """Test numbers only."""
    result = sanitize("Price: $123.45", mode="numeric")
    assert result == "12345"


def test_strip_html():
    """Test HTML stripping."""
    result = strip_html("<p>Hello <strong>World</strong></p>")
    assert result == "Hello World"


def test_normalize_whitespace():
    """Test whitespace normalization."""
    result = normalize_whitespace("Hello    \n\t  World")
    assert result == "Hello World"


def test_truncate():
    """Test string truncation."""
    result = truncate("Hello World", 8)
    assert result == "Hello..."
    assert len(result) == 8
    
    # No truncation needed
    result = truncate("Short", 10)
    assert result == "Short"
