"""Redis Database implementation."""

from typing import Any, Dict, Mapping, Optional, Type

from motor.motor_asyncio import AsyncIOMotorClient, AsyncIOMotorDatabase
from objectdb.database import Database, DatabaseItem, PyObjectId, T, UnknownEntityError


class MongoDBDatabase(Database):
    """MongoDB database implementation."""

    def __init__(self, mongodb_client: AsyncIOMotorClient, name: str) -> None:
        self.connection: AsyncIOMotorClient[Mapping[str, dict[str, Any]]] = mongodb_client
        self.database: AsyncIOMotorDatabase[Mapping[str, dict[str, Any]]] = self.connection[name]

    async def update(self, item: DatabaseItem):
        """Update data."""
        item_type = type(item)
        item.model_validate(item)
        await self.database[item_type.__name__].update_one(
            filter={"_id": item.identifier},
            update={"$set": item.model_dump(by_alias=True, exclude={"_id"})},
            upsert=True,
        )

    async def get(self, schema: Type[T], identifier: PyObjectId) -> T:
        collection = self.database[schema.__name__]
        if res := await collection.find_one(filter={"_id": identifier}):
            return schema.model_validate(res)
        raise UnknownEntityError(f"Unknown identifier: {identifier}")

    async def get_all(self, schema: Type[T]) -> Dict[str, T]:
        raise NotImplementedError

    async def delete(self, schema: Type[T], identifier: PyObjectId, cascade: bool = False) -> None:
        collection = self.database[schema.__name__]
        result = await collection.delete_one(filter={"_id": identifier})
        if result.deleted_count != 1:
            raise UnknownEntityError(f"Unknown identifier: {identifier}")

    async def find(self, schema: Type[T], **kwargs: Any) -> Optional[Dict[PyObjectId, T]]:
        collection = self.database[schema.__name__]
        if results := collection.find(filter=kwargs):
            return {res["_id"]: schema.model_validate(res) async for res in results}
        return None

    async def find_one(self, schema: Type[T], **kwargs: Any) -> Optional[T]:
        """Find one item matching the criteria."""
        collection = self.database[schema.__name__]
        if result := await collection.find_one(filter=kwargs):
            return schema.model_validate(result)
        return None
