# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['langroid',
 'langroid.agent',
 'langroid.agent.special',
 'langroid.agent.special.sql',
 'langroid.agent.special.sql.utils',
 'langroid.agent.tools',
 'langroid.cachedb',
 'langroid.embedding_models',
 'langroid.language_models',
 'langroid.language_models.prompt_formatter',
 'langroid.parsing',
 'langroid.prompts',
 'langroid.utils',
 'langroid.utils.llms',
 'langroid.utils.output',
 'langroid.utils.web',
 'langroid.vector_store']

package_data = \
{'': ['*']}

install_requires = \
['async-generator>=1.10,<2.0',
 'autopep8>=2.0.2,<3.0.0',
 'black[jupyter]>=23.3.0,<24.0.0',
 'bs4>=0.0.1,<0.0.2',
 'chromadb>=0.3.21,<0.4.0',
 'colorlog>=6.7.0,<7.0.0',
 'docstring-parser>=0.15,<0.16',
 'faker>=18.9.0,<19.0.0',
 'fakeredis>=2.12.1,<3.0.0',
 'farm-haystack[file-conversion,ocr,pdf,preprocessing]>=1.21.1,<2.0.0',
 'fire>=0.5.0,<0.6.0',
 'flake8>=6.0.0,<7.0.0',
 'google-api-python-client>=2.95.0,<3.0.0',
 'halo>=0.0.31,<0.0.32',
 'jinja2>=3.1.2,<4.0.0',
 'lancedb>=0.3.0,<0.4.0',
 'litellm>=0.6.2,<0.7.0',
 'lxml>=4.9.3,<5.0.0',
 'meilisearch-python-sdk>=2.0.1,<3.0.0',
 'meilisearch>=0.28.3,<0.29.0',
 'mkdocs-awesome-pages-plugin>=2.8.0,<3.0.0',
 'mkdocs-gen-files>=0.4.0,<0.5.0',
 'mkdocs-jupyter>=0.24.1,<0.25.0',
 'mkdocs-literate-nav>=0.6.0,<0.7.0',
 'mkdocs-material>=9.1.5,<10.0.0',
 'mkdocs-rss-plugin>=1.8.0,<2.0.0',
 'mkdocs-section-index>=0.3.5,<0.4.0',
 'mkdocs>=1.4.2,<2.0.0',
 'mkdocstrings[python]>=0.21.2,<0.22.0',
 'momento>=1.10.2,<2.0.0',
 'mypy>=1.2.0,<2.0.0',
 'nltk>=3.8.1,<4.0.0',
 'openai>=0.27.5,<0.28.0',
 'pandas>=2.0.3,<3.0.0',
 'pdfplumber>=0.10.2,<0.11.0',
 'pre-commit>=3.3.2,<4.0.0',
 'prettytable>=3.8.0,<4.0.0',
 'pydantic==1.10.11',
 'pygithub>=1.58.1,<2.0.0',
 'pygments>=2.15.1,<3.0.0',
 'pymupdf>=1.23.3,<2.0.0',
 'pyparsing>=3.0.9,<4.0.0',
 'pypdf>=3.12.2,<4.0.0',
 'pytest-asyncio>=0.21.1,<0.22.0',
 'python-dotenv>=1.0.0,<2.0.0',
 'qdrant-client>=1.3.1,<2.0.0',
 'rank-bm25>=0.2.2,<0.3.0',
 'redis>=4.5.5,<5.0.0',
 'requests-oauthlib>=1.3.1,<2.0.0',
 'requests>=2.31.0,<3.0.0',
 'rich>=13.3.4,<14.0.0',
 'ruff>=0.0.270,<0.0.271',
 'scrapy>=2.11.0,<3.0.0',
 'sqlalchemy>=2.0.19,<3.0.0',
 'tantivy==0.20.1',
 'thefuzz>=0.20.0,<0.21.0',
 'tiktoken>=0.5.1,<0.6.0',
 'trafilatura>=1.5.0,<2.0.0',
 'typer>=0.9.0,<0.10.0',
 'types-redis>=4.5.5.2,<5.0.0.0',
 'types-requests>=2.31.0.1,<3.0.0.0',
 'unstructured[docx,pdf,pptx]>=0.10.16,<0.10.18',
 'wget>=3.2,<4.0']

extras_require = \
{'hf-embeddings': ['sentence-transformers==2.2.2', 'torch==2.0.0'],
 'mysql': ['pymysql>=1.1.0,<2.0.0', 'pytest-mysql>=2.4.2,<3.0.0'],
 'postgres': ['psycopg2>=2.9.7,<3.0.0', 'pytest-postgresql>=5.0.0,<6.0.0']}

setup_kwargs = {
    'name': 'langroid',
    'version': '0.1.100',
    'description': 'Harness LLMs with Multi-Agent Programming',
    'long_description': '<div align="center">\n  <img src="docs/assets/langroid-card-lambda-ossem-rust-1200-630.png" alt="Logo" \n        width="400" align="center">\n</div>\n\n<div align="center">\n\n[![PyPI - Version](https://img.shields.io/pypi/v/langroid)](https://pypi.org/project/langroid/)\n[![Pytest](https://github.com/langroid/langroid/actions/workflows/pytest.yml/badge.svg)](https://github.com/langroid/langroid/actions/workflows/pytest.yml)\n[![codecov](https://codecov.io/gh/langroid/langroid/branch/main/graph/badge.svg?token=H94BX5F0TE)](https://codecov.io/gh/langroid/langroid)\n[![Lint](https://github.com/langroid/langroid/actions/workflows/validate.yml/badge.svg)](https://github.com/langroid/langroid/actions/workflows/validate.yml)\n[![Docs](https://github.com/langroid/langroid/actions/workflows/mkdocs-deploy.yml/badge.svg)](https://github.com/langroid/langroid/actions/workflows/mkdocs-deploy.yml)\n\n[![Static Badge](https://img.shields.io/badge/Documentation-blue?link=https%3A%2F%2Flangroid.github.io%2Flangroid%2F&link=https%3A%2F%2Flangroid.github.io%2Flangroid%2F)](https://langroid.github.io/langroid)\n[![Discord](https://img.shields.io/badge/Discord-%235865F2.svg?style=for-the-badge&logo=discord&logoColor=white)](https://discord.gg/ZU36McDgDs)\n[![Open in Colab](https://colab.research.google.com/assets/colab-badge.svg)](https://colab.research.google.com/github/langroid/langroid/blob/main/examples/langroid_quick_examples.ipynb)\n\n[![Docker Pulls](https://img.shields.io/docker/pulls/langroid/langroid.svg)](https://hub.docker.com/r/langroid/langroid)\n![Docker Image Size (tag)](https://img.shields.io/docker/image-size/langroid/langroid/latest)\n[![Multi-Architecture DockerHub](https://github.com/langroid/langroid/actions/workflows/docker-publish.yml/badge.svg)](https://github.com/langroid/langroid/actions/workflows/docker-publish.yml)\n\n[![Substack](https://img.shields.io/badge/Substack-%23006f5c.svg?style=for-the-badge&logo=substack&logoColor=FF6719)](https://langroid.substack.com/p/langroid-harness-llms-with-multi-agent-programming)\n\n[![Share on Hacker News](https://img.shields.io/badge/-Share%20on%20Hacker%20News-orange)](https://news.ycombinator.com/submitlink?u=https%3A%2F%2Fgithub.com%2Flangroid%2Flangroid&t=Harness%20LLMs%20with%20Multi-Agent%20Programming)\n[![Share on Reddit](https://img.shields.io/badge/Reddit-FF4500?style=for-the-badge&logo=reddit&logoColor=white)](https://www.reddit.com/submit?url=https%3A%2F%2Fgithub.com%2Flangroid%2Flangroid&title=Harness%20LLMs%20with%20Multi-Agent%20Programming)\n[![Share on Twitter](https://img.shields.io/twitter/url?style=social&url=https://github.com/langroid/langroid)](https://twitter.com/intent/tweet?text=Langroid%20is%20a%20powerful,%20elegant%20new%20framework%20to%20easily%20build%20%23LLM%20applications.%20You%20set%20up%20LLM-powered%20Agents%20with%20vector-stores,%20assign%20tasks,%20and%20have%20them%20collaboratively%20solve%20problems%20via%20message-transformations.%20https://github.com/langroid/langroid)\n[![LinkedIn](https://img.shields.io/badge/linkedin-%230077B5.svg?style=for-the-badge&logo=linkedin&logoColor=white)](https://www.linkedin.com/shareArticle?mini=true&url=https://github.com/langroid/langroid&title=Langroid:%20A%20Powerful,%20Elegant%20Framework&summary=Langroid%20is%20a%20powerful,%20elegant%20new%20framework%20to%20easily%20build%20%23LLM%20applications.%20You%20set%20up%20LLM-powered%20Agents%20with%20vector-stores,%20assign%20tasks,%20and%20have%20them%20collaboratively%20solve%20problems%20via%20message-transformations.)\n\n\n</div>\n\n<h3 align="center">\n  <a target="_blank" \n    href="https://langroid.github.io/langroid/" rel="dofollow">\n      <strong>Documentation</strong></a>\n  &middot;\n  <a target="_blank" href="https://github.com/langroid/langroid-examples" rel="dofollow">\n      <strong>Examples Repo</strong></a>\n  &middot;\n  <a target="_blank" href="https://discord.gg/ZU36McDgDs" rel="dofollow">\n      <strong>Discord</strong></a>\n  &middot;\n  <a target="_blank" href="./CONTRIBUTING.md" rel="dofollow">\n      <strong>Contributing</strong></a>\n\n  <br />\n</h3>\n\n`Langroid` is an intuitive, lightweight, extensible and principled\nPython framework to easily build LLM-powered applications. \nYou set up Agents, equip them with optional components (LLM, \nvector-store and methods), assign them tasks, and have them \ncollaboratively solve a problem by exchanging messages. \nThis Multi-Agent paradigm is inspired by the\n[Actor Framework](https://en.wikipedia.org/wiki/Actor_model)\n(but you do not need to know anything about this!). \n\n`Langroid` is a fresh take on LLM app-development, where considerable thought has gone \ninto simplifying the developer experience; it does not use `Langchain`.\n\nWe welcome contributions -- See the [contributions](./CONTRIBUTING.md) document\nfor ideas on what to contribute.\n\n**Questions, Feedback, Ideas? Join us on [Discord](https://discord.gg/ZU36McDgDs)!**\n\n<details>\n<summary> <b>:fire: Updates/Releases</b></summary>\n\n- **Oct 2023:**\n  - **0.1.99:** Convenience batch functions to run tasks, agent methods on a list of inputs concurrently in async mode. See examples in [test_batch.py](https://github.com/langroid/langroid/blob/main/tests/main/test_batch.py).\n  - **0.1.95:** Added support for [Momento Serverless Vector Index](https://docs.momentohq.com/vector-index)\n  - **0.1.94:** Added support for [LanceDB](https://lancedb.github.io/lancedb/) vector-store -- allows vector, Full-text, SQL search.\n  - **0.1.84:** Added [LiteLLM](https://docs.litellm.ai/docs/providers), so now Langroid can be used with over 100 LLM providers (remote or local)! \n     See guide [here](https://langroid.github.io/langroid/tutorials/non-openai-llms/).\n- **Sep 2023:**\n  - **0.1.78:** Async versions of several Task, Agent and LLM methods; \n      Nested Pydantic classes are now supported for LLM Function-calling, Tools, Structured Output.    \n  - **0.1.76:** DocChatAgent: support for loading `docx` files (preliminary).\n  - **0.1.72:** Many improvements to DocChatAgent: better embedding model, \n          hybrid search to improve retrieval, better pdf parsing, re-ranking retrieved results with cross-encoders. \n  - **Use with local LLama Models:** see tutorial [here](https://langroid.github.io/langroid/blog/2023/09/14/using-langroid-with-local-llms/)\n  - **Langroid Blog/Newsletter Launched!**: First post is [here](https://substack.com/notes/post/p-136704592) -- Please subscribe to stay updated. \n  - **0.1.56:** Support Azure OpenAI. \n  - **0.1.55:** Improved [`SQLChatAgent`](https://github.com/langroid/langroid/blob/main/langroid/agent/special/sql/sql_chat_agent.py) that efficiently retrieves relevant schema info when translating natural language to SQL.  \n- **Aug 2023:**\n  - **[Hierarchical computation](https://langroid.github.io/langroid/examples/agent-tree/)** example using Langroid agents and task orchestration.\n  - **0.1.51:** Support for global state, see [test_global_state.py](tests/main/test_global_state.py).\n  - **:whale: Langroid Docker image**, available, see instructions below.\n  - [**RecipientTool**](langroid/agent/tools/recipient_tool.py) enables (+ enforces) LLM to \nspecify an intended recipient when talking to 2 or more agents. \nSee [this test](tests/main/test_recipient_tool.py) for example usage.\n  - **Example:** [Answer questions](examples/docqa/chat-search.py) using Google Search + vecdb-retrieval from URL contents. \n  - **0.1.39:** [`GoogleSearchTool`](langroid/agent/tools/google_search_tool.py) to enable Agents (their LLM) to do Google searches via function-calling/tools.\n    See [this chat example](examples/basic/chat-search.py) for how easy it is to add this tool to an agent.\n  - **Colab notebook** to try the quick-start examples: [![Open in Colab](https://colab.research.google.com/assets/colab-badge.svg)](https://colab.research.google.com/github/langroid/langroid/blob/main/examples/langroid_quick_examples.ipynb) \n  - **0.1.37:** Added [`SQLChatAgent`](langroid/agent/special/sql_chat_agent.py) -- thanks to our latest contributor [Rithwik Babu](https://github.com/rithwikbabu)!\n  - Multi-agent Example: [Autocorrect chat](examples/basic/autocorrect.py)\n- **July 2023:** \n  - **0.1.30:** Added [`TableChatAgent`](langroid/agent/special/table_chat_agent.py) to \n    [chat](examples/data-qa/table_chat.py) with tabular datasets (dataframes, files, URLs): LLM generates Pandas code,\n    and code is executed using Langroid\'s tool/function-call mechanism. \n  - **Demo:** 3-agent system for Audience [Targeting](https://langroid.github.io/langroid/demos/targeting/audience-targeting/).\n  - **0.1.27**: Added [support](langroid/cachedb/momento_cachedb.py) \n    for [Momento Serverless Cache](https://www.gomomento.com/) as an alternative to Redis.\n  - **0.1.24**: [`DocChatAgent`](langroid/agent/special/doc_chat_agent.py) \n    now [accepts](langroid/parsing/document_parser.py) PDF files or URLs.\n\n</details>\n\n# :rocket: Demo\nSuppose you want to extract structured information about the key terms \nof a commercial lease document. You can easily do this with Langroid using a two-agent system,\nas we show in the [langroid-examples](https://github.com/langroid/langroid-examples/blob/main/examples/docqa/chat_multi_extract.py) repo.\nThe demo showcases just a few of the many features of Langroid, such as:\n- Multi-agent collaboration: `LeaseExtractor` is in charge of the task, and its LLM (GPT4) generates questions \nto be answered by the `DocAgent`.\n- Retrieval augmented question-answering, with **source-citation**: `DocAgent` LLM (GPT4) uses retrieval from a vector-store to \nanswer the `LeaseExtractor`\'s questions, cites the specific excerpt supporting the answer. \n- Function-calling (also known as tool/plugin): When it has all the information it \nneeds, the `LeaseExtractor` LLM presents the information in a structured \nformat using a Function-call. \n\nHere is what it looks like in action \n(a pausable mp4 video is [here](https://vimeo.com/871429249)).\n\n![Demo](docs/assets/demos/lease-extractor-demo.gif)\n\n\n# :zap: Highlights\n\n- **Agents as first-class citizens:** The [Agent](https://langroid.github.io/langroid/reference/agent/base/#langroid.agent.base.Agent) class encapsulates LLM conversation state,\n  and optionally a vector-store and tools. Agents are a core abstraction in Langroid;\n  Agents act as _message transformers_, and by default provide 3 _responder_ methods, one corresponding to each entity: LLM, Agent, User.\n- **Tasks:** A [Task](https://langroid.github.io/langroid/reference/agent/task/) class wraps an Agent, and gives the agent instructions (or roles, or goals), \n  manages iteration over an Agent\'s responder methods, \n  and orchestrates multi-agent interactions via hierarchical, recursive\n  task-delegation. The `Task.run()` method has the same \n  type-signature as an Agent\'s responder\'s methods, and this is key to how \n  a task of an agent can delegate to other sub-tasks: from the point of view of a Task,\n  sub-tasks are simply additional responders, to be used in a round-robin fashion \n  after the agent\'s own responders.\n- **Modularity, Reusabilily, Loose coupling:** The `Agent` and `Task` abstractions allow users to design\n  Agents with specific skills, wrap them in Tasks, and combine tasks in a flexible way.\n- **LLM Support**: Langroid supports OpenAI LLMs as well as LLMs from hundreds of \nproviders (local/open or remote/commercial) via proxy libraries and local model servers\nsuch as [LiteLLM](https://docs.litellm.ai/docs/providers) that in effect mimic the OpenAI API. \n- **Caching of LLM responses:** Langroid supports [Redis](https://redis.com/try-free/) and \n  [Momento](https://www.gomomento.com/) to cache LLM responses.\n- **Vector-stores**: [Qdrant](https://qdrant.tech/) and [Chroma](https://www.trychroma.com/) are currently supported.\n  Vector stores allow for Retrieval-Augmented-Generation (RAG).\n- **Grounding and source-citation:** Access to external documents via vector-stores \n   allows for grounding and source-citation.\n- **Observability, Logging, Lineage:** Langroid generates detailed logs of multi-agent interactions and\n  maintains provenance/lineage of messages, so that you can trace back\n  the origin of a message.\n- **Tools/Plugins/Function-calling**: Langroid supports OpenAI\'s recently\n  released [function calling](https://platform.openai.com/docs/guides/gpt/function-calling)\n  feature. In addition, Langroid has its own native equivalent, which we\n  call **tools** (also known as "plugins" in other contexts). Function\n  calling and tools have the same developer-facing interface, implemented\n  using [Pydantic](https://docs.pydantic.dev/latest/),\n  which makes it very easy to define tools/functions and enable agents\n  to use them. Benefits of using Pydantic are that you never have to write\n  complex JSON specs for function calling, and when the LLM\n  hallucinates malformed JSON, the Pydantic error message is sent back to\n  the LLM so it can fix it!\n\n--- \n\n# :gear: Installation and Setup\n\n### Install `langroid`\nLangroid requires Python 3.11+. We recommend using a virtual environment.\nUse `pip` to install `langroid` (from PyPi) to your virtual environment:\n```bash\npip install langroid\n```\nThe core Langroid package lets you use OpenAI Embeddings models via their API. \nIf you instead want to use the `sentence-transformers` embedding models from HuggingFace, \ninstall Langroid like this: \n```bash\npip install langroid[hf-embeddings]\n```\n\n<details>\n<summary><b>Optional Installs for using SQL Chat with a PostgreSQL DB </b></summary>\n\nIf you are using `SQLChatAgent` \n(e.g. the script [`examples/data-qa/sql-chat/sql_chat.py`](examples/data-qa/sql-chat/sql_chat.py)),\nwith a postgres db, you will need to:\n\n- Install PostgreSQL dev libraries for your platform, e.g.\n  - `sudo apt-get install libpq-dev` on Ubuntu,\n  - `brew install postgresql` on Mac, etc.\n- Install langroid with the postgres extra, e.g. `pip install langroid[postgres]`\n  or `poetry add langroid[postgres]` or `poetry install -E postgres`.\n  If this gives you an error, try `pip install psycopg2-binary` in your virtualenv.\n</details>\n\n### Set up environment variables (API keys, etc)\n\nTo get started, all you need is an OpenAI API Key.\nIf you don\'t have one, see [this OpenAI Page](https://help.openai.com/en/collections/3675940-getting-started-with-openai-api).\nCurrently only OpenAI models are supported. Others will be added later\n(Pull Requests welcome!).\n\nIn the root of the repo, copy the `.env-template` file to a new file `.env`: \n```bash\ncp .env-template .env\n```\nThen insert your OpenAI API Key. \nYour `.env` file should look like this:\n```bash\nOPENAI_API_KEY=your-key-here-without-quotes\n````\n\nAlternatively, you can set this as an environment variable in your shell\n(you will need to do this every time you open a new shell):\n```bash\nexport OPENAI_API_KEY=your-key-here-without-quotes\n```\n\n\n<details>\n<summary><b>Optional Setup Instructions (click to expand) </b></summary>\n\nAll of the following environment variable settings are optional, and some are only needed \nto use specific features (as noted below).\n\n- **Qdrant** Vector Store API Key, URL. This is only required if you want to use Qdrant cloud.\n  You can sign up for a free 1GB account at [Qdrant cloud](https://cloud.qdrant.io).\n  If you skip setting up these, Langroid will use Qdrant in local-storage mode.\n  Alternatively [Chroma](https://docs.trychroma.com/) is also currently supported. \n  We use the local-storage version of Chroma, so there is no need for an API key.\n  Langroid uses Qdrant by default.\n- **Redis** Password, host, port: This is optional, and only needed to cache LLM API responses\n  using Redis Cloud. Redis [offers](https://redis.com/try-free/) a free 30MB Redis account\n  which is more than sufficient to try out Langroid and even beyond.\n  If you don\'t set up these, Langroid will use a pure-python \n  Redis in-memory cache via the [Fakeredis](https://fakeredis.readthedocs.io/en/latest/) library.\n- **Momento** Serverless Caching of LLM API responses (as an alternative to Redis). \n   To use Momento instead of Redis:\n  - enter your Momento Token in the `.env` file, as the value of `MOMENTO_AUTH_TOKEN` (see example file below),\n  - in the `.env` file set `CACHE_TYPE=momento` (instead of `CACHE_TYPE=redis` which is the default).\n- **GitHub** Personal Access Token (required for apps that need to analyze git\n  repos; token-based API calls are less rate-limited). See this\n  [GitHub page](https://docs.github.com/en/authentication/keeping-your-account-and-data-secure/managing-your-personal-access-tokens).\n- **Google Custom Search API Credentials:** Only needed to enable an Agent to use the `GoogleSearchTool`.\n  To use Google Search as an LLM Tool/Plugin/function-call, \n  you\'ll need to set up \n  [a Google API key](https://developers.google.com/custom-search/v1/introduction#identify_your_application_to_google_with_api_key),\n  then [setup a Google Custom Search Engine (CSE) and get the CSE ID](https://developers.google.com/custom-search/docs/tutorial/creatingcse).\n  (Documentation for these can be challenging, we suggest asking GPT4 for a step-by-step guide.)\n  After obtaining these credentials, store them as values of \n  `GOOGLE_API_KEY` and `GOOGLE_CSE_ID` in your `.env` file. \n  Full documentation on using this (and other such "stateless" tools) is coming soon, but \n  in the meantime take a peek at this [chat example](examples/basic/chat-search.py), which \n  shows how you can easily equip an Agent with a `GoogleSearchtool`.\n  \n\n\nIf you add all of these optional variables, your `.env` file should look like this:\n```bash\nOPENAI_API_KEY=your-key-here-without-quotes\nGITHUB_ACCESS_TOKEN=your-personal-access-token-no-quotes\nCACHE_TYPE=redis # or momento\nREDIS_PASSWORD=your-redis-password-no-quotes\nREDIS_HOST=your-redis-hostname-no-quotes\nREDIS_PORT=your-redis-port-no-quotes\nMOMENTO_AUTH_TOKEN=your-momento-token-no-quotes # instead of REDIS* variables\nQDRANT_API_KEY=your-key\nQDRANT_API_URL=https://your.url.here:6333 # note port number must be included\nGOOGLE_API_KEY=your-key\nGOOGLE_CSE_ID=your-cse-id\n```\n</details>\n\n<details>\n<summary><b>Optional setup instructions for Microsoft Azure OpenAI(click to expand)</b></summary> \n\nWhen using Azure OpenAI, additional environment variables are required in the \n`.env` file.\nThis page [Microsoft Azure OpenAI](https://learn.microsoft.com/en-us/azure/ai-services/openai/chatgpt-quickstart?tabs=command-line&pivots=programming-language-python#environment-variables)\nprovides more information, and you can set each environment variable as follows:\n\n- `AZURE_API_KEY`, from the value of `API_KEY`\n- `AZURE_OPENAI_API_BASE` from the value of `ENDPOINT`, typically looks like `https://your.domain.azure.com`.\n- For `AZURE_OPENAI_API_VERSION`, you can use the default value in `.env-template`, and latest version can be found [here](https://learn.microsoft.com/en-us/azure/ai-services/openai/whats-new#azure-openai-chat-completion-general-availability-ga)\n- `AZURE_OPENAI_DEPLOYMENT_NAME` is the name of the deployed model, which is defined by the user during the model setup \n- `AZURE_GPT_MODEL_NAME` GPT-3.5-Turbo or GPT-4 model names that you chose when you setup your Azure OpenAI account.\n\n</details>\n\n---\n\n# :whale: Docker Instructions\n\nWe provide a containerized version of the [`langroid-examples`](https://github.com/langroid/langroid-examples) \nrepository via this [Docker Image](https://hub.docker.com/r/langroid/langroid).\nAll you need to do is set up environment variables in the `.env` file.\nPlease follow these steps to setup the container:\n\n```bash\n# get the .env file template from `langroid` repo\nwget https://github.com/langroid/langroid/blob/main/.env-template .env\n\n# Edit the .env file with your favorite editor (here nano), \n# and add API keys as explained above\nnano .env\n\n# launch the container\ndocker run -it  -v ./.env:/.env langroid/langroid\n\n# Use this command to run any of the scripts in the `examples` directory\npython examples/<Path/To/Example.py> \n``` \n\n\n\n# :tada: Usage Examples\n\nThese are quick teasers to give a glimpse of what you can do with Langroid\nand how your code would look. \n\n:warning: The code snippets below are intended to give a flavor of the code\nand they are **not** complete runnable examples! For that we encourage you to \nconsult the [`langroid-examples`](https://github.com/langroid/langroid-examples) \nrepository.\n\n:information_source: The various LLM prompts and instructions in Langroid\nhave been tested to work well with GPT4.\nSwitching to GPT3.5-Turbo is easy via a config flag\n(e.g., `cfg = OpenAIGPTConfig(chat_model=OpenAIChatModel.GPT3_5_TURBO)`),\nand may suffice for some applications, but in general you may see inferior results.\n\n\n:book: Also see the\n[`Getting Started Guide`](https://langroid.github.io/langroid/quick-start/)\nfor a detailed tutorial.\n\n\n\nClick to expand any of the code examples below.\nAll of these can be run in a Colab notebook:\n[![Open in Colab](https://colab.research.google.com/assets/colab-badge.svg)](https://colab.research.google.com/github/langroid/langroid/blob/main/examples/langroid_quick_examples.ipynb)\n\n<details>\n<summary> <b> Direct interaction with OpenAI LLM </b> </summary>\n\n```python\nfrom langroid.language_models.openai_gpt import ( \n        OpenAIGPTConfig, OpenAIChatModel, OpenAIGPT,\n)\nfrom langroid.language_models.base import LLMMessage, Role\n\ncfg = OpenAIGPTConfig(chat_model=OpenAIChatModel.GPT4)\n\nmdl = OpenAIGPT(cfg)\n\nmessages = [\n  LLMMessage(content="You are a helpful assistant",  role=Role.SYSTEM), \n  LLMMessage(content="What is the capital of Ontario?",  role=Role.USER),\n]\nresponse = mdl.chat(messages, max_tokens=200)\nprint(response.message)\n```\n</details>\n\n<details>\n<summary> <b> Interaction with non-OpenAI LLM (local or remote) </b> </summary>\nLocal model: if model is served at `http://localhost:8000`:\n\n```python\ncfg = OpenAIGPTConfig(\n  chat_model="local/localhost:8000", \n  chat_context_length=4096\n)\nmdl = OpenAIGPT(cfg)\n# now interact with it as above, or create an Agent + Task as shown below.\n```\n\nIf the model is [supported by `liteLLM`](https://docs.litellm.ai/docs/providers), \nthen no need to launch the proxy server.\nJust set the `chat_model` param above to `litellm/[provider]/[model]`, e.g. \n`litellm/anthropic/claude-instant-1` and use the config object as above.\nFor remote models, you will typically need to set API Keys etc as environment variables.\nYou can set those based on the LiteLLM docs. \nIf any required environment variables are missing, Langroid gives a helpful error\nmessage indicating which ones are needed.\n</details>\n\n<details>\n<summary> <b> Define an agent, set up a task, and run it </b> </summary>\n\n```python\nfrom langroid.agent.chat_agent import ChatAgent, ChatAgentConfig\nfrom langroid.agent.task import Task\nfrom langroid.language_models.openai_gpt import OpenAIChatModel, OpenAIGPTConfig\n\nconfig = ChatAgentConfig(\n    llm = OpenAIGPTConfig(\n        chat_model=OpenAIChatModel.GPT4,\n    ),\n    vecdb=None, # no vector store\n)\nagent = ChatAgent(config)\n# get response from agent\'s LLM, and put this in an interactive loop...\n# answer = agent.llm_response("What is the capital of Ontario?")\n  # ... OR instead, set up a task (which has a built-in loop) and run it\ntask = Task(agent, name="Bot") \ntask.run() # ... a loop seeking response from LLM or User at each turn\n```\n</details>\n\n<details>\n<summary><b> Three communicating agents </b></summary>\n\nA toy numbers game, where when given a number `n`:\n- `repeater_agent`\'s LLM simply returns `n`,\n- `even_agent`\'s LLM returns `n/2` if `n` is even, else says "DO-NOT-KNOW"\n- `odd_agent`\'s LLM returns `3*n+1` if `n` is odd, else says "DO-NOT-KNOW"\n\nFirst define the 3 agents, and set up their tasks with instructions:\n\n```python\nfrom langroid.utils.constants import NO_ANSWER\nfrom langroid.agent.chat_agent import ChatAgent, ChatAgentConfig\nfrom langroid.agent.task import Task\nfrom langroid.language_models.openai_gpt import OpenAIChatModel, OpenAIGPTConfig\nconfig = ChatAgentConfig(\n    llm = OpenAIGPTConfig(\n        chat_model=OpenAIChatModel.GPT4,\n    ),\n    vecdb = None,\n)\nrepeater_agent = ChatAgent(config)\nrepeater_task = Task(\n    repeater_agent,\n    name = "Repeater",\n    system_message="""\n    Your job is to repeat whatever number you receive.\n    """,\n    llm_delegate=True, # LLM takes charge of task\n    single_round=False, \n)\neven_agent = ChatAgent(config)\neven_task = Task(\n    even_agent,\n    name = "EvenHandler",\n    system_message=f"""\n    You will be given a number. \n    If it is even, divide by 2 and say the result, nothing else.\n    If it is odd, say {NO_ANSWER}\n    """,\n    single_round=True,  # task done after 1 step() with valid response\n)\n\nodd_agent = ChatAgent(config)\nodd_task = Task(\n    odd_agent,\n    name = "OddHandler",\n    system_message=f"""\n    You will be given a number n. \n    If it is odd, return (n*3+1), say nothing else. \n    If it is even, say {NO_ANSWER}\n    """,\n    single_round=True,  # task done after 1 step() with valid response\n)\n```\nThen add the `even_task` and `odd_task` as sub-tasks of `repeater_task`, \nand run the `repeater_task`, kicking it off with a number as input:\n```python\nrepeater_task.add_sub_task([even_task, odd_task])\nrepeater_task.run("3")\n```\n\n</details>\n\n<details>\n<summary><b> Simple Tool/Function-calling example </b></summary>\n\nLangroid leverages Pydantic to support OpenAI\'s\n[Function-calling API](https://platform.openai.com/docs/guides/gpt/function-calling)\nas well as its own native tools. The benefits are that you don\'t have to write\nany JSON to specify the schema, and also if the LLM hallucinates a malformed\ntool syntax, Langroid sends the Pydantic validation error (suitiably sanitized) \nto the LLM so it can fix it!\n\nSimple example: Say the agent has a secret list of numbers, \nand we want the LLM to find the smallest number in the list. \nWe want to give the LLM a `probe` tool/function which takes a\nsingle number `n` as argument. The tool handler method in the agent\nreturns how many numbers in its list are at most `n`.\n\nFirst define the tool using Langroid\'s `ToolMessage` class:\n\n\n```python\nfrom langroid.agent.tool_message import ToolMessage\nclass ProbeTool(ToolMessage):\n  request: str = "probe" # specifies which agent method handles this tool\n  purpose: str = """\n        To find how many numbers in my list are less than or equal to  \n        the <number> you specify.\n        """ # description used to instruct the LLM on when/how to use the tool\n  number: int  # required argument to the tool\n```\n\nThen define a `SpyGameAgent` as a subclass of `ChatAgent`, \nwith a method `probe` that handles this tool:\n\n```python\nfrom langroid.agent.chat_agent import ChatAgent, ChatAgentConfig\nclass SpyGameAgent(ChatAgent):\n  def __init__(self, config: ChatAgentConfig):\n    super().__init__(config)\n    self.numbers = [3, 4, 8, 11, 15, 25, 40, 80, 90]\n\n  def probe(self, msg: ProbeTool) -> str:\n    # return how many numbers in self.numbers are less or equal to msg.number\n    return str(len([n for n in self.numbers if n <= msg.number]))\n```\n\nWe then instantiate the agent and enable it to use and respond to the tool:\n\n```python\nfrom langroid.language_models.openai_gpt import OpenAIChatModel, OpenAIGPTConfig\nspy_game_agent = SpyGameAgent(\n    ChatAgentConfig(\n        name="Spy",\n        llm = OpenAIGPTConfig(\n            chat_model=OpenAIChatModel.GPT4,\n        ),\n        vecdb=None,\n        use_tools=False, #  don\'t use Langroid native tool\n        use_functions_api=True, # use OpenAI function-call API\n    )\n)\nspy_game_agent.enable_message(ProbeTool)\n```\n\nFor a full working example see the\n[chat-agent-tool.py](https://github.com/langroid/langroid-examples/blob/main/examples/quick-start/chat-agent-tool.py)\nscript in the `langroid-examples` repo.\n</details>\n\n<details>\n<summary> <b>Tool/Function-calling to extract structured information from text </b> </summary>\n\nSuppose you want an agent to extract \nthe key terms of a lease, from a lease document, as a nested JSON structure.\nFirst define the desired structure via Pydantic models:\n\n```python\nfrom pydantic import BaseModel\nclass LeasePeriod(BaseModel):\n    start_date: str\n    end_date: str\n\n\nclass LeaseFinancials(BaseModel):\n    monthly_rent: str\n    deposit: str\n\nclass Lease(BaseModel):\n    period: LeasePeriod\n    financials: LeaseFinancials\n    address: str\n```\n\nThen define the `LeaseMessage` tool as a subclass of Langroid\'s `ToolMessage`.\nNote the tool has a required argument `terms` of type `Lease`:\n\n```python\nclass LeaseMessage(ToolMessage):\n    request: str = "lease_info"\n    purpose: str = """\n        Collect information about a Commercial Lease.\n        """\n    terms: Lease\n```\n\nThen define a `LeaseExtractorAgent` with a method `lease_info` that handles this tool,\ninstantiate the agent, and enable it to use and respond to this tool:\n\n```python\nclass LeaseExtractorAgent(ChatAgent):\n    def lease_info(self, message: LeaseMessage) -> str:\n        print(\n            f"""\n        DONE! Successfully extracted Lease Info:\n        {message.terms}\n        """\n        )\n        return json.dumps(message.terms.dict())\n    \nlease_extractor_agent = LeaseExtractorAgent(\n  ChatAgentConfig(\n    llm=OpenAIGPTConfig(),\n    use_functions_api=False,\n    use_tools=True,\n  )\n)\nlease_extractor_agent.enable_message(LeaseMessage)\n```\n\nSee the [`chat_multi_extract.py`](https://github.com/langroid/langroid-examples/blob/main/examples/docqa/chat_multi_extract.py)\nscript in the `langroid-examples` repo for a full working example.\n</details>\n\n<details>\n<summary><b> Chat with documents (file paths, URLs, etc) </b></summary>\n\nLangroid provides a specialized agent class `DocChatAgent` for this purpose.\nIt incorporates document sharding, embedding, storage in a vector-DB, \nand retrieval-augmented query-answer generation.\nUsing this class to chat with a collection of documents is easy.\nFirst create a `DocChatAgentConfig` instance, with a \n`doc_paths` field that specifies the documents to chat with.\n\n```python\nfrom langroid.agent.doc_chat_agent import DocChatAgentConfig\nfrom langroid.vector_store.qdrantdb import QdrantDBConfig\nconfig = DocChatAgentConfig(\n  doc_paths = [\n    "https://en.wikipedia.org/wiki/Language_model",\n    "https://en.wikipedia.org/wiki/N-gram_language_model",\n    "/path/to/my/notes-on-language-models.txt",\n  ]\n  llm = OpenAIGPTConfig(\n    chat_model=OpenAIChatModel.GPT4,\n  ),\n  vecdb=QdrantDBConfig()\n)\n```\n\nThen instantiate the `DocChatAgent` (this ingests the docs into the vector-store):\n\n```python\nagent = DocChatAgent(config)\n```\nThen we can either ask the agent one-off questions,\n```python\nagent.chat("What is a language model?")\n```\nor wrap it in a `Task` and run an interactive loop with the user:\n```python\nfrom langroid.task import Task\ntask = Task(agent)\ntask.run()\n```\n\nSee full working scripts in the \n[`docqa`](https://github.com/langroid/langroid-examples/tree/main/examples/docqa)\nfolder of the `langroid-examples` repo.\n</details>\n\n<details>\n<summary><b> :fire: Chat with tabular data (file paths, URLs, dataframes) </b></summary>\n\nUsing Langroid you can set up a `TableChatAgent` with a dataset (file path, URL or dataframe),\nand query it. The Agent\'s LLM generates Pandas code to answer the query, \nvia function-calling (or tool/plugin), and the Agent\'s function-handling method\nexecutes the code and returns the answer.\n\nHere is how you can do this:\n\n```python\nfrom langroid.agent.special.table_chat_agent import TableChatAgent, TableChatAgentConfig\nfrom langroid.agent.task import Task\nfrom langroid.language_models.openai_gpt import OpenAIChatModel, OpenAIGPTConfig\n```\n\nSet up a `TableChatAgent` for a data file, URL or dataframe\n(Ensure the data table has a header row; the delimiter/separator is auto-detected):\n```python\ndataset =  "https://archive.ics.uci.edu/ml/machine-learning-databases/wine-quality/winequality-red.csv"\n# or dataset = "/path/to/my/data.csv"\n# or dataset = pd.read_csv("/path/to/my/data.csv")\nagent = TableChatAgent(\n    config=TableChatAgentConfig(\n        data=dataset,  \n        llm=OpenAIGPTConfig(\n            chat_model=OpenAIChatModel.GPT4,\n        ),\n    )\n)\n```\nSet up a task, and ask one-off questions like this: \n\n```python\ntask = Task(\n  agent, \n  name = "DataAssistant",\n  default_human_response="", # to avoid waiting for user input\n)\nresult = task.run(\n  "What is the average alcohol content of wines with a quality rating above 7?",\n  turns=2 # return after user question, LLM fun-call/tool response, Agent code-exec result\n) \nprint(result.content)\n```\nOr alternatively, set up a task and run it in an interactive loop with the user:\n\n```python\ntask = Task(agent, name="DataAssistant")\ntask.run()\n``` \n\nFor a full working example see the \n[`table_chat.py`](https://github.com/langroid/langroid-examples/tree/main/examples/data-qa/table_chat.py)\nscript in the `langroid-examples` repo.\n\n\n</details>\n\n---\n\n# :heart: Thank you to our [supporters](https://github.com/langroid/langroid/stargazers)\n\nIf you like this project, please give it a star ⭐ and 📢 spread the word in your network or social media:\n\n[![Share on Twitter](https://img.shields.io/twitter/url?style=social&url=https://github.com/langroid/langroid)](https://twitter.com/intent/tweet?text=Langroid%20is%20a%20powerful,%20elegant%20new%20framework%20to%20easily%20build%20%23LLM%20applications.%20You%20set%20up%20LLM-powered%20Agents%20with%20vector-stores,%20assign%20tasks,%20and%20have%20them%20collaboratively%20solve%20problems%20via%20message-transformations.%20https://github.com/langroid/langroid)\n[![Share on LinkedIn](https://img.shields.io/badge/Share%20on-LinkedIn-blue)](https://www.linkedin.com/shareArticle?mini=true&url=https://github.com/langroid/langroid&title=Langroid:%20A%20Powerful,%20Elegant%20Framework&summary=Langroid%20is%20a%20powerful,%20elegant%20new%20framework%20to%20easily%20build%20%23LLM%20applications.%20You%20set%20up%20LLM-powered%20Agents%20with%20vector-stores,%20assign%20tasks,%20and%20have%20them%20collaboratively%20solve%20problems%20via%20message-transformations.)\n[![Share on Hacker News](https://img.shields.io/badge/-Share%20on%20Hacker%20News-orange)](https://news.ycombinator.com/submitlink?u=https%3A%2F%2Fgithub.com%2Flangroid%2Flangroid&t=Harness%20LLMs%20with%20Multi-Agent%20Programming)\n[![Share on Reddit](https://img.shields.io/badge/-Share%20on%20Reddit-blue)](https://www.reddit.com/submit?url=https%3A%2F%2Fgithub.com%2Flangroid%2Flangroid&title=Harness%20LLMs%20with%20Multi-Agent%20Programming)\n\n\n\n\nYour support will help build Langroid\'s momentum and community.\n\n\n\n\n# Langroid Co-Founders\n\n- [Prasad Chalasani](https://www.linkedin.com/in/pchalasani/) (IIT BTech/CS, CMU PhD/ML; Independent ML Consultant)\n- [Somesh Jha](https://www.linkedin.com/in/somesh-jha-80208015/) (IIT BTech/CS, CMU PhD/CS; Professor of CS, U Wisc at Madison)\n\n\n\n',
    'author': 'Prasad Chalasani',
    'author_email': 'pchalasani@gmail.com',
    'maintainer': 'None',
    'maintainer_email': 'None',
    'url': 'None',
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'extras_require': extras_require,
    'python_requires': '>=3.9.1,<3.12',
}


setup(**setup_kwargs)
