"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.NagPack = exports.NagMessageLevel = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
/*
Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
SPDX-License-Identifier: Apache-2.0
*/
const fs_1 = require("fs");
const path_1 = require("path");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const nag_rules_1 = require("./nag-rules");
const nag_suppression_helper_1 = require("./utils/nag-suppression-helper");
const VALIDATION_FAILURE_ID = 'CdkNagValidationFailure';
const SUPPRESSION_ID = 'CdkNagSuppression';
/**
 * The level of the message that the rule applies.
 */
var NagMessageLevel;
(function (NagMessageLevel) {
    NagMessageLevel["WARN"] = "Warning";
    NagMessageLevel["ERROR"] = "Error";
})(NagMessageLevel = exports.NagMessageLevel || (exports.NagMessageLevel = {}));
/**
 * Base class for all rule packs.
 */
class NagPack {
    constructor(props) {
        this.reportStacks = new Array();
        this.packName = '';
        this.verbose =
            props == undefined || props.verbose == undefined ? false : props.verbose;
        this.logIgnores =
            props == undefined || props.logIgnores == undefined
                ? false
                : props.logIgnores;
        this.reports =
            props == undefined || props.reports == undefined ? true : props.reports;
    }
    get readPackName() {
        return this.packName;
    }
    get readReportStacks() {
        return this.reportStacks;
    }
    /**
     * Create a rule to be used in the NagPack.
     * @param params The @IApplyRule interface with rule details.
     */
    applyRule(params) {
        if (this.packName === '') {
            throw Error('The NagPack does not have a pack name, therefore the rule could not be applied. Set a packName in the NagPack constructor.');
        }
        const allIgnores = nag_suppression_helper_1.NagSuppressionHelper.getSuppressions(params.node);
        const ruleSuffix = params.ruleSuffixOverride
            ? params.ruleSuffixOverride
            : params.rule.name;
        const ruleId = `${this.packName}-${ruleSuffix}`;
        try {
            const ruleCompliance = params.rule(params.node);
            if (this.reports === true) {
                this.initializeStackReport(params);
                if (ruleCompliance === nag_rules_1.NagRuleCompliance.COMPLIANT) {
                    this.writeToStackComplianceReport(params, ruleId, ruleCompliance);
                }
            }
            if (this.isNonCompliant(ruleCompliance)) {
                const findings = this.asFindings(ruleCompliance);
                for (const findingId of findings) {
                    const suppressionReason = this.ignoreRule(allIgnores, ruleId, findingId);
                    if (this.reports === true) {
                        this.writeToStackComplianceReport(params, ruleId, nag_rules_1.NagRuleCompliance.NON_COMPLIANT, suppressionReason);
                    }
                    if (suppressionReason) {
                        if (this.logIgnores === true) {
                            const message = this.createMessage(SUPPRESSION_ID, findingId, `${ruleId} was triggered but suppressed.`, `Provided reason: "${suppressionReason}"`);
                            aws_cdk_lib_1.Annotations.of(params.node).addInfo(message);
                        }
                    }
                    else {
                        const message = this.createMessage(ruleId, findingId, params.info, params.explanation);
                        if (params.level == NagMessageLevel.ERROR) {
                            aws_cdk_lib_1.Annotations.of(params.node).addError(message);
                        }
                        else if (params.level == NagMessageLevel.WARN) {
                            aws_cdk_lib_1.Annotations.of(params.node).addWarning(message);
                        }
                    }
                }
            }
        }
        catch (error) {
            const reason = this.ignoreRule(allIgnores, VALIDATION_FAILURE_ID, '');
            if (this.reports === true) {
                this.writeToStackComplianceReport(params, ruleId, 'UNKNOWN', reason);
            }
            if (reason) {
                if (this.logIgnores === true) {
                    const message = this.createMessage(SUPPRESSION_ID, '', `${VALIDATION_FAILURE_ID} was triggered but suppressed.`, reason);
                    aws_cdk_lib_1.Annotations.of(params.node).addInfo(message);
                }
            }
            else {
                const information = `'${ruleId}' threw an error during validation. This is generally caused by a parameter referencing an intrinsic function. For more details enable verbose logging.'`;
                const message = this.createMessage(VALIDATION_FAILURE_ID, '', information, error.message);
                aws_cdk_lib_1.Annotations.of(params.node).addWarning(message);
            }
        }
    }
    /**
     * Check whether a specific rule should be ignored.
     * @param ignores The ignores listed in cdk-nag metadata.
     * @param ruleId The id of the rule to ignore.
     * @param findingId The id of the finding that is being checked.
     * @returns The reason the rule was ignored, or an empty string.
     */
    ignoreRule(ignores, ruleId, findingId) {
        for (let ignore of ignores) {
            if (nag_suppression_helper_1.NagSuppressionHelper.doesApply(ignore, ruleId, findingId)) {
                if (!ignore.appliesTo) {
                    // the rule is not granular so it always applies
                    return ignore.reason;
                }
                else {
                    return `[${findingId}] ${ignore.reason}`;
                }
            }
        }
        return '';
    }
    /**
     * The message to output to the console when a rule is triggered.
     * @param ruleId The id of the rule.
     * @param findingId The id of the finding.
     * @param info Why the rule was triggered.
     * @param explanation Why the rule exists.
     * @returns The formatted message string.
     */
    createMessage(ruleId, findingId, info, explanation) {
        let message = findingId
            ? `${ruleId}[${findingId}]: ${info}`
            : `${ruleId}: ${info}`;
        return this.verbose ? `${message} ${explanation}\n` : `${message}\n`;
    }
    /**
     * Write a line to the rule pack's compliance report for the resource's Stack
     * @param params The @IApplyRule interface with rule details.
     * @param ruleId The id of the rule.
     * @param compliance The compliance status of the rule.
     * @param explanation The explanation for suppressed rules.
     */
    writeToStackComplianceReport(params, ruleId, compliance, explanation = '') {
        const line = this.createComplianceReportLine(params, ruleId, compliance, explanation);
        const outDir = aws_cdk_lib_1.App.of(params.node)?.outdir;
        const stackName = params.node.stack.nested
            ? aws_cdk_lib_1.Names.uniqueId(params.node.stack)
            : params.node.stack.stackName;
        const fileName = `${this.packName}-${stackName}-NagReport.csv`;
        const filePath = path_1.join(outDir ? outDir : '', fileName);
        fs_1.appendFileSync(filePath, line);
    }
    /**
     * Initialize the report for the rule pack's compliance report for the resource's Stack if it doesn't exist
     * @param params
     */
    initializeStackReport(params) {
        const stackName = params.node.stack.nested
            ? aws_cdk_lib_1.Names.uniqueId(params.node.stack)
            : params.node.stack.stackName;
        const fileName = `${this.packName}-${stackName}-NagReport.csv`;
        if (!this.reportStacks.includes(fileName)) {
            const outDir = aws_cdk_lib_1.App.of(params.node)?.outdir;
            const filePath = path_1.join(outDir ? outDir : '', fileName);
            this.reportStacks.push(fileName);
            fs_1.writeFileSync(filePath, 'Rule ID,Resource ID,Compliance,Exception Reason,Rule Level,Rule Info\n');
        }
    }
    /**
     * Helper function to create a line for the compliance report
     * @param params The @IApplyRule interface with rule details.
     * @param ruleId The id of the rule.
     * @param compliance The compliance status of the rule.
     * @param explanation The explanation for suppressed rules.
     */
    createComplianceReportLine(params, ruleId, compliance, explanation = '') {
        //| Rule ID | Resource ID | Compliance | Exception Reason | Rule Level | Rule Info
        const line = Array();
        line.push(ruleId);
        line.push(params.node.node.path);
        if ((compliance === nag_rules_1.NagRuleCompliance.NON_COMPLIANT ||
            compliance === 'UNKNOWN') &&
            explanation !== '') {
            line.push('Suppressed');
            line.push(explanation);
        }
        else {
            line.push(compliance);
            line.push('N/A');
        }
        line.push(params.level);
        line.push(params.info);
        return line.map((i) => '"' + i.replace(/"/g, '""') + '"').join(',') + '\n';
    }
    isNonCompliant(ruleResult) {
        return (ruleResult === nag_rules_1.NagRuleCompliance.NON_COMPLIANT ||
            Array.isArray(ruleResult));
    }
    asFindings(ruleResult) {
        if (Array.isArray(ruleResult)) {
            return ruleResult;
        }
        else {
            return [''];
        }
    }
}
exports.NagPack = NagPack;
_a = JSII_RTTI_SYMBOL_1;
NagPack[_a] = { fqn: "cdk-nag.NagPack", version: "2.21.45" };
//# sourceMappingURL=data:application/json;base64,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