"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
/*
Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
SPDX-License-Identifier: Apache-2.0
*/
const path_1 = require("path");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const aws_apigateway_1 = require("aws-cdk-lib/aws-apigateway");
const aws_wafv2_1 = require("aws-cdk-lib/aws-wafv2");
const nag_rules_1 = require("../../nag-rules");
/**
 * Rest API stages are associated with AWS WAFv2 web ACLs
 * @param node the CfnResource to check
 */
exports.default = Object.defineProperty((node) => {
    if (node instanceof aws_apigateway_1.CfnStage) {
        const stageLogicalId = nag_rules_1.NagRules.resolveResourceFromInstrinsic(node, node.ref);
        const stageName = nag_rules_1.NagRules.resolveResourceFromInstrinsic(node, node.stageName);
        const restApiId = nag_rules_1.NagRules.resolveResourceFromInstrinsic(node, node.restApiId);
        let found = false;
        for (const child of aws_cdk_lib_1.Stack.of(node).node.findAll()) {
            if (child instanceof aws_wafv2_1.CfnWebACLAssociation) {
                if (isMatchingWebACLAssociation(child, stageLogicalId, stageName, restApiId)) {
                    found = true;
                    break;
                }
            }
        }
        if (!found) {
            return nag_rules_1.NagRuleCompliance.NON_COMPLIANT;
        }
        return nag_rules_1.NagRuleCompliance.COMPLIANT;
    }
    else {
        return nag_rules_1.NagRuleCompliance.NOT_APPLICABLE;
    }
}, 'name', { value: path_1.parse(__filename).name });
/**
 * Helper function to check whether a given Web ACL Association is associated with the given Rest API
 * @param node the CfnWebACLAssociation to check
 * @param stageLogicalId the Cfn Logical ID of the REST API Stage
 * @param stageName the name of the REST API Stage
 * @param restApiId the ID or Cfn Resource ID of the REST API associated with the Stage
 * returns whether the CfnWebACLAssociation is associates with the given Rest API
 */
function isMatchingWebACLAssociation(node, stageLogicalId, stageName, restApiId) {
    const resourceLogicalId = JSON.stringify(aws_cdk_lib_1.Stack.of(node).resolve(node.resourceArn));
    const regexes = Array();
    regexes.push(`${restApiId}.+${stageLogicalId}(?![\\w])`);
    if (stageName !== undefined) {
        regexes.push(`${restApiId}.+${stageName}(?![\\w])`);
    }
    const regex = new RegExp(regexes.join('|'), 'gm');
    if (regex.test(resourceLogicalId)) {
        return true;
    }
    return false;
}
//# sourceMappingURL=data:application/json;base64,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