"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
/*
Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
SPDX-License-Identifier: Apache-2.0
*/
const path_1 = require("path");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const aws_cloudfront_1 = require("aws-cdk-lib/aws-cloudfront");
const nag_rules_1 = require("../../nag-rules");
/**
 * CloudFront distributions do not use SSLv3 or TLSv1 for communication to the origin
 * @param node the CfnResource to check
 */
exports.default = Object.defineProperty((node) => {
    if (node instanceof aws_cloudfront_1.CfnDistribution) {
        const distributionConfig = aws_cdk_lib_1.Stack.of(node).resolve(node.distributionConfig);
        if (distributionConfig.origins != undefined) {
            const origins = aws_cdk_lib_1.Stack.of(node).resolve(distributionConfig.origins);
            for (const origin of origins) {
                const resolvedOrigin = aws_cdk_lib_1.Stack.of(node).resolve(origin);
                if (resolvedOrigin.customOriginConfig != undefined) {
                    const customOriginConfig = aws_cdk_lib_1.Stack.of(node).resolve(resolvedOrigin.customOriginConfig);
                    const originProtocolPolicy = nag_rules_1.NagRules.resolveIfPrimitive(node, customOriginConfig.originProtocolPolicy);
                    if (originProtocolPolicy != aws_cloudfront_1.OriginProtocolPolicy.HTTPS_ONLY) {
                        return nag_rules_1.NagRuleCompliance.NON_COMPLIANT;
                    }
                    if (customOriginConfig.originSslProtocols == undefined) {
                        return nag_rules_1.NagRuleCompliance.NON_COMPLIANT;
                    }
                    const outdatedValues = [
                        aws_cloudfront_1.SecurityPolicyProtocol.SSL_V3,
                        aws_cloudfront_1.SecurityPolicyProtocol.TLS_V1,
                    ];
                    const originSslProtocols = (aws_cdk_lib_1.Stack.of(node).resolve(customOriginConfig.originSslProtocols));
                    const lowerCaseProtocols = originSslProtocols.map((i) => {
                        return i.toLowerCase();
                    });
                    for (const outdated of outdatedValues) {
                        if (lowerCaseProtocols.includes(outdated.toLowerCase())) {
                            return nag_rules_1.NagRuleCompliance.NON_COMPLIANT;
                        }
                    }
                }
            }
        }
        return nag_rules_1.NagRuleCompliance.COMPLIANT;
    }
    else {
        return nag_rules_1.NagRuleCompliance.NOT_APPLICABLE;
    }
}, 'name', { value: path_1.parse(__filename).name });
//# sourceMappingURL=data:application/json;base64,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