"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
/*
Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
SPDX-License-Identifier: Apache-2.0
*/
const path_1 = require("path");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const aws_cognito_1 = require("aws-cdk-lib/aws-cognito");
const nag_rules_1 = require("../../nag-rules");
/**
 * Cognito user pools have password policies that minimally specify a password length of at least 8 characters, as well as requiring uppercase, numeric, and special characters
 * @param node the CfnResource to check
 */
exports.default = Object.defineProperty((node) => {
    if (node instanceof aws_cognito_1.CfnUserPool) {
        const policies = aws_cdk_lib_1.Stack.of(node).resolve(node.policies);
        if (policies == undefined) {
            return nag_rules_1.NagRuleCompliance.NON_COMPLIANT;
        }
        const passwordPolicy = aws_cdk_lib_1.Stack.of(node).resolve(policies.passwordPolicy);
        if (passwordPolicy == undefined) {
            return nag_rules_1.NagRuleCompliance.NON_COMPLIANT;
        }
        const minimumLength = nag_rules_1.NagRules.resolveIfPrimitive(node, passwordPolicy.minimumLength);
        if (minimumLength == undefined || minimumLength < 8) {
            return nag_rules_1.NagRuleCompliance.NON_COMPLIANT;
        }
        const requireUppercase = nag_rules_1.NagRules.resolveIfPrimitive(node, passwordPolicy.requireUppercase);
        if (requireUppercase !== true) {
            return nag_rules_1.NagRuleCompliance.NON_COMPLIANT;
        }
        const requireNumbers = nag_rules_1.NagRules.resolveIfPrimitive(node, passwordPolicy.requireNumbers);
        if (requireNumbers !== true) {
            return nag_rules_1.NagRuleCompliance.NON_COMPLIANT;
        }
        const requireSymbols = nag_rules_1.NagRules.resolveIfPrimitive(node, passwordPolicy.requireSymbols);
        if (requireSymbols !== true) {
            return nag_rules_1.NagRuleCompliance.NON_COMPLIANT;
        }
        return nag_rules_1.NagRuleCompliance.COMPLIANT;
    }
    else {
        return nag_rules_1.NagRuleCompliance.NOT_APPLICABLE;
    }
}, 'name', { value: path_1.parse(__filename).name });
//# sourceMappingURL=data:application/json;base64,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