"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
/*
Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
SPDX-License-Identifier: Apache-2.0
*/
const path_1 = require("path");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const aws_ecr_1 = require("aws-cdk-lib/aws-ecr");
const nag_rules_1 = require("../../nag-rules");
/**
 * ECR Repositories do not allow open access
 * @param node the CfnResource to check
 */
exports.default = Object.defineProperty((node) => {
    if (node instanceof aws_ecr_1.CfnRegistryPolicy) {
        const policyText = aws_cdk_lib_1.Stack.of(node).resolve(node.policyText);
        const compliant = checkStatement(policyText);
        if (compliant !== true) {
            return nag_rules_1.NagRuleCompliance.NON_COMPLIANT;
        }
        return nag_rules_1.NagRuleCompliance.COMPLIANT;
    }
    else if (node instanceof aws_ecr_1.CfnRepository) {
        const policyText = aws_cdk_lib_1.Stack.of(node).resolve(node.repositoryPolicyText);
        const compliant = checkStatement(policyText);
        if (compliant !== true) {
            return nag_rules_1.NagRuleCompliance.NON_COMPLIANT;
        }
        return nag_rules_1.NagRuleCompliance.COMPLIANT;
    }
    else {
        return nag_rules_1.NagRuleCompliance.NOT_APPLICABLE;
    }
}, 'name', { value: path_1.parse(__filename).name });
/**
 * Helper function for parsing through the policy statement
 * @param policyText the JSON policy text
 * @returns boolean
 */
function checkStatement(policyText) {
    if (policyText == undefined || policyText.Statement == undefined) {
        return true;
    }
    for (const statement of policyText.Statement) {
        const effect = statement.Effect;
        if (effect == 'Allow') {
            const awsString = statement.Principal
                ? JSON.stringify(statement.Principal)
                : '';
            if (awsString.includes('*')) {
                return false;
            }
        }
    }
    return true;
}
//# sourceMappingURL=data:application/json;base64,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