"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
/*
Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
SPDX-License-Identifier: Apache-2.0
*/
const path_1 = require("path");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const aws_elasticloadbalancing_1 = require("aws-cdk-lib/aws-elasticloadbalancing");
const nag_rules_1 = require("../../nag-rules");
/**
 * CLBs use ACM-managed certificates
 * @param node the CfnResource to check
 */
exports.default = Object.defineProperty((node) => {
    if (node instanceof aws_elasticloadbalancing_1.CfnLoadBalancer) {
        //For each listener, ensure that it's utilizing an ACM SSL/HTTPS cert
        const listeners = aws_cdk_lib_1.Stack.of(node).resolve(node.listeners);
        if (listeners != undefined) {
            //Iterate through listeners, checking if secured ACM certs are used
            for (const listener of listeners) {
                const resolvedListener = aws_cdk_lib_1.Stack.of(node).resolve(listener);
                const listenerARN = nag_rules_1.NagRules.resolveIfPrimitive(node, resolvedListener.sslCertificateId);
                //Use the ARN to check if this is an ACM managed cert
                if (listenerARN == undefined) {
                    return nag_rules_1.NagRuleCompliance.NON_COMPLIANT;
                }
                else {
                    const acmRegex = /^arn:[^:]+:acm:.+$/;
                    if (!acmRegex.test(listenerARN)) {
                        return nag_rules_1.NagRuleCompliance.NON_COMPLIANT;
                    }
                }
            }
        }
        return nag_rules_1.NagRuleCompliance.COMPLIANT;
    }
    else {
        return nag_rules_1.NagRuleCompliance.NOT_APPLICABLE;
    }
}, 'name', { value: path_1.parse(__filename).name });
//# sourceMappingURL=data:application/json;base64,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