"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
/*
Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
SPDX-License-Identifier: Apache-2.0
*/
const path_1 = require("path");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const aws_glue_1 = require("aws-cdk-lib/aws-glue");
const nag_rules_1 = require("../../nag-rules");
/**
 * Glue job bookmarks have encryption at rest enabled
 * @param node the CfnResource to check
 */
exports.default = Object.defineProperty((node) => {
    if (node instanceof aws_glue_1.CfnJob) {
        const securityConfigurationId = nag_rules_1.NagRules.resolveResourceFromInstrinsic(node, node.securityConfiguration);
        if (securityConfigurationId == undefined) {
            return nag_rules_1.NagRuleCompliance.NON_COMPLIANT;
        }
        let found = false;
        for (const child of aws_cdk_lib_1.Stack.of(node).node.findAll()) {
            if (child instanceof aws_glue_1.CfnSecurityConfiguration) {
                if (isMatchingSecurityConfig(child, securityConfigurationId)) {
                    found = true;
                    break;
                }
            }
        }
        if (!found) {
            return nag_rules_1.NagRuleCompliance.NON_COMPLIANT;
        }
        return nag_rules_1.NagRuleCompliance.COMPLIANT;
    }
    else {
        return nag_rules_1.NagRuleCompliance.NOT_APPLICABLE;
    }
}, 'name', { value: path_1.parse(__filename).name });
/**
 * Helper function to check whether the referenced Security Configuration encrypts job bookmarks
 * @param node the CfnSecurityConfiguration to check
 * @param securityConfigurationId the Cfn Logical ID of the security configuration
 * returns whether the CfnSecurityConfiguration encrypts job bookmarks
 */
function isMatchingSecurityConfig(node, securityConfigurationId) {
    const resolvedSecurityConfigurationLogicalId = nag_rules_1.NagRules.resolveResourceFromInstrinsic(node, node.ref);
    if (resolvedSecurityConfigurationLogicalId === securityConfigurationId ||
        node.name === securityConfigurationId) {
        const encryptionConfiguration = aws_cdk_lib_1.Stack.of(node).resolve(node.encryptionConfiguration);
        const jobBookmarksEncryption = encryptionConfiguration.jobBookmarksEncryption;
        if (jobBookmarksEncryption !== undefined) {
            const jobBookmarksEncryptionMode = nag_rules_1.NagRules.resolveIfPrimitive(node, jobBookmarksEncryption.jobBookmarksEncryptionMode);
            if (jobBookmarksEncryptionMode === 'CSE-KMS') {
                return true;
            }
        }
    }
    return false;
}
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiR2x1ZUpvYkJvb2ttYXJrRW5jcnlwdGVkLmpzIiwic291cmNlUm9vdCI6IiIsInNvdXJjZXMiOlsiLi4vLi4vLi4vc3JjL3J1bGVzL2dsdWUvR2x1ZUpvYkJvb2ttYXJrRW5jcnlwdGVkLnRzIl0sIm5hbWVzIjpbXSwibWFwcGluZ3MiOiI7O0FBQUE7OztFQUdFO0FBQ0YsK0JBQTZCO0FBQzdCLDZDQUFpRDtBQUNqRCxtREFBd0U7QUFDeEUsK0NBQThEO0FBRTlEOzs7R0FHRztBQUNILGtCQUFlLE1BQU0sQ0FBQyxjQUFjLENBQ2xDLENBQUMsSUFBaUIsRUFBcUIsRUFBRTtJQUN2QyxJQUFJLElBQUksWUFBWSxpQkFBTSxFQUFFO1FBQzFCLE1BQU0sdUJBQXVCLEdBQUcsb0JBQVEsQ0FBQyw2QkFBNkIsQ0FDcEUsSUFBSSxFQUNKLElBQUksQ0FBQyxxQkFBcUIsQ0FDM0IsQ0FBQztRQUNGLElBQUksdUJBQXVCLElBQUksU0FBUyxFQUFFO1lBQ3hDLE9BQU8sNkJBQWlCLENBQUMsYUFBYSxDQUFDO1NBQ3hDO1FBQ0QsSUFBSSxLQUFLLEdBQUcsS0FBSyxDQUFDO1FBQ2xCLEtBQUssTUFBTSxLQUFLLElBQUksbUJBQUssQ0FBQyxFQUFFLENBQUMsSUFBSSxDQUFDLENBQUMsSUFBSSxDQUFDLE9BQU8sRUFBRSxFQUFFO1lBQ2pELElBQUksS0FBSyxZQUFZLG1DQUF3QixFQUFFO2dCQUM3QyxJQUFJLHdCQUF3QixDQUFDLEtBQUssRUFBRSx1QkFBdUIsQ0FBQyxFQUFFO29CQUM1RCxLQUFLLEdBQUcsSUFBSSxDQUFDO29CQUNiLE1BQU07aUJBQ1A7YUFDRjtTQUNGO1FBQ0QsSUFBSSxDQUFDLEtBQUssRUFBRTtZQUNWLE9BQU8sNkJBQWlCLENBQUMsYUFBYSxDQUFDO1NBQ3hDO1FBQ0QsT0FBTyw2QkFBaUIsQ0FBQyxTQUFTLENBQUM7S0FDcEM7U0FBTTtRQUNMLE9BQU8sNkJBQWlCLENBQUMsY0FBYyxDQUFDO0tBQ3pDO0FBQ0gsQ0FBQyxFQUNELE1BQU0sRUFDTixFQUFFLEtBQUssRUFBRSxZQUFLLENBQUMsVUFBVSxDQUFDLENBQUMsSUFBSSxFQUFFLENBQ2xDLENBQUM7QUFFRjs7Ozs7R0FLRztBQUNILFNBQVMsd0JBQXdCLENBQy9CLElBQThCLEVBQzlCLHVCQUErQjtJQUUvQixNQUFNLHNDQUFzQyxHQUMxQyxvQkFBUSxDQUFDLDZCQUE2QixDQUFDLElBQUksRUFBRSxJQUFJLENBQUMsR0FBRyxDQUFDLENBQUM7SUFDekQsSUFDRSxzQ0FBc0MsS0FBSyx1QkFBdUI7UUFDbEUsSUFBSSxDQUFDLElBQUksS0FBSyx1QkFBdUIsRUFDckM7UUFDQSxNQUFNLHVCQUF1QixHQUFHLG1CQUFLLENBQUMsRUFBRSxDQUFDLElBQUksQ0FBQyxDQUFDLE9BQU8sQ0FDcEQsSUFBSSxDQUFDLHVCQUF1QixDQUM3QixDQUFDO1FBQ0YsTUFBTSxzQkFBc0IsR0FDMUIsdUJBQXVCLENBQUMsc0JBQXNCLENBQUM7UUFDakQsSUFBSSxzQkFBc0IsS0FBSyxTQUFTLEVBQUU7WUFDeEMsTUFBTSwwQkFBMEIsR0FBRyxvQkFBUSxDQUFDLGtCQUFrQixDQUM1RCxJQUFJLEVBQ0osc0JBQXNCLENBQUMsMEJBQTBCLENBQ2xELENBQUM7WUFDRixJQUFJLDBCQUEwQixLQUFLLFNBQVMsRUFBRTtnQkFDNUMsT0FBTyxJQUFJLENBQUM7YUFDYjtTQUNGO0tBQ0Y7SUFDRCxPQUFPLEtBQUssQ0FBQztBQUNmLENBQUMiLCJzb3VyY2VzQ29udGVudCI6WyIvKlxuQ29weXJpZ2h0IEFtYXpvbi5jb20sIEluYy4gb3IgaXRzIGFmZmlsaWF0ZXMuIEFsbCBSaWdodHMgUmVzZXJ2ZWQuXG5TUERYLUxpY2Vuc2UtSWRlbnRpZmllcjogQXBhY2hlLTIuMFxuKi9cbmltcG9ydCB7IHBhcnNlIH0gZnJvbSAncGF0aCc7XG5pbXBvcnQgeyBDZm5SZXNvdXJjZSwgU3RhY2sgfSBmcm9tICdhd3MtY2RrLWxpYic7XG5pbXBvcnQgeyBDZm5Kb2IsIENmblNlY3VyaXR5Q29uZmlndXJhdGlvbiB9IGZyb20gJ2F3cy1jZGstbGliL2F3cy1nbHVlJztcbmltcG9ydCB7IE5hZ1J1bGVDb21wbGlhbmNlLCBOYWdSdWxlcyB9IGZyb20gJy4uLy4uL25hZy1ydWxlcyc7XG5cbi8qKlxuICogR2x1ZSBqb2IgYm9va21hcmtzIGhhdmUgZW5jcnlwdGlvbiBhdCByZXN0IGVuYWJsZWRcbiAqIEBwYXJhbSBub2RlIHRoZSBDZm5SZXNvdXJjZSB0byBjaGVja1xuICovXG5leHBvcnQgZGVmYXVsdCBPYmplY3QuZGVmaW5lUHJvcGVydHkoXG4gIChub2RlOiBDZm5SZXNvdXJjZSk6IE5hZ1J1bGVDb21wbGlhbmNlID0+IHtcbiAgICBpZiAobm9kZSBpbnN0YW5jZW9mIENmbkpvYikge1xuICAgICAgY29uc3Qgc2VjdXJpdHlDb25maWd1cmF0aW9uSWQgPSBOYWdSdWxlcy5yZXNvbHZlUmVzb3VyY2VGcm9tSW5zdHJpbnNpYyhcbiAgICAgICAgbm9kZSxcbiAgICAgICAgbm9kZS5zZWN1cml0eUNvbmZpZ3VyYXRpb25cbiAgICAgICk7XG4gICAgICBpZiAoc2VjdXJpdHlDb25maWd1cmF0aW9uSWQgPT0gdW5kZWZpbmVkKSB7XG4gICAgICAgIHJldHVybiBOYWdSdWxlQ29tcGxpYW5jZS5OT05fQ09NUExJQU5UO1xuICAgICAgfVxuICAgICAgbGV0IGZvdW5kID0gZmFsc2U7XG4gICAgICBmb3IgKGNvbnN0IGNoaWxkIG9mIFN0YWNrLm9mKG5vZGUpLm5vZGUuZmluZEFsbCgpKSB7XG4gICAgICAgIGlmIChjaGlsZCBpbnN0YW5jZW9mIENmblNlY3VyaXR5Q29uZmlndXJhdGlvbikge1xuICAgICAgICAgIGlmIChpc01hdGNoaW5nU2VjdXJpdHlDb25maWcoY2hpbGQsIHNlY3VyaXR5Q29uZmlndXJhdGlvbklkKSkge1xuICAgICAgICAgICAgZm91bmQgPSB0cnVlO1xuICAgICAgICAgICAgYnJlYWs7XG4gICAgICAgICAgfVxuICAgICAgICB9XG4gICAgICB9XG4gICAgICBpZiAoIWZvdW5kKSB7XG4gICAgICAgIHJldHVybiBOYWdSdWxlQ29tcGxpYW5jZS5OT05fQ09NUExJQU5UO1xuICAgICAgfVxuICAgICAgcmV0dXJuIE5hZ1J1bGVDb21wbGlhbmNlLkNPTVBMSUFOVDtcbiAgICB9IGVsc2Uge1xuICAgICAgcmV0dXJuIE5hZ1J1bGVDb21wbGlhbmNlLk5PVF9BUFBMSUNBQkxFO1xuICAgIH1cbiAgfSxcbiAgJ25hbWUnLFxuICB7IHZhbHVlOiBwYXJzZShfX2ZpbGVuYW1lKS5uYW1lIH1cbik7XG5cbi8qKlxuICogSGVscGVyIGZ1bmN0aW9uIHRvIGNoZWNrIHdoZXRoZXIgdGhlIHJlZmVyZW5jZWQgU2VjdXJpdHkgQ29uZmlndXJhdGlvbiBlbmNyeXB0cyBqb2IgYm9va21hcmtzXG4gKiBAcGFyYW0gbm9kZSB0aGUgQ2ZuU2VjdXJpdHlDb25maWd1cmF0aW9uIHRvIGNoZWNrXG4gKiBAcGFyYW0gc2VjdXJpdHlDb25maWd1cmF0aW9uSWQgdGhlIENmbiBMb2dpY2FsIElEIG9mIHRoZSBzZWN1cml0eSBjb25maWd1cmF0aW9uXG4gKiByZXR1cm5zIHdoZXRoZXIgdGhlIENmblNlY3VyaXR5Q29uZmlndXJhdGlvbiBlbmNyeXB0cyBqb2IgYm9va21hcmtzXG4gKi9cbmZ1bmN0aW9uIGlzTWF0Y2hpbmdTZWN1cml0eUNvbmZpZyhcbiAgbm9kZTogQ2ZuU2VjdXJpdHlDb25maWd1cmF0aW9uLFxuICBzZWN1cml0eUNvbmZpZ3VyYXRpb25JZDogc3RyaW5nXG4pOiBib29sZWFuIHtcbiAgY29uc3QgcmVzb2x2ZWRTZWN1cml0eUNvbmZpZ3VyYXRpb25Mb2dpY2FsSWQgPVxuICAgIE5hZ1J1bGVzLnJlc29sdmVSZXNvdXJjZUZyb21JbnN0cmluc2ljKG5vZGUsIG5vZGUucmVmKTtcbiAgaWYgKFxuICAgIHJlc29sdmVkU2VjdXJpdHlDb25maWd1cmF0aW9uTG9naWNhbElkID09PSBzZWN1cml0eUNvbmZpZ3VyYXRpb25JZCB8fFxuICAgIG5vZGUubmFtZSA9PT0gc2VjdXJpdHlDb25maWd1cmF0aW9uSWRcbiAgKSB7XG4gICAgY29uc3QgZW5jcnlwdGlvbkNvbmZpZ3VyYXRpb24gPSBTdGFjay5vZihub2RlKS5yZXNvbHZlKFxuICAgICAgbm9kZS5lbmNyeXB0aW9uQ29uZmlndXJhdGlvblxuICAgICk7XG4gICAgY29uc3Qgam9iQm9va21hcmtzRW5jcnlwdGlvbiA9XG4gICAgICBlbmNyeXB0aW9uQ29uZmlndXJhdGlvbi5qb2JCb29rbWFya3NFbmNyeXB0aW9uO1xuICAgIGlmIChqb2JCb29rbWFya3NFbmNyeXB0aW9uICE9PSB1bmRlZmluZWQpIHtcbiAgICAgIGNvbnN0IGpvYkJvb2ttYXJrc0VuY3J5cHRpb25Nb2RlID0gTmFnUnVsZXMucmVzb2x2ZUlmUHJpbWl0aXZlKFxuICAgICAgICBub2RlLFxuICAgICAgICBqb2JCb29rbWFya3NFbmNyeXB0aW9uLmpvYkJvb2ttYXJrc0VuY3J5cHRpb25Nb2RlXG4gICAgICApO1xuICAgICAgaWYgKGpvYkJvb2ttYXJrc0VuY3J5cHRpb25Nb2RlID09PSAnQ1NFLUtNUycpIHtcbiAgICAgICAgcmV0dXJuIHRydWU7XG4gICAgICB9XG4gICAgfVxuICB9XG4gIHJldHVybiBmYWxzZTtcbn1cbiJdfQ==