"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
/*
Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
SPDX-License-Identifier: Apache-2.0
*/
const path_1 = require("path");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const aws_iam_1 = require("aws-cdk-lib/aws-iam");
const nag_rules_1 = require("../../nag-rules");
const flatten_cfn_reference_1 = require("../../utils/flatten-cfn-reference");
/**
 * IAM entities with wildcard permissions have a cdk-nag rule suppression with evidence for those permission
 * @param node the CfnResource to check
 */
exports.default = Object.defineProperty((node) => {
    if (node instanceof aws_iam_1.CfnGroup ||
        node instanceof aws_iam_1.CfnUser ||
        node instanceof aws_iam_1.CfnRole) {
        const inlinePolicies = aws_cdk_lib_1.Stack.of(node).resolve(node.policies);
        const findings = new Set();
        if (inlinePolicies != undefined) {
            for (const policy of inlinePolicies) {
                const resolvedPolicy = aws_cdk_lib_1.Stack.of(node).resolve(policy);
                const resolvedPolicyDocument = aws_cdk_lib_1.Stack.of(node).resolve(resolvedPolicy.policyDocument);
                analyzePolicy(resolvedPolicyDocument).forEach((finding) => findings.add(finding));
            }
        }
        return findings.size ? [...findings] : nag_rules_1.NagRuleCompliance.COMPLIANT;
    }
    else if (node instanceof aws_iam_1.CfnPolicy || node instanceof aws_iam_1.CfnManagedPolicy) {
        const policyDocument = aws_cdk_lib_1.Stack.of(node).resolve(node.policyDocument);
        const findings = analyzePolicy(policyDocument);
        return findings.length ? findings : nag_rules_1.NagRuleCompliance.COMPLIANT;
    }
    else {
        return nag_rules_1.NagRuleCompliance.NOT_APPLICABLE;
    }
}, 'name', { value: path_1.parse(__filename).name });
const analyzePolicy = (policy) => {
    const result = new Set();
    for (const statement of policy.Statement ?? []) {
        // we don't report on denys
        if (statement.Effect === 'Allow') {
            const actions = normalizeArray(statement.Action);
            actions
                .filter((action) => action.includes('*'))
                .map((action) => `Action::${action}`)
                .forEach((action) => result.add(action));
            const resources = normalizeArray(statement.Resource);
            resources
                .map(flatten_cfn_reference_1.flattenCfnReference)
                .filter((resource) => resource.includes('*'))
                .map((resource) => `Resource::${resource}`)
                .forEach((resource) => result.add(resource));
        }
    }
    return [...result];
};
const normalizeArray = (arrOrStr) => Array.isArray(arrOrStr) ? arrOrStr : [arrOrStr];
//# sourceMappingURL=data:application/json;base64,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