"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
/*
Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
SPDX-License-Identifier: Apache-2.0
*/
const path_1 = require("path");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const aws_mediastore_1 = require("aws-cdk-lib/aws-mediastore");
const nag_rules_1 = require("../../nag-rules");
/**
 * Media Store containers require requests to use SSL
 * @param node the CfnResource to check
 */
exports.default = Object.defineProperty((node) => {
    if (node instanceof aws_mediastore_1.CfnContainer) {
        const policy = aws_cdk_lib_1.Stack.of(node).resolve(node.policy);
        if (policy === undefined) {
            return nag_rules_1.NagRuleCompliance.NON_COMPLIANT;
        }
        const parsedPolicy = JSON.parse(policy);
        let found = false;
        for (const statement of parsedPolicy?.Statement) {
            const resolvedStatement = aws_cdk_lib_1.Stack.of(node).resolve(statement);
            const secureTransport = resolvedStatement?.Condition?.Bool?.['aws:SecureTransport'];
            if (resolvedStatement.Effect === 'Deny' &&
                checkMatchingAction(resolvedStatement.Action) === true &&
                checkMatchingPrincipal(resolvedStatement.Principal) === true &&
                (secureTransport === 'false' || secureTransport === false)) {
                found = true;
                break;
            }
        }
        if (!found) {
            return nag_rules_1.NagRuleCompliance.NON_COMPLIANT;
        }
        return nag_rules_1.NagRuleCompliance.COMPLIANT;
    }
    else {
        return nag_rules_1.NagRuleCompliance.NOT_APPLICABLE;
    }
}, 'name', { value: path_1.parse(__filename).name });
/**
 * Helper function to check whether the Bucket Policy applies to all actions
 * @param node The policy statement to check
 * @param actions The action in the bucket policy
 * @returns Whether the policy statement applies to all actions
 */
function checkMatchingAction(actions) {
    if (Array.isArray(actions)) {
        for (const action of actions) {
            if (action === '*' || action.toLowerCase() === 'mediastore:*') {
                return true;
            }
        }
    }
    else if (actions === '*' || actions.toLowerCase() === 'mediastore:*') {
        return true;
    }
    return false;
}
/**
 * Helper function to check whether the Bucket Policy applies to all principals
 * @param node The policy statement to check
 * @param principal The principals in the bucket policy
 * @returns Whether the policy statement applies to all principals
 */
function checkMatchingPrincipal(principals) {
    if (principals === '*') {
        return true;
    }
    const awsPrincipal = principals.AWS;
    if (Array.isArray(awsPrincipal)) {
        for (const account of awsPrincipal) {
            if (account === '*') {
                return true;
            }
        }
    }
    else if (awsPrincipal === '*') {
        return true;
    }
    return false;
}
//# sourceMappingURL=data:application/json;base64,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