"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
/*
Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
SPDX-License-Identifier: Apache-2.0
*/
const path_1 = require("path");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const aws_s3_1 = require("aws-cdk-lib/aws-s3");
const nag_rules_1 = require("../../nag-rules");
/**
 * S3 Buckets have default server-side encryption enabled
 * @param node the CfnResource to check
 */
exports.default = Object.defineProperty((node) => {
    if (node instanceof aws_s3_1.CfnBucket) {
        if (node.bucketEncryption == undefined) {
            return nag_rules_1.NagRuleCompliance.NON_COMPLIANT;
        }
        const encryption = aws_cdk_lib_1.Stack.of(node).resolve(node.bucketEncryption);
        if (encryption.serverSideEncryptionConfiguration == undefined) {
            return nag_rules_1.NagRuleCompliance.NON_COMPLIANT;
        }
        const sse = aws_cdk_lib_1.Stack.of(node).resolve(encryption.serverSideEncryptionConfiguration);
        for (const rule of sse) {
            const defaultEncryption = aws_cdk_lib_1.Stack.of(node).resolve(rule.serverSideEncryptionByDefault);
            if (defaultEncryption == undefined) {
                return nag_rules_1.NagRuleCompliance.NON_COMPLIANT;
            }
            const sseAlgorithm = nag_rules_1.NagRules.resolveIfPrimitive(node, defaultEncryption.sseAlgorithm);
            if (sseAlgorithm.toLowerCase() != 'aes256' &&
                sseAlgorithm.toLowerCase() != 'aws:kms') {
                return nag_rules_1.NagRuleCompliance.NON_COMPLIANT;
            }
        }
        return nag_rules_1.NagRuleCompliance.COMPLIANT;
    }
    else {
        return nag_rules_1.NagRuleCompliance.NOT_APPLICABLE;
    }
}, 'name', { value: path_1.parse(__filename).name });
//# sourceMappingURL=data:application/json;base64,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