from __future__ import annotations

import os
from importlib.metadata import PackageNotFoundError
from importlib.metadata import version as pkg_version
from pathlib import Path

import click
from dotenv import load_dotenv
import signal
import time
import errno

from guildbotics.cli.setup_tool import SetupTool
from guildbotics.drivers import TaskScheduler
from guildbotics.utils.import_utils import instantiate_class
from guildbotics.utils.fileio import get_storage_path


def get_setup_tool() -> SetupTool:
    name = os.getenv("GUILDBOTICS_EDITION", "simple")
    if "." not in name:
        name = f"guildbotics.cli.{name}.{name}_setup_tool.{name.capitalize()}SetupTool"
    return instantiate_class(name, expected_type=SetupTool)


def _resolve_version() -> str:
    try:
        return pkg_version("guildbotics")
    except PackageNotFoundError:
        try:
            from guildbotics._version import __version__ as v  # type: ignore

            return v
        except Exception:
            return "0.0.0+unknown"


def _load_env_from_cwd() -> None:
    dotenv_path = Path.cwd() / ".env"
    if dotenv_path.exists():
        load_dotenv(dotenv_path=dotenv_path, override=False)


def _pid_file_path() -> Path:
    # Store PID under user's home storage path to avoid CWD dependency
    return get_storage_path() / "run" / "scheduler.pid"


def _pid_is_running(pid: int) -> bool:
    try:
        # Signal 0 checks for existence without sending a signal
        os.kill(pid, 0)
    except OSError as e:
        return e.errno == errno.EPERM
    else:
        return True


def _read_pidfile(path: Path) -> int | None:
    try:
        txt = path.read_text().strip()
        return int(txt) if txt else None
    except Exception:
        return None


def _write_pidfile(path: Path, pid: int) -> None:
    path.parent.mkdir(parents=True, exist_ok=True)
    path.write_text(str(pid))


def _remove_pidfile(path: Path) -> None:
    try:
        if path.exists():
            path.unlink()
    except Exception:
        pass


@click.group()
@click.version_option(
    version=_resolve_version(),
    prog_name="guildbotics",
    message="%(prog)s %(version)s",
)
def main() -> None:
    """GuildBotics CLI entrypoint."""
    pass


@main.command()
def run() -> None:
    """Run the GuildBotics application."""
    _load_env_from_cwd()
    pid_path = _pid_file_path()
    # Prevent multiple instances
    if pid_path.exists():
        old_pid = _read_pidfile(pid_path)
        if old_pid and _pid_is_running(old_pid):
            click.echo(
                f"Scheduler already running with PID {old_pid} (pidfile: {pid_path})."
            )
            return
        else:
            # Stale pidfile
            _remove_pidfile(pid_path)

    _write_pidfile(pid_path, os.getpid())

    scheduler = TaskScheduler(get_setup_tool().get_context())

    def _handle_signal(signum, frame):  # type: ignore[no-untyped-def]
        click.echo(f"Received signal {signum}. Shutting down...")
        try:
            scheduler.shutdown(graceful=True)
        finally:
            _remove_pidfile(pid_path)

    # Register signal handlers for graceful shutdown
    signal.signal(signal.SIGINT, _handle_signal)
    signal.signal(signal.SIGTERM, _handle_signal)

    try:
        scheduler.start()
    except KeyboardInterrupt:
        _handle_signal(signal.SIGINT, None)  # type: ignore[arg-type]
    finally:
        _remove_pidfile(pid_path)


@main.command()
def add() -> None:
    """Add a new member to the GuildBotics project."""
    _load_env_from_cwd()
    get_setup_tool().add_member()


@main.command()
def init() -> None:
    """Initialize the GuildBotics environment.

    This function sets up the necessary environment for GuildBotics to run.
    """
    _load_env_from_cwd()
    get_setup_tool().init_project()


@main.command()
def verify() -> None:
    """Verify the GuildBotics environment.

    This function checks the necessary environment for GuildBotics to run.
    """
    _load_env_from_cwd()
    get_setup_tool().verify_environment()


@main.command(name="version")
def version_cmd() -> None:
    """Print version."""
    click.echo(_resolve_version())


@main.command()
@click.option("--timeout", default=30, show_default=True, help="Seconds to wait")
@click.option("--force", is_flag=True, help="Force kill after timeout")
def stop(timeout: int, force: bool) -> None:
    """Gracefully stop the running scheduler process."""
    _load_env_from_cwd()
    pid_path = _pid_file_path()

    if not pid_path.exists():
        click.echo("No pidfile found. Is the scheduler running?")
        return

    pid = _read_pidfile(pid_path)
    if not pid:
        click.echo(f"Invalid pidfile: {pid_path}")
        _remove_pidfile(pid_path)
        return

    if not _pid_is_running(pid):
        click.echo(f"Process {pid} is not running. Cleaning up pidfile.")
        _remove_pidfile(pid_path)
        return

    try:
        os.kill(pid, signal.SIGTERM)
    except PermissionError:
        click.echo(f"Permission denied to signal process {pid}.")
        return
    except ProcessLookupError:
        click.echo(f"Process {pid} does not exist. Cleaning up pidfile.")
        _remove_pidfile(pid_path)
        return

    # Wait for graceful shutdown
    deadline = time.time() + max(0, timeout)
    while time.time() < deadline:
        if not _pid_is_running(pid):
            click.echo("Scheduler stopped.")
            _remove_pidfile(pid_path)
            return
        time.sleep(0.5)

    if force and _pid_is_running(pid):
        try:
            os.kill(pid, signal.SIGKILL)
        except Exception as e:  # noqa: BLE001 - report and continue
            click.echo(f"Failed to SIGKILL {pid}: {e}")
        else:
            click.echo("Force killed scheduler.")
        # Best effort cleanup
        if not _pid_is_running(pid):
            _remove_pidfile(pid_path)
    else:
        click.echo(
            "Timeout reached and process still running. Use --force to SIGKILL."
        )
