"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.FSxWindows = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
/**
 *  Copyright 2021 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
// Imports
const aws_cdk_lib_1 = require("aws-cdk-lib");
const constructs_1 = require("constructs");
const skylight_compute_1 = require("../skylight-compute");
/**
* A FSxWindows represents an integration pattern of Amazon FSx and Managed AD in a specific VPC.

* The Construct creates Amazon FSx for Windows
* The construct also creates (optionally) t3.nano machine that is part of the domain that can be used to run admin-tasks (such as createFolder)
*
* The createFolder() method creates an SMB Folder in the FSx filesystem, using the domain admin user.
* Please note: When calling createFolder() API, a Lambda will be created to start the worker machine (Using AWS-SDK),
* then each command will be scheduled with State Manager, and the instance will be shut down after complete .
 *
 */
class FSxWindows extends constructs_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        this.props = props;
        this.props.fileSystemInPrivateSubnet =
            props.fileSystemInPrivateSubnet ?? true;
        this.props.throughputMbps = props.throughputMbps ?? 128;
        this.props.fileSystemSize = props.fileSystemSize ?? 200;
        this.props.multiAZ = props.multiAZ ?? true;
        this.ssmParameters = props.ssmParameters ?? {};
        this.ssmParameters.dnsEndpoint =
            this.ssmParameters?.dnsEndpoint ?? 'FSxEndpoint-DNS';
        if (this.ssmParameters.namespace) {
            this.ssmParameters.namespace = `${this.ssmParameters.namespace}/storage/fsx`;
        }
        else {
            this.ssmParameters.namespace = 'cdk-skylight/storage/fsx';
        }
        const subnets = this.props.vpc.selectSubnets({
            subnetType: props.fileSystemInPrivateSubnet
                ? aws_cdk_lib_1.aws_ec2.SubnetType.PRIVATE_WITH_NAT
                : aws_cdk_lib_1.aws_ec2.SubnetType.PUBLIC,
        }).subnetIds;
        const windows_configuration = {
            throughputCapacity: this.props.throughputMbps,
            activeDirectoryId: props.directoryId,
            deploymentType: this.props.multiAZ ? 'MULTI_AZ_1' : 'SINGLE_AZ_2',
            preferredSubnetId: this.props.multiAZ ? subnets[0] : undefined,
        };
        const sg = new aws_cdk_lib_1.aws_ec2.SecurityGroup(this, id + '-FSxSG', {
            vpc: this.props.vpc,
        });
        // Allow access from inside the VPC
        sg.addIngressRule(aws_cdk_lib_1.aws_ec2.Peer.ipv4(props.vpc.vpcCidrBlock), aws_cdk_lib_1.aws_ec2.Port.allTcp());
        const fsx_props = {
            fileSystemType: 'WINDOWS',
            subnetIds: props.multiAZ ? [subnets[0], subnets[1]] : [subnets[0]],
            windowsConfiguration: windows_configuration,
            storageCapacity: props.fileSystemSize,
            securityGroupIds: [sg.securityGroupId],
        };
        this.fsxObject = new aws_cdk_lib_1.aws_fsx.CfnFileSystem(this, (id = id + '-FSxObject'), fsx_props);
        new aws_cdk_lib_1.aws_ssm.StringParameter(this, 'ssm-dns-fsxEndpoint', {
            parameterName: `/${this.ssmParameters.namespace}/${this.ssmParameters.dnsEndpoint}`,
            stringValue: this.fsxObject.getAtt('DNSName').toString(),
        });
    }
    createWorker(domainName, domainPassword) {
        return new skylight_compute_1.DomainWindowsNode(this, 'FSxWindowsWorker', {
            vpc: this.props.vpc,
            instanceType: 't3.small',
            iamManagedPoliciesList: [
                aws_cdk_lib_1.aws_iam.ManagedPolicy.fromAwsManagedPolicyName('AmazonSSMManagedInstanceCore'),
                aws_cdk_lib_1.aws_iam.ManagedPolicy.fromAwsManagedPolicyName('SecretsManagerReadWrite'),
                aws_cdk_lib_1.aws_iam.ManagedPolicy.fromAwsManagedPolicyName('AmazonFSxReadOnlyAccess'),
            ],
            domainName: domainName,
            passwordObject: domainPassword,
        });
    }
    createFolder(worker, folderName, secretName) {
        worker.startInstance();
        worker.runPSwithDomainAdmin([
            `$FSX = '${this.fsxObject
                .getAtt('DNSName')
                .toString()}' ## Amazon FSx DNS Name`,
            '$FSxPS = (Get-FSXFileSystem | ? {$_.DNSName -contains $FSX}).WindowsConfiguration.RemoteAdministrationEndpoint',
            `$FolderName = '${folderName}'`,
            `[string]$SecretAD  = '${secretName}'`,
            '$SecretObj = Get-SECSecretValue -SecretId $SecretAD',
            '[PSCustomObject]$Secret = ($SecretObj.SecretString  | ConvertFrom-Json)',
            '$password   = $Secret.Password | ConvertTo-SecureString -asPlainText -Force',
            " $username   = $Secret.Domain + '\\' + $Secret.UserID ",
            '$domain_admin_credential = New-Object System.Management.Automation.PSCredential($username,$password)',
            '# Create the folder (the shared driver to the hosts)',
            'New-Item -ItemType Directory -Name $FolderName -Path \\\\$FSX\\D$\\',
            '# Set NTFS Permissions',
            '# ACL',
            '$ACL = Get-Acl \\\\$FSx\\D$\\$FolderName',
            '$permission = "NT AUTHORITY\\Authenticated Users","FullControl","Allow"',
            '$Ar = New-Object System.Security.AccessControl.FileSystemAccessRule $permission',
            '$ACL.SetAccessRule($Ar)',
            'Set-Acl \\\\$FSX\\D$\\$FolderName $ACL',
            '# Create the Share and set the share permissions',
            '$Session = New-PSSession -ComputerName $FSxPS -ConfigurationName FsxRemoteAdmin',
            'Import-PsSession $Session',
            'New-FSxSmbShare -Name $FolderName -Path "D:\\$FolderName" -Description "Shared folder with gMSA access" -Credential $domain_admin_credential -FolderEnumerationMode AccessBased',
            '$accessList="NT AUTHORITY\\Authenticated Users"',
            'Grant-FSxSmbShareaccess -Name $FolderName -AccountName $accessList -accessRight Full -Confirm:$false',
            'Disconnect-PSSession -Session $Session',
            'Stop-Computer -ComputerName localhost',
        ], 'createFolder');
    }
}
exports.FSxWindows = FSxWindows;
_a = JSII_RTTI_SYMBOL_1;
FSxWindows[_a] = { fqn: "cdk-skylight.storage.FSxWindows", version: "1.1.97" };
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiZnN4LXdpbmRvd3MuanMiLCJzb3VyY2VSb290IjoiIiwic291cmNlcyI6WyIuLi8uLi9zcmMvc2t5bGlnaHQtc3RvcmFnZS9mc3gtd2luZG93cy50cyJdLCJuYW1lcyI6W10sIm1hcHBpbmdzIjoiOzs7OztBQUFBOzs7Ozs7Ozs7OztHQVdHO0FBRUgsVUFBVTtBQUNWLDZDQUF3RTtBQUV4RSwyQ0FBdUM7QUFDdkMsMERBQXdEO0FBdUR4RDs7Ozs7Ozs7OztHQVVHO0FBQ0gsTUFBYSxVQUFXLFNBQVEsc0JBQVM7SUFJdkMsWUFBWSxLQUFnQixFQUFFLEVBQVUsRUFBRSxLQUF1QjtRQUMvRCxLQUFLLENBQUMsS0FBSyxFQUFFLEVBQUUsQ0FBQyxDQUFDO1FBQ2pCLElBQUksQ0FBQyxLQUFLLEdBQUcsS0FBSyxDQUFDO1FBQ25CLElBQUksQ0FBQyxLQUFLLENBQUMseUJBQXlCO1lBQ2xDLEtBQUssQ0FBQyx5QkFBeUIsSUFBSSxJQUFJLENBQUM7UUFDMUMsSUFBSSxDQUFDLEtBQUssQ0FBQyxjQUFjLEdBQUcsS0FBSyxDQUFDLGNBQWMsSUFBSSxHQUFHLENBQUM7UUFDeEQsSUFBSSxDQUFDLEtBQUssQ0FBQyxjQUFjLEdBQUcsS0FBSyxDQUFDLGNBQWMsSUFBSSxHQUFHLENBQUM7UUFDeEQsSUFBSSxDQUFDLEtBQUssQ0FBQyxPQUFPLEdBQUcsS0FBSyxDQUFDLE9BQU8sSUFBSSxJQUFJLENBQUM7UUFDM0MsSUFBSSxDQUFDLGFBQWEsR0FBRyxLQUFLLENBQUMsYUFBYSxJQUFJLEVBQUUsQ0FBQztRQUMvQyxJQUFJLENBQUMsYUFBYSxDQUFDLFdBQVc7WUFDNUIsSUFBSSxDQUFDLGFBQWEsRUFBRSxXQUFXLElBQUksaUJBQWlCLENBQUM7UUFFdkQsSUFBSSxJQUFJLENBQUMsYUFBYSxDQUFDLFNBQVMsRUFBRTtZQUNoQyxJQUFJLENBQUMsYUFBYSxDQUFDLFNBQVMsR0FBRyxHQUFHLElBQUksQ0FBQyxhQUFhLENBQUMsU0FBUyxjQUFjLENBQUM7U0FDOUU7YUFBTTtZQUNMLElBQUksQ0FBQyxhQUFhLENBQUMsU0FBUyxHQUFHLDBCQUEwQixDQUFDO1NBQzNEO1FBRUQsTUFBTSxPQUFPLEdBQUcsSUFBSSxDQUFDLEtBQUssQ0FBQyxHQUFHLENBQUMsYUFBYSxDQUFDO1lBQzNDLFVBQVUsRUFBRSxLQUFLLENBQUMseUJBQXlCO2dCQUN6QyxDQUFDLENBQUMscUJBQUcsQ0FBQyxVQUFVLENBQUMsZ0JBQWdCO2dCQUNqQyxDQUFDLENBQUMscUJBQUcsQ0FBQyxVQUFVLENBQUMsTUFBTTtTQUMxQixDQUFDLENBQUMsU0FBUyxDQUFDO1FBRWIsTUFBTSxxQkFBcUIsR0FDekI7WUFDRSxrQkFBa0IsRUFBRSxJQUFJLENBQUMsS0FBSyxDQUFDLGNBQWM7WUFDN0MsaUJBQWlCLEVBQUUsS0FBSyxDQUFDLFdBQVc7WUFDcEMsY0FBYyxFQUFFLElBQUksQ0FBQyxLQUFLLENBQUMsT0FBTyxDQUFDLENBQUMsQ0FBQyxZQUFZLENBQUMsQ0FBQyxDQUFDLGFBQWE7WUFDakUsaUJBQWlCLEVBQUUsSUFBSSxDQUFDLEtBQUssQ0FBQyxPQUFPLENBQUMsQ0FBQyxDQUFDLE9BQU8sQ0FBQyxDQUFDLENBQUMsQ0FBQyxDQUFDLENBQUMsU0FBUztTQUMvRCxDQUFDO1FBRUosTUFBTSxFQUFFLEdBQUcsSUFBSSxxQkFBRyxDQUFDLGFBQWEsQ0FBQyxJQUFJLEVBQUUsRUFBRSxHQUFHLFFBQVEsRUFBRTtZQUNwRCxHQUFHLEVBQUUsSUFBSSxDQUFDLEtBQUssQ0FBQyxHQUFHO1NBQ3BCLENBQUMsQ0FBQztRQUVILG1DQUFtQztRQUNuQyxFQUFFLENBQUMsY0FBYyxDQUFDLHFCQUFHLENBQUMsSUFBSSxDQUFDLElBQUksQ0FBQyxLQUFLLENBQUMsR0FBRyxDQUFDLFlBQVksQ0FBQyxFQUFFLHFCQUFHLENBQUMsSUFBSSxDQUFDLE1BQU0sRUFBRSxDQUFDLENBQUM7UUFFNUUsTUFBTSxTQUFTLEdBQStCO1lBQzVDLGNBQWMsRUFBRSxTQUFTO1lBQ3pCLFNBQVMsRUFBRSxLQUFLLENBQUMsT0FBTyxDQUFDLENBQUMsQ0FBQyxDQUFDLE9BQU8sQ0FBQyxDQUFDLENBQUMsRUFBRSxPQUFPLENBQUMsQ0FBQyxDQUFDLENBQUMsQ0FBQyxDQUFDLENBQUMsQ0FBQyxPQUFPLENBQUMsQ0FBQyxDQUFDLENBQUM7WUFDbEUsb0JBQW9CLEVBQUUscUJBQXFCO1lBQzNDLGVBQWUsRUFBRSxLQUFLLENBQUMsY0FBYztZQUNyQyxnQkFBZ0IsRUFBRSxDQUFDLEVBQUUsQ0FBQyxlQUFlLENBQUM7U0FDdkMsQ0FBQztRQUVGLElBQUksQ0FBQyxTQUFTLEdBQUcsSUFBSSxxQkFBTyxDQUFDLGFBQWEsQ0FDeEMsSUFBSSxFQUNKLENBQUMsRUFBRSxHQUFHLEVBQUUsR0FBRyxZQUFZLENBQUMsRUFDeEIsU0FBUyxDQUNWLENBQUM7UUFFRixJQUFJLHFCQUFPLENBQUMsZUFBZSxDQUFDLElBQUksRUFBRSxxQkFBcUIsRUFBRTtZQUN2RCxhQUFhLEVBQUUsSUFBSSxJQUFJLENBQUMsYUFBYSxDQUFDLFNBQVMsSUFBSSxJQUFJLENBQUMsYUFBYSxDQUFDLFdBQVcsRUFBRTtZQUNuRixXQUFXLEVBQUUsSUFBSSxDQUFDLFNBQVMsQ0FBQyxNQUFNLENBQUMsU0FBUyxDQUFDLENBQUMsUUFBUSxFQUFFO1NBQ3pELENBQUMsQ0FBQztJQUNMLENBQUM7SUFFRCxZQUFZLENBQUMsVUFBa0IsRUFBRSxjQUF1QjtRQUN0RCxPQUFPLElBQUksb0NBQWlCLENBQUMsSUFBSSxFQUFFLGtCQUFrQixFQUFFO1lBQ3JELEdBQUcsRUFBRSxJQUFJLENBQUMsS0FBSyxDQUFDLEdBQUc7WUFDbkIsWUFBWSxFQUFFLFVBQVU7WUFDeEIsc0JBQXNCLEVBQUU7Z0JBQ3RCLHFCQUFPLENBQUMsYUFBYSxDQUFDLHdCQUF3QixDQUM1Qyw4QkFBOEIsQ0FDL0I7Z0JBQ0QscUJBQU8sQ0FBQyxhQUFhLENBQUMsd0JBQXdCLENBQzVDLHlCQUF5QixDQUMxQjtnQkFDRCxxQkFBTyxDQUFDLGFBQWEsQ0FBQyx3QkFBd0IsQ0FDNUMseUJBQXlCLENBQzFCO2FBQ0Y7WUFDRCxVQUFVLEVBQUUsVUFBVTtZQUN0QixjQUFjLEVBQUUsY0FBYztTQUMvQixDQUFDLENBQUM7SUFDTCxDQUFDO0lBRUQsWUFBWSxDQUNWLE1BQXlCLEVBQ3pCLFVBQWtCLEVBQ2xCLFVBQW1CO1FBRW5CLE1BQU0sQ0FBQyxhQUFhLEVBQUUsQ0FBQztRQUN2QixNQUFNLENBQUMsb0JBQW9CLENBQ3pCO1lBQ0UsV0FBVyxJQUFJLENBQUMsU0FBUztpQkFDdEIsTUFBTSxDQUFDLFNBQVMsQ0FBQztpQkFDakIsUUFBUSxFQUFFLDBCQUEwQjtZQUN2QyxnSEFBZ0g7WUFDaEgsa0JBQWtCLFVBQVUsR0FBRztZQUMvQix5QkFBeUIsVUFBVSxHQUFHO1lBQ3RDLHFEQUFxRDtZQUNyRCx5RUFBeUU7WUFDekUsNkVBQTZFO1lBQzdFLHdEQUF3RDtZQUN4RCxzR0FBc0c7WUFDdEcsc0RBQXNEO1lBQ3RELHFFQUFxRTtZQUNyRSx3QkFBd0I7WUFDeEIsT0FBTztZQUNQLDBDQUEwQztZQUMxQyx5RUFBeUU7WUFDekUsaUZBQWlGO1lBQ2pGLHlCQUF5QjtZQUN6Qix3Q0FBd0M7WUFDeEMsa0RBQWtEO1lBQ2xELGlGQUFpRjtZQUNqRiwyQkFBMkI7WUFDM0IsaUxBQWlMO1lBQ2pMLGlEQUFpRDtZQUNqRCxzR0FBc0c7WUFDdEcsd0NBQXdDO1lBQ3hDLHVDQUF1QztTQUN4QyxFQUNELGNBQWMsQ0FDZixDQUFDO0lBQ0osQ0FBQzs7QUExSEgsZ0NBMkhDIiwic291cmNlc0NvbnRlbnQiOlsiLyoqXG4gKiAgQ29weXJpZ2h0IDIwMjEgQW1hem9uLmNvbSwgSW5jLiBvciBpdHMgYWZmaWxpYXRlcy4gQWxsIFJpZ2h0cyBSZXNlcnZlZC5cbiAqXG4gKiAgTGljZW5zZWQgdW5kZXIgdGhlIEFwYWNoZSBMaWNlbnNlLCBWZXJzaW9uIDIuMCAodGhlIFwiTGljZW5zZVwiKS4gWW91IG1heSBub3QgdXNlIHRoaXMgZmlsZSBleGNlcHQgaW4gY29tcGxpYW5jZVxuICogIHdpdGggdGhlIExpY2Vuc2UuIEEgY29weSBvZiB0aGUgTGljZW5zZSBpcyBsb2NhdGVkIGF0XG4gKlxuICogICAgICBodHRwOi8vd3d3LmFwYWNoZS5vcmcvbGljZW5zZXMvTElDRU5TRS0yLjBcbiAqXG4gKiAgb3IgaW4gdGhlICdsaWNlbnNlJyBmaWxlIGFjY29tcGFueWluZyB0aGlzIGZpbGUuIFRoaXMgZmlsZSBpcyBkaXN0cmlidXRlZCBvbiBhbiAnQVMgSVMnIEJBU0lTLCBXSVRIT1VUIFdBUlJBTlRJRVNcbiAqICBPUiBDT05ESVRJT05TIE9GIEFOWSBLSU5ELCBleHByZXNzIG9yIGltcGxpZWQuIFNlZSB0aGUgTGljZW5zZSBmb3IgdGhlIHNwZWNpZmljIGxhbmd1YWdlIGdvdmVybmluZyBwZXJtaXNzaW9uc1xuICogIGFuZCBsaW1pdGF0aW9ucyB1bmRlciB0aGUgTGljZW5zZS5cbiAqL1xuXG4vLyBJbXBvcnRzXG5pbXBvcnQgeyBhd3NfZWMyIGFzIGVjMiwgYXdzX3NzbSwgYXdzX2ZzeCwgYXdzX2lhbSB9IGZyb20gJ2F3cy1jZGstbGliJztcbmltcG9ydCB7IElTZWNyZXQgfSBmcm9tICdhd3MtY2RrLWxpYi9hd3Mtc2VjcmV0c21hbmFnZXInO1xuaW1wb3J0IHsgQ29uc3RydWN0IH0gZnJvbSAnY29uc3RydWN0cyc7XG5pbXBvcnQgeyBEb21haW5XaW5kb3dzTm9kZSB9IGZyb20gJy4uL3NreWxpZ2h0LWNvbXB1dGUnO1xuXG4vKipcbiAqIFRoZSBwcm9wZXJ0aWVzIGZvciB0aGUgUGVyc2lzdGVudFN0b3JhZ2UgY2xhc3MuXG4gKi9cbmV4cG9ydCBpbnRlcmZhY2UgSUZTeFdpbmRvd3NQcm9wcyB7XG4gIC8qKlxuICAgKiBUaGUgRmlsZXN5c3RlbSBzaXplIGluIEdCXG4gICAqXG4gICAqIEBkZWZhdWx0IC0gMjAwLlxuICAgKi9cbiAgZmlsZVN5c3RlbVNpemU/OiBudW1iZXI7XG4gIC8qKlxuICAgKiBUaGUgRmlsZXN5c3RlbSB0aHJvdWdocHV0IGluIE1CcHNcbiAgICpcbiAgICogQGRlZmF1bHQgLSAxMjguXG4gICAqL1xuICB0aHJvdWdocHV0TWJwcz86IG51bWJlcjtcbiAgLyoqXG4gICAqIENob29zaW5nIFNpbmdsZS1BWiBvciBNdWx0aS1BWiBmaWxlIHN5c3RlbSBkZXBsb3ltZW50XG4gICAqIFNlZTogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2ZzeC9sYXRlc3QvV2luZG93c0d1aWRlL2hpZ2gtYXZhaWxhYmlsaXR5LW11bHRpQVouaHRtbFxuICAgKiBAZGVmYXVsdCAtIHRydWUuXG4gICAqL1xuICBtdWx0aUFaPzogYm9vbGVhbjtcbiAgLyoqXG4gICAqIERlcGxveSB0aGUgQW1hem9uIEZTeCBmaWxlIHN5c3RlbSBpbiBwcml2YXRlIHN1Ym5ldCBvciBwdWJsaWMgc3VibmV0XG4gICAqIFNlZTogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2ZzeC9sYXRlc3QvV2luZG93c0d1aWRlL2hpZ2gtYXZhaWxhYmlsaXR5LW11bHRpQVouaHRtbFxuICAgKiBAZGVmYXVsdCAtIHRydWUuXG4gICAqL1xuICBmaWxlU3lzdGVtSW5Qcml2YXRlU3VibmV0PzogYm9vbGVhbjtcbiAgLyoqXG4gICAqIFRoZSBWUEMgdG8gdXNlLCBtdXN0IGhhdmUgcHJpdmF0ZSBzdWJuZXRzLlxuICAgKi9cblxuICB2cGM6IGVjMi5JVnBjO1xuXG4gIGRpcmVjdG9yeUlkOiBzdHJpbmc7XG5cbiAgc3NtUGFyYW1ldGVycz86IElGU3hXaW5kb3dzUGFyYW1ldGVycztcbn1cblxuZXhwb3J0IGludGVyZmFjZSBJRlN4V2luZG93c1BhcmFtZXRlcnMge1xuICAvKipcbiAgICogVGhlIG5hbWUgb2YgdGhlIHBhcmFtZXRlciB0byBzYXZlIHRoZSBGU3hFbmRwb2ludCBETlMgRW5kcG9pbnRcbiAgICogQGRlZmF1bHQgLSAnRlN4RW5kcG9pbnQtRE5TJy5cbiAgICovXG4gIGRuc0VuZHBvaW50Pzogc3RyaW5nO1xuXG4gIC8qKlxuICAgKiBUaGUgU1NNIG5hbWVzcGFjZSB0byByZWFkL3dyaXRlIHBhcmFtZXRlcnMgdG9cbiAgICogQGRlZmF1bHQgLSAnY2RrLXNreWxpZ2h0Jy5cbiAgICovXG4gIG5hbWVzcGFjZT86IHN0cmluZztcbn1cblxuLyoqXG4qIEEgRlN4V2luZG93cyByZXByZXNlbnRzIGFuIGludGVncmF0aW9uIHBhdHRlcm4gb2YgQW1hem9uIEZTeCBhbmQgTWFuYWdlZCBBRCBpbiBhIHNwZWNpZmljIFZQQy5cblxuKiBUaGUgQ29uc3RydWN0IGNyZWF0ZXMgQW1hem9uIEZTeCBmb3IgV2luZG93c1xuKiBUaGUgY29uc3RydWN0IGFsc28gY3JlYXRlcyAob3B0aW9uYWxseSkgdDMubmFubyBtYWNoaW5lIHRoYXQgaXMgcGFydCBvZiB0aGUgZG9tYWluIHRoYXQgY2FuIGJlIHVzZWQgdG8gcnVuIGFkbWluLXRhc2tzIChzdWNoIGFzIGNyZWF0ZUZvbGRlcilcbipcbiogVGhlIGNyZWF0ZUZvbGRlcigpIG1ldGhvZCBjcmVhdGVzIGFuIFNNQiBGb2xkZXIgaW4gdGhlIEZTeCBmaWxlc3lzdGVtLCB1c2luZyB0aGUgZG9tYWluIGFkbWluIHVzZXIuXG4qIFBsZWFzZSBub3RlOiBXaGVuIGNhbGxpbmcgY3JlYXRlRm9sZGVyKCkgQVBJLCBhIExhbWJkYSB3aWxsIGJlIGNyZWF0ZWQgdG8gc3RhcnQgdGhlIHdvcmtlciBtYWNoaW5lIChVc2luZyBBV1MtU0RLKSxcbiogdGhlbiBlYWNoIGNvbW1hbmQgd2lsbCBiZSBzY2hlZHVsZWQgd2l0aCBTdGF0ZSBNYW5hZ2VyLCBhbmQgdGhlIGluc3RhbmNlIHdpbGwgYmUgc2h1dCBkb3duIGFmdGVyIGNvbXBsZXRlIC5cbiAqXG4gKi9cbmV4cG9ydCBjbGFzcyBGU3hXaW5kb3dzIGV4dGVuZHMgQ29uc3RydWN0IHtcbiAgcmVhZG9ubHkgc3NtUGFyYW1ldGVyczogSUZTeFdpbmRvd3NQYXJhbWV0ZXJzO1xuICByZWFkb25seSBmc3hPYmplY3Q6IGF3c19mc3guQ2ZuRmlsZVN5c3RlbTtcbiAgcmVhZG9ubHkgcHJvcHM6IElGU3hXaW5kb3dzUHJvcHM7XG4gIGNvbnN0cnVjdG9yKHNjb3BlOiBDb25zdHJ1Y3QsIGlkOiBzdHJpbmcsIHByb3BzOiBJRlN4V2luZG93c1Byb3BzKSB7XG4gICAgc3VwZXIoc2NvcGUsIGlkKTtcbiAgICB0aGlzLnByb3BzID0gcHJvcHM7XG4gICAgdGhpcy5wcm9wcy5maWxlU3lzdGVtSW5Qcml2YXRlU3VibmV0ID1cbiAgICAgIHByb3BzLmZpbGVTeXN0ZW1JblByaXZhdGVTdWJuZXQgPz8gdHJ1ZTtcbiAgICB0aGlzLnByb3BzLnRocm91Z2hwdXRNYnBzID0gcHJvcHMudGhyb3VnaHB1dE1icHMgPz8gMTI4O1xuICAgIHRoaXMucHJvcHMuZmlsZVN5c3RlbVNpemUgPSBwcm9wcy5maWxlU3lzdGVtU2l6ZSA/PyAyMDA7XG4gICAgdGhpcy5wcm9wcy5tdWx0aUFaID0gcHJvcHMubXVsdGlBWiA/PyB0cnVlO1xuICAgIHRoaXMuc3NtUGFyYW1ldGVycyA9IHByb3BzLnNzbVBhcmFtZXRlcnMgPz8ge307XG4gICAgdGhpcy5zc21QYXJhbWV0ZXJzLmRuc0VuZHBvaW50ID1cbiAgICAgIHRoaXMuc3NtUGFyYW1ldGVycz8uZG5zRW5kcG9pbnQgPz8gJ0ZTeEVuZHBvaW50LUROUyc7XG5cbiAgICBpZiAodGhpcy5zc21QYXJhbWV0ZXJzLm5hbWVzcGFjZSkge1xuICAgICAgdGhpcy5zc21QYXJhbWV0ZXJzLm5hbWVzcGFjZSA9IGAke3RoaXMuc3NtUGFyYW1ldGVycy5uYW1lc3BhY2V9L3N0b3JhZ2UvZnN4YDtcbiAgICB9IGVsc2Uge1xuICAgICAgdGhpcy5zc21QYXJhbWV0ZXJzLm5hbWVzcGFjZSA9ICdjZGstc2t5bGlnaHQvc3RvcmFnZS9mc3gnO1xuICAgIH1cblxuICAgIGNvbnN0IHN1Ym5ldHMgPSB0aGlzLnByb3BzLnZwYy5zZWxlY3RTdWJuZXRzKHtcbiAgICAgIHN1Ym5ldFR5cGU6IHByb3BzLmZpbGVTeXN0ZW1JblByaXZhdGVTdWJuZXRcbiAgICAgICAgPyBlYzIuU3VibmV0VHlwZS5QUklWQVRFX1dJVEhfTkFUXG4gICAgICAgIDogZWMyLlN1Ym5ldFR5cGUuUFVCTElDLFxuICAgIH0pLnN1Ym5ldElkcztcblxuICAgIGNvbnN0IHdpbmRvd3NfY29uZmlndXJhdGlvbjogYXdzX2ZzeC5DZm5GaWxlU3lzdGVtLldpbmRvd3NDb25maWd1cmF0aW9uUHJvcGVydHkgPVxuICAgICAge1xuICAgICAgICB0aHJvdWdocHV0Q2FwYWNpdHk6IHRoaXMucHJvcHMudGhyb3VnaHB1dE1icHMsXG4gICAgICAgIGFjdGl2ZURpcmVjdG9yeUlkOiBwcm9wcy5kaXJlY3RvcnlJZCxcbiAgICAgICAgZGVwbG95bWVudFR5cGU6IHRoaXMucHJvcHMubXVsdGlBWiA/ICdNVUxUSV9BWl8xJyA6ICdTSU5HTEVfQVpfMicsXG4gICAgICAgIHByZWZlcnJlZFN1Ym5ldElkOiB0aGlzLnByb3BzLm11bHRpQVogPyBzdWJuZXRzWzBdIDogdW5kZWZpbmVkLFxuICAgICAgfTtcblxuICAgIGNvbnN0IHNnID0gbmV3IGVjMi5TZWN1cml0eUdyb3VwKHRoaXMsIGlkICsgJy1GU3hTRycsIHtcbiAgICAgIHZwYzogdGhpcy5wcm9wcy52cGMsXG4gICAgfSk7XG5cbiAgICAvLyBBbGxvdyBhY2Nlc3MgZnJvbSBpbnNpZGUgdGhlIFZQQ1xuICAgIHNnLmFkZEluZ3Jlc3NSdWxlKGVjMi5QZWVyLmlwdjQocHJvcHMudnBjLnZwY0NpZHJCbG9jayksIGVjMi5Qb3J0LmFsbFRjcCgpKTtcblxuICAgIGNvbnN0IGZzeF9wcm9wczogYXdzX2ZzeC5DZm5GaWxlU3lzdGVtUHJvcHMgPSB7XG4gICAgICBmaWxlU3lzdGVtVHlwZTogJ1dJTkRPV1MnLFxuICAgICAgc3VibmV0SWRzOiBwcm9wcy5tdWx0aUFaID8gW3N1Ym5ldHNbMF0sIHN1Ym5ldHNbMV1dIDogW3N1Ym5ldHNbMF1dLFxuICAgICAgd2luZG93c0NvbmZpZ3VyYXRpb246IHdpbmRvd3NfY29uZmlndXJhdGlvbixcbiAgICAgIHN0b3JhZ2VDYXBhY2l0eTogcHJvcHMuZmlsZVN5c3RlbVNpemUsXG4gICAgICBzZWN1cml0eUdyb3VwSWRzOiBbc2cuc2VjdXJpdHlHcm91cElkXSxcbiAgICB9O1xuXG4gICAgdGhpcy5mc3hPYmplY3QgPSBuZXcgYXdzX2ZzeC5DZm5GaWxlU3lzdGVtKFxuICAgICAgdGhpcyxcbiAgICAgIChpZCA9IGlkICsgJy1GU3hPYmplY3QnKSxcbiAgICAgIGZzeF9wcm9wcyxcbiAgICApO1xuXG4gICAgbmV3IGF3c19zc20uU3RyaW5nUGFyYW1ldGVyKHRoaXMsICdzc20tZG5zLWZzeEVuZHBvaW50Jywge1xuICAgICAgcGFyYW1ldGVyTmFtZTogYC8ke3RoaXMuc3NtUGFyYW1ldGVycy5uYW1lc3BhY2V9LyR7dGhpcy5zc21QYXJhbWV0ZXJzLmRuc0VuZHBvaW50fWAsXG4gICAgICBzdHJpbmdWYWx1ZTogdGhpcy5mc3hPYmplY3QuZ2V0QXR0KCdETlNOYW1lJykudG9TdHJpbmcoKSxcbiAgICB9KTtcbiAgfVxuXG4gIGNyZWF0ZVdvcmtlcihkb21haW5OYW1lOiBzdHJpbmcsIGRvbWFpblBhc3N3b3JkOiBJU2VjcmV0KTogRG9tYWluV2luZG93c05vZGUge1xuICAgIHJldHVybiBuZXcgRG9tYWluV2luZG93c05vZGUodGhpcywgJ0ZTeFdpbmRvd3NXb3JrZXInLCB7XG4gICAgICB2cGM6IHRoaXMucHJvcHMudnBjLFxuICAgICAgaW5zdGFuY2VUeXBlOiAndDMuc21hbGwnLFxuICAgICAgaWFtTWFuYWdlZFBvbGljaWVzTGlzdDogW1xuICAgICAgICBhd3NfaWFtLk1hbmFnZWRQb2xpY3kuZnJvbUF3c01hbmFnZWRQb2xpY3lOYW1lKFxuICAgICAgICAgICdBbWF6b25TU01NYW5hZ2VkSW5zdGFuY2VDb3JlJyxcbiAgICAgICAgKSxcbiAgICAgICAgYXdzX2lhbS5NYW5hZ2VkUG9saWN5LmZyb21Bd3NNYW5hZ2VkUG9saWN5TmFtZShcbiAgICAgICAgICAnU2VjcmV0c01hbmFnZXJSZWFkV3JpdGUnLFxuICAgICAgICApLFxuICAgICAgICBhd3NfaWFtLk1hbmFnZWRQb2xpY3kuZnJvbUF3c01hbmFnZWRQb2xpY3lOYW1lKFxuICAgICAgICAgICdBbWF6b25GU3hSZWFkT25seUFjY2VzcycsXG4gICAgICAgICksXG4gICAgICBdLFxuICAgICAgZG9tYWluTmFtZTogZG9tYWluTmFtZSxcbiAgICAgIHBhc3N3b3JkT2JqZWN0OiBkb21haW5QYXNzd29yZCxcbiAgICB9KTtcbiAgfVxuXG4gIGNyZWF0ZUZvbGRlcihcbiAgICB3b3JrZXI6IERvbWFpbldpbmRvd3NOb2RlLFxuICAgIGZvbGRlck5hbWU6IHN0cmluZyxcbiAgICBzZWNyZXROYW1lOiBJU2VjcmV0LFxuICApIHtcbiAgICB3b3JrZXIuc3RhcnRJbnN0YW5jZSgpO1xuICAgIHdvcmtlci5ydW5QU3dpdGhEb21haW5BZG1pbihcbiAgICAgIFtcbiAgICAgICAgYCRGU1ggPSAnJHt0aGlzLmZzeE9iamVjdFxuICAgICAgICAgIC5nZXRBdHQoJ0ROU05hbWUnKVxuICAgICAgICAgIC50b1N0cmluZygpfScgIyMgQW1hem9uIEZTeCBETlMgTmFtZWAsXG4gICAgICAgICckRlN4UFMgPSAoR2V0LUZTWEZpbGVTeXN0ZW0gfCA/IHskXy5ETlNOYW1lIC1jb250YWlucyAkRlNYfSkuV2luZG93c0NvbmZpZ3VyYXRpb24uUmVtb3RlQWRtaW5pc3RyYXRpb25FbmRwb2ludCcsXG4gICAgICAgIGAkRm9sZGVyTmFtZSA9ICcke2ZvbGRlck5hbWV9J2AsXG4gICAgICAgIGBbc3RyaW5nXSRTZWNyZXRBRCAgPSAnJHtzZWNyZXROYW1lfSdgLFxuICAgICAgICAnJFNlY3JldE9iaiA9IEdldC1TRUNTZWNyZXRWYWx1ZSAtU2VjcmV0SWQgJFNlY3JldEFEJyxcbiAgICAgICAgJ1tQU0N1c3RvbU9iamVjdF0kU2VjcmV0ID0gKCRTZWNyZXRPYmouU2VjcmV0U3RyaW5nICB8IENvbnZlcnRGcm9tLUpzb24pJyxcbiAgICAgICAgJyRwYXNzd29yZCAgID0gJFNlY3JldC5QYXNzd29yZCB8IENvbnZlcnRUby1TZWN1cmVTdHJpbmcgLWFzUGxhaW5UZXh0IC1Gb3JjZScsXG4gICAgICAgIFwiICR1c2VybmFtZSAgID0gJFNlY3JldC5Eb21haW4gKyAnXFxcXCcgKyAkU2VjcmV0LlVzZXJJRCBcIixcbiAgICAgICAgJyRkb21haW5fYWRtaW5fY3JlZGVudGlhbCA9IE5ldy1PYmplY3QgU3lzdGVtLk1hbmFnZW1lbnQuQXV0b21hdGlvbi5QU0NyZWRlbnRpYWwoJHVzZXJuYW1lLCRwYXNzd29yZCknLFxuICAgICAgICAnIyBDcmVhdGUgdGhlIGZvbGRlciAodGhlIHNoYXJlZCBkcml2ZXIgdG8gdGhlIGhvc3RzKScsXG4gICAgICAgICdOZXctSXRlbSAtSXRlbVR5cGUgRGlyZWN0b3J5IC1OYW1lICRGb2xkZXJOYW1lIC1QYXRoIFxcXFxcXFxcJEZTWFxcXFxEJFxcXFwnLFxuICAgICAgICAnIyBTZXQgTlRGUyBQZXJtaXNzaW9ucycsXG4gICAgICAgICcjIEFDTCcsXG4gICAgICAgICckQUNMID0gR2V0LUFjbCBcXFxcXFxcXCRGU3hcXFxcRCRcXFxcJEZvbGRlck5hbWUnLFxuICAgICAgICAnJHBlcm1pc3Npb24gPSBcIk5UIEFVVEhPUklUWVxcXFxBdXRoZW50aWNhdGVkIFVzZXJzXCIsXCJGdWxsQ29udHJvbFwiLFwiQWxsb3dcIicsXG4gICAgICAgICckQXIgPSBOZXctT2JqZWN0IFN5c3RlbS5TZWN1cml0eS5BY2Nlc3NDb250cm9sLkZpbGVTeXN0ZW1BY2Nlc3NSdWxlICRwZXJtaXNzaW9uJyxcbiAgICAgICAgJyRBQ0wuU2V0QWNjZXNzUnVsZSgkQXIpJyxcbiAgICAgICAgJ1NldC1BY2wgXFxcXFxcXFwkRlNYXFxcXEQkXFxcXCRGb2xkZXJOYW1lICRBQ0wnLFxuICAgICAgICAnIyBDcmVhdGUgdGhlIFNoYXJlIGFuZCBzZXQgdGhlIHNoYXJlIHBlcm1pc3Npb25zJyxcbiAgICAgICAgJyRTZXNzaW9uID0gTmV3LVBTU2Vzc2lvbiAtQ29tcHV0ZXJOYW1lICRGU3hQUyAtQ29uZmlndXJhdGlvbk5hbWUgRnN4UmVtb3RlQWRtaW4nLFxuICAgICAgICAnSW1wb3J0LVBzU2Vzc2lvbiAkU2Vzc2lvbicsXG4gICAgICAgICdOZXctRlN4U21iU2hhcmUgLU5hbWUgJEZvbGRlck5hbWUgLVBhdGggXCJEOlxcXFwkRm9sZGVyTmFtZVwiIC1EZXNjcmlwdGlvbiBcIlNoYXJlZCBmb2xkZXIgd2l0aCBnTVNBIGFjY2Vzc1wiIC1DcmVkZW50aWFsICRkb21haW5fYWRtaW5fY3JlZGVudGlhbCAtRm9sZGVyRW51bWVyYXRpb25Nb2RlIEFjY2Vzc0Jhc2VkJyxcbiAgICAgICAgJyRhY2Nlc3NMaXN0PVwiTlQgQVVUSE9SSVRZXFxcXEF1dGhlbnRpY2F0ZWQgVXNlcnNcIicsXG4gICAgICAgICdHcmFudC1GU3hTbWJTaGFyZWFjY2VzcyAtTmFtZSAkRm9sZGVyTmFtZSAtQWNjb3VudE5hbWUgJGFjY2Vzc0xpc3QgLWFjY2Vzc1JpZ2h0IEZ1bGwgLUNvbmZpcm06JGZhbHNlJyxcbiAgICAgICAgJ0Rpc2Nvbm5lY3QtUFNTZXNzaW9uIC1TZXNzaW9uICRTZXNzaW9uJyxcbiAgICAgICAgJ1N0b3AtQ29tcHV0ZXIgLUNvbXB1dGVyTmFtZSBsb2NhbGhvc3QnLFxuICAgICAgXSxcbiAgICAgICdjcmVhdGVGb2xkZXInLFxuICAgICk7XG4gIH1cbn1cbiJdfQ==