"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Between = exports.BinaryCondition = exports.BeginsWith = exports.BaseKeyCondition = exports.shapeAddition = void 0;
/**
 * Generates an addition to the schema
 *
 * @param options the options to produced a stringfied addition
 *
 * @returns the following shape:
 *
 * ```
 * prefix name interfaces directives {
 *   field
 *   field
 *   ...
 * }
 * ```
 */
function shapeAddition(options) {
    const typeName = () => { return options.name ? ` ${options.name}` : ''; };
    const interfaces = generateInterfaces(options.interfaceTypes);
    const directives = generateDirectives({
        directives: options.directives,
        modes: options.modes,
    });
    return options.fields.reduce((acc, field) => `${acc}  ${field}\n`, `${options.prefix}${typeName()}${interfaces}${directives} {\n`) + '}';
}
exports.shapeAddition = shapeAddition;
/**
 * Utility class to represent DynamoDB key conditions.
 */
class BaseKeyCondition {
    and(cond) {
        return new (class extends BaseKeyCondition {
            constructor(left, right) {
                super();
                this.left = left;
                this.right = right;
            }
            renderCondition() {
                return `${this.left.renderCondition()} AND ${this.right.renderCondition()}`;
            }
            keyNames() {
                return concatAndDedup(this.left.keyNames(), this.right.keyNames());
            }
            args() {
                return concatAndDedup(this.left.args(), this.right.args());
            }
        })(this, cond);
    }
    renderExpressionNames() {
        return this.keyNames()
            .map((keyName) => {
            return `"#${keyName}" : "${keyName}"`;
        })
            .join(', ');
    }
    renderExpressionValues() {
        return this.args()
            .map((arg) => {
            return `":${arg}" : $util.dynamodb.toDynamoDBJson($ctx.args.${arg})`;
        })
            .join(', ');
    }
}
exports.BaseKeyCondition = BaseKeyCondition;
/**
 * Utility class to represent DynamoDB "begins_with" key conditions.
 */
class BeginsWith extends BaseKeyCondition {
    constructor(keyName, arg) {
        super();
        this.keyName = keyName;
        this.arg = arg;
    }
    renderCondition() {
        return `begins_with(#${this.keyName}, :${this.arg})`;
    }
    keyNames() {
        return [this.keyName];
    }
    args() {
        return [this.arg];
    }
}
exports.BeginsWith = BeginsWith;
/**
 * Utility class to represent DynamoDB binary key conditions.
 */
class BinaryCondition extends BaseKeyCondition {
    constructor(keyName, op, arg) {
        super();
        this.keyName = keyName;
        this.op = op;
        this.arg = arg;
    }
    renderCondition() {
        return `#${this.keyName} ${this.op} :${this.arg}`;
    }
    keyNames() {
        return [this.keyName];
    }
    args() {
        return [this.arg];
    }
}
exports.BinaryCondition = BinaryCondition;
/**
 * Utility class to represent DynamoDB "between" key conditions.
 */
class Between extends BaseKeyCondition {
    constructor(keyName, arg1, arg2) {
        super();
        this.keyName = keyName;
        this.arg1 = arg1;
        this.arg2 = arg2;
    }
    renderCondition() {
        return `#${this.keyName} BETWEEN :${this.arg1} AND :${this.arg2}`;
    }
    keyNames() {
        return [this.keyName];
    }
    args() {
        return [this.arg1, this.arg2];
    }
}
exports.Between = Between;
function concatAndDedup(left, right) {
    return left.concat(right).filter((elem, index, self) => {
        return index === self.indexOf(elem);
    });
}
/**
 * Utility function to generate interfaces for object types
 *
 * @param interfaceTypes the interfaces this object type implements
 */
function generateInterfaces(interfaceTypes) {
    if (!interfaceTypes || interfaceTypes.length === 0)
        return '';
    return interfaceTypes.reduce((acc, interfaceType) => `${acc} ${interfaceType.name} &`, ' implements').slice(0, -2);
}
/**
 * Utility function to generate directives
 */
function generateDirectives(options) {
    if (!options.directives || options.directives.length === 0)
        return '';
    // reduce over all directives and get string version of the directive
    // pass in the auth modes for checks to happen on compile time
    return options.directives.reduce((acc, directive) => `${acc}${directive._bindToAuthModes(options.modes).toString()}${options.delimiter ?? ' '}`, ' ').slice(0, -1);
}
//# sourceMappingURL=data:application/json;base64,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