"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.Type = exports.Directive = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const aws_appsync_1 = require("aws-cdk-lib/aws-appsync");
/**
 * Directives for types
 *
 * i.e. @aws_iam or @aws_subscribe
 *
 */
class Directive {
    constructor(statement, options) {
        this.statement = statement;
        this.mode = options?.mode;
        this.mutationFields = options?.mutationFields;
    }
    /**
     * Add the @aws_iam directive
     */
    static iam() {
        return new Directive('@aws_iam', { mode: aws_appsync_1.AuthorizationType.IAM });
    }
    /**
     * Add the @aws_oidc directive
     */
    static oidc() {
        return new Directive('@aws_oidc', { mode: aws_appsync_1.AuthorizationType.OIDC });
    }
    /**
     * Add the @aws_api_key directive
     */
    static apiKey() {
        return new Directive('@aws_api_key', { mode: aws_appsync_1.AuthorizationType.API_KEY });
    }
    /**
     * Add the @aws_auth or @aws_cognito_user_pools directive
     *
     * @param groups the groups to allow access to
     */
    static cognito(...groups) {
        if (groups.length === 0) {
            throw new Error(`Cognito authorization requires at least one Cognito group to be supplied. Received: ${groups.length}`);
        }
        // this function creates the cognito groups as a string (i.e. ["group1", "group2", "group3"])
        const stringify = (array) => {
            return array.reduce((acc, element) => `${acc}"${element}", `, '').slice(0, -2);
        };
        return new Directive(`@aws_auth(cognito_groups: [${stringify(groups)}])`, { mode: aws_appsync_1.AuthorizationType.USER_POOL });
    }
    /**
     * Add the @aws_subscribe directive. Only use for top level Subscription type.
     *
     * @param mutations the mutation fields to link to
     */
    static subscribe(...mutations) {
        if (mutations.length === 0) {
            throw new Error(`Subscribe directive requires at least one mutation field to be supplied. Received: ${mutations.length}`);
        }
        // this function creates the subscribe directive as a string (i.e. ["mutation_field_1", "mutation_field_2"])
        const stringify = (array) => {
            return array.reduce((acc, mutation) => `${acc}"${mutation}", `, '').slice(0, -2);
        };
        return new Directive(`@aws_subscribe(mutations: [${stringify(mutations)}])`, { mutationFields: mutations });
    }
    /**
     * Add a custom directive
     *
     * @param statement - the directive statement to append
     */
    static custom(statement) {
        return new Directive(statement);
    }
    /**
     * Method called when the stringifying Directive for schema generation
     *
     * @param modes the authorization modes
     *
     * @internal
     */
    _bindToAuthModes(modes) {
        this.modes = modes;
        return this;
    }
    /**
     * Generate the directive statement
     */
    toString() {
        if (this.modes && this.mode && !this.modes.some((mode) => mode === this.mode)) {
            throw new Error(`No Authorization Type ${this.mode} declared in GraphQL Api.`);
        }
        if (this.mode === aws_appsync_1.AuthorizationType.USER_POOL && this.modes && this.modes.length > 1) {
            this.statement = this.statement.replace('@aws_auth', '@aws_cognito_user_pools');
        }
        return this.statement;
    }
}
exports.Directive = Directive;
_a = JSII_RTTI_SYMBOL_1;
Directive[_a] = { fqn: "awscdk-appsync-utils.Directive", version: "0.0.438" };
/**
 * Enum containing the Types that can be used to define ObjectTypes
 */
var Type;
(function (Type) {
    /**
     * `ID` scalar type is a unique identifier. `ID` type is serialized similar to `String`.
     *
     * Often used as a key for a cache and not intended to be human-readable.
     */
    Type["ID"] = "ID";
    /**
     * `String` scalar type is a free-form human-readable text.
     */
    Type["STRING"] = "String";
    /**
     * `Int` scalar type is a signed non-fractional numerical value.
     */
    Type["INT"] = "Int";
    /**
     * `Float` scalar type is a signed double-precision fractional value.
     */
    Type["FLOAT"] = "Float";
    /**
     * `Boolean` scalar type is a boolean value: true or false.
     */
    Type["BOOLEAN"] = "Boolean";
    /**
     * `AWSDate` scalar type represents a valid extended `ISO 8601 Date` string.
     *
     * In other words, accepts date strings in the form of `YYYY-MM-DD`. It accepts time zone offsets.
     *
     * @see https://en.wikipedia.org/wiki/ISO_8601#Calendar_dates
     */
    Type["AWS_DATE"] = "AWSDate";
    /**
     * `AWSTime` scalar type represents a valid extended `ISO 8601 Time` string.
     *
     * In other words, accepts date strings in the form of `hh:mm:ss.sss`. It accepts time zone offsets.
     *
     * @see https://en.wikipedia.org/wiki/ISO_8601#Times
     */
    Type["AWS_TIME"] = "AWSTime";
    /**
     * `AWSDateTime` scalar type represents a valid extended `ISO 8601 DateTime` string.
     *
     * In other words, accepts date strings in the form of `YYYY-MM-DDThh:mm:ss.sssZ`. It accepts time zone offsets.
     *
     * @see https://en.wikipedia.org/wiki/ISO_8601#Combined_date_and_time_representations
     */
    Type["AWS_DATE_TIME"] = "AWSDateTime";
    /**
     * `AWSTimestamp` scalar type represents the number of seconds since `1970-01-01T00:00Z`.
     *
     * Timestamps are serialized and deserialized as numbers.
     */
    Type["AWS_TIMESTAMP"] = "AWSTimestamp";
    /**
     * `AWSEmail` scalar type represents an email address string (i.e.`username@example.com`)
     */
    Type["AWS_EMAIL"] = "AWSEmail";
    /**
     * `AWSJson` scalar type represents a JSON string.
     */
    Type["AWS_JSON"] = "AWSJSON";
    /**
     * `AWSURL` scalar type represetns a valid URL string.
     *
     * URLs wihtout schemes or contain double slashes are considered invalid.
     */
    Type["AWS_URL"] = "AWSURL";
    /**
     * `AWSPhone` scalar type represents a valid phone number. Phone numbers maybe be whitespace delimited or hyphenated.
     *
     * The number can specify a country code at the beginning, but is not required for US phone numbers.
     */
    Type["AWS_PHONE"] = "AWSPhone";
    /**
     * `AWSIPAddress` scalar type respresents a valid `IPv4` of `IPv6` address string.
     */
    Type["AWS_IP_ADDRESS"] = "AWSIPAddress";
    /**
     * Type used for Intermediate Types
     * (i.e. an interface or an object type)
     */
    Type["INTERMEDIATE"] = "INTERMEDIATE";
})(Type = exports.Type || (exports.Type = {}));
//# sourceMappingURL=data:application/json;base64,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