import { Duration } from 'aws-cdk-lib';
import { ISourceApiAssociation } from 'aws-cdk-lib/aws-appsync';
import { SingletonFunction } from 'aws-cdk-lib/aws-lambda';
import { Construct, IConstruct } from 'constructs';
/**
 * This interface for the provider of the custom resource that will be used to initiate a merge operation during Cloudformation update.
 */
export interface ISourceApiAssociationMergeOperationProvider extends IConstruct {
    /**
       * Service token which is used for identifying the handler used for the merge operation custom resource.
       */
    readonly serviceToken: string;
    /**
       * This function associates a source api association with the provider.
       * This method can be used for adding permissions to merge a specific source api association to the custom resource provider.
       * @param sourceApiAssociation The association to associate.
       */
    associateSourceApiAssociation(sourceApiAssociation: ISourceApiAssociation): void;
}
/**
 * Properties for SourceApiAssociationMergeOperationProvider.
 */
export interface SourceApiAssociationMergeOperationProviderProps {
    /**
       * Time between calls to the polling Lambda function which determines whether the merge operation is finished or not.
       *
       * @default Duration.seconds(5)
       */
    readonly pollingInterval?: Duration;
    /**
       * Total timeout in waiting for the source api association merge operation to complete.
       *
       * @default Duration.minutes(15)
       */
    readonly totalTimeout?: Duration;
}
/**
 * SourceApiAssociationMergeProvider class is responsible for constructing the custom resource that will be used for initiating the
 * source API merge during a Cloudformation update.
 */
export declare class SourceApiAssociationMergeOperationProvider extends Construct implements ISourceApiAssociationMergeOperationProvider {
    /**
       * Service token for the resource provider.
       */
    readonly serviceToken: string;
    /**
       * The lambda function responsible for kicking off the merge operation.
       */
    readonly schemaMergeLambda: SingletonFunction;
    /**
       * The lambda function response for ensuring that the merge operation finished.
       */
    readonly sourceApiStablizationLambda: SingletonFunction;
    constructor(scope: Construct, id: string, props: SourceApiAssociationMergeOperationProviderProps);
    associateSourceApiAssociation(sourceApiAssociation: ISourceApiAssociation): void;
}
/**
 * Properties for SourceApiAssociationMergeOperation which handles triggering a merge operation as a custom resource
 * during a Cloudformation stack update.
 */
export interface SourceApiAssociationMergeOperationProps {
    /**
       * The source api association resource which will be merged.
       */
    readonly sourceApiAssociation: ISourceApiAssociation;
    /**
       * The merge operation provider construct which is responsible for configuring the Lambda resource that will be invoked during
       * Cloudformation update.
       */
    readonly mergeOperationProvider?: ISourceApiAssociationMergeOperationProvider;
    /**
       * The version identifier for the schema merge operation. Any change to the version identifier will trigger a merge on the next
       * update. Use the version identifier property to control when the source API metadata is merged.
       *
       * @default null
       */
    readonly versionIdentifier?: string;
    /**
       * Flag indicating whether the source api should be merged on every CFN update or not.
       * If set to true and there are no changes to the source API, this will result in a no-op merge operation.
       *
       * @default False
       */
    readonly alwaysMergeOnStackUpdate?: boolean;
}
/**
 * The SourceApiAssociationMergeOperation triggers a merge of a source API during a Cloudformation stack update.
 * This can be used to propagate changes from the source API to the Merged API when the association is using type MANUAL_MERGE.
 * If the merge operation fails, it will fail the Cloudformation update and rollback the stack.
 */
export declare class SourceApiAssociationMergeOperation extends Construct {
    constructor(scope: Construct, id: string, props: SourceApiAssociationMergeOperationProps);
    /**
     * Get an existing merge operation provider from the current stack or create a new stack scoped merge operation provider.
     * @returns SourceApiAssociationMergeOperationProvider
     */
    private getOrCreateMergeOperationProvider;
}
