"use strict";
var _a, _b;
Object.defineProperty(exports, "__esModule", { value: true });
exports.SourceApiAssociationMergeOperation = exports.SourceApiAssociationMergeOperationProvider = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const crypto_1 = require("crypto");
const path = require("path");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const aws_iam_1 = require("aws-cdk-lib/aws-iam");
const aws_lambda_1 = require("aws-cdk-lib/aws-lambda");
const custom_resources_1 = require("aws-cdk-lib/custom-resources");
const constructs_1 = require("constructs");
/**
 * SourceApiAssociationMergeProvider class is responsible for constructing the custom resource that will be used for initiating the
 * source API merge during a Cloudformation update.
 */
class SourceApiAssociationMergeOperationProvider extends constructs_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        this.schemaMergeLambda = new aws_lambda_1.SingletonFunction(this, 'MergeSourceApiSchemaLambda', {
            runtime: aws_lambda_1.Runtime.NODEJS_20_X,
            code: aws_lambda_1.Code.fromAsset(path.join(__dirname, 'mergeSourceApiSchemaHandler')),
            handler: 'index.onEvent',
            timeout: aws_cdk_lib_1.Duration.minutes(2),
            uuid: '6148f39b-95bb-47e7-8a35-40adb8b93a7b',
        });
        this.sourceApiStablizationLambda = new aws_lambda_1.SingletonFunction(this, 'PollSourceApiMergeLambda', {
            runtime: aws_lambda_1.Runtime.NODEJS_20_X,
            code: aws_lambda_1.Code.fromAsset(path.join(__dirname, 'mergeSourceApiSchemaHandler')),
            handler: 'index.isComplete',
            timeout: aws_cdk_lib_1.Duration.minutes(2),
            uuid: '163e01ec-6f29-4bf4-b3b1-11245b00a6bc',
        });
        const provider = new custom_resources_1.Provider(this, 'SchemaMergeOperationProvider', {
            onEventHandler: this.schemaMergeLambda,
            isCompleteHandler: this.sourceApiStablizationLambda,
            queryInterval: props.pollingInterval ?? aws_cdk_lib_1.Duration.seconds(5),
            totalTimeout: props.totalTimeout ?? aws_cdk_lib_1.Duration.minutes(15),
        });
        this.serviceToken = provider.serviceToken;
    }
    associateSourceApiAssociation(sourceApiAssociation) {
        this.schemaMergeLambda.addToRolePolicy(new aws_iam_1.PolicyStatement({
            effect: aws_iam_1.Effect.ALLOW,
            resources: [sourceApiAssociation.associationArn],
            actions: ['appsync:StartSchemaMerge'],
        }));
        this.sourceApiStablizationLambda.addToRolePolicy(new aws_iam_1.PolicyStatement({
            effect: aws_iam_1.Effect.ALLOW,
            resources: [sourceApiAssociation.associationArn],
            actions: ['appsync:GetSourceApiAssociation'],
        }));
    }
}
exports.SourceApiAssociationMergeOperationProvider = SourceApiAssociationMergeOperationProvider;
_a = JSII_RTTI_SYMBOL_1;
SourceApiAssociationMergeOperationProvider[_a] = { fqn: "awscdk-appsync-utils.SourceApiAssociationMergeOperationProvider", version: "0.0.438" };
/**
 * The SourceApiAssociationMergeOperation triggers a merge of a source API during a Cloudformation stack update.
 * This can be used to propagate changes from the source API to the Merged API when the association is using type MANUAL_MERGE.
 * If the merge operation fails, it will fail the Cloudformation update and rollback the stack.
 */
class SourceApiAssociationMergeOperation extends constructs_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        if (!props.alwaysMergeOnStackUpdate && !props.versionIdentifier) {
            throw new Error('A version identifier must be specified when the alwaysMergeOnStackUpdate flag is false');
        }
        var mergeOperationProvider = props.mergeOperationProvider;
        if (!mergeOperationProvider) {
            mergeOperationProvider = this.getOrCreateMergeOperationProvider();
        }
        mergeOperationProvider.associateSourceApiAssociation(props.sourceApiAssociation);
        var properties = {
            associationId: props.sourceApiAssociation.associationId,
            mergedApiIdentifier: props.sourceApiAssociation.mergedApi.arn,
            sourceApiIdentifier: props.sourceApiAssociation.sourceApi.arn,
        };
        // When versionIdentifier property is passed, it allows the developer to explicitly control when the source api is merged via the merge operation.
        // Changing the version identifier will allow you to control whether the merge operation occurs for a specific merge operation of not.
        if (props.versionIdentifier) {
            properties.versionIdentifier = props.versionIdentifier;
        }
        // When alwaysMergeOnStackUpdate flag is set to true, everytime the stack is deployed we create a new custom resource which ensures that this
        // merge operation is always ran. This potentially will result in no-ops.
        if (props.alwaysMergeOnStackUpdate) {
            properties.alwaysUpdate = crypto_1.randomUUID();
        }
        // Custom resource for the merge of this specific source api association.
        const customResource = new aws_cdk_lib_1.CustomResource(this, 'SourceApiSchemaMergeOperation', {
            serviceToken: mergeOperationProvider.serviceToken,
            resourceType: 'Custom::AppSyncSourceApiMergeOperation',
            properties: {
                ...properties,
            },
        });
        // If a reference to the source API exists,
        // add a dependency on all children of the source api in order to ensure that this resource is created at the end.
        props.sourceApiAssociation.sourceApi.node.children.forEach((child) => {
            if (aws_cdk_lib_1.CfnResource.isCfnResource(child)) {
                customResource.node.addDependency(child);
            }
            if (constructs_1.Construct.isConstruct(child) && child.node.defaultChild && aws_cdk_lib_1.CfnResource.isCfnResource(child.node.defaultChild)) {
                customResource.node.addDependency(child.node.defaultChild);
            }
        });
    }
    /**
     * Get an existing merge operation provider from the current stack or create a new stack scoped merge operation provider.
     * @returns SourceApiAssociationMergeOperationProvider
     */
    getOrCreateMergeOperationProvider() {
        const constructName = 'SchemaMergeOperationProvider';
        const stack = aws_cdk_lib_1.Stack.of(this);
        const existing = stack.node.tryFindChild(constructName);
        if (!existing) {
            return new SourceApiAssociationMergeOperationProvider(stack, 'SchemaMergeOperationProvider', {
                pollingInterval: aws_cdk_lib_1.Duration.seconds(30),
            });
        }
        else {
            return existing;
        }
    }
}
exports.SourceApiAssociationMergeOperation = SourceApiAssociationMergeOperation;
_b = JSII_RTTI_SYMBOL_1;
SourceApiAssociationMergeOperation[_b] = { fqn: "awscdk-appsync-utils.SourceApiAssociationMergeOperation", version: "0.0.438" };
//# sourceMappingURL=data:application/json;base64,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