"""
[File: Storage]
===============
Purpose: File upload/download operations via backend storage API
Data Flow: local file → upload() → backend OSS → resource_id → get_url() → download()
Core Data Structures:
  - resource_id: string - Unique identifier for stored file
  - file_path: string - Local filesystem path
Related Files:
  @poping/client.py → _HTTPClient for API calls
  @poping/_agent.py → Session.storage property

Usage Examples:
  Auto-detection (recommended, OpenAI-style):
    import poping
    poping.set(api_key="...")
    storage = poping.Storage()  # Auto-detects global client
    rid = storage.upload("/path/to/file.png")

  Backward compatibility (explicit client):
    from poping import Poping
    client = Poping(api_key="...")
    storage = poping.Storage(client=client._http)  # Explicit HTTP client
    rid = storage.upload("/path/to/file.png")
"""

from typing import Optional
from pathlib import Path
import mimetypes
from .uri_parser import URIParser


class Storage:
    """
    Storage operations for file upload/download

    Provides interface to backend storage API (OSS/S3).
    Files are uploaded, stored with unique IDs, and can be downloaded.

    Auto-detection behavior:
      If `client` is not provided, this class auto-detects the globally
      configured Poping client set via `poping.set(...)` and uses its
      internal HTTP client.
    """

    def __init__(
        self,
        client=None,
        session_id: Optional[str] = None,
        client_id: Optional[str] = None,
        project_id: Optional[str] = None,
        end_user_id: Optional[str] = None,
    ):
        """
        Initialize storage with optional HTTP client and session scope

        Args:
            client: Optional `_HTTPClient` instance for API calls. If not
                provided, the global client configured via `poping.set(...)`
                is used automatically.
            session_id: Optional session identifier for URI expansion
            client_id: Backward-compatible alias for end_user_id. Prefer
                passing end_user_id explicitly in new code.
            project_id: Optional project identifier for project-scoped storage APIs
            end_user_id: End user identifier within your application. Used to
                expand `@context://` and `@storage://` URIs in the v2 storage system.

        Raises:
            ValueError: If no global client is configured and `client` is not provided

        Example (auto-detection):
            import poping
            poping.set(api_key="...")
            storage = poping.Storage()

        Example (explicit client, backward compatible):
            from poping import Poping
            client = Poping(api_key="...")
            storage = poping.Storage(client=client._http)

        Input: Optional _HTTPClient
        Output: Storage instance
        Role in Flow: Entry point for storage operations
        """
        if client is None:
            # Auto-detect global client (OpenAI-style), matching poping.agent()
            from . import get_client

            global_client = get_client()
            if global_client is None:
                raise ValueError(
                    "Storage not configured. Either:\n"
                    "1. Call poping.set(api_key='...') first, or\n"
                    "2. Pass client explicitly: Storage(client=...)"
                )

            self.client = global_client._http
        else:
            self.client = client

        # Session-scoped storage (for @context:// artifacts)
        self.session_id = session_id
        # Logical end-user identifier used in URIs (@context[<client_id>/<session_id>]://, @storage[<client_id>]://)
        self.client_id = client_id
        # Optional project context; when omitted, backend defaults are used where available
        self.project_id = project_id
        # Optional explicit end_user_id for project storage APIs; falls back to client_id
        self.end_user_id = end_user_id or client_id  # For URI expansion and v2 storage APIs

    def _expand_uri(self, uri: str) -> str:
        """
        Expand simplified URI if session context available

        Args:
            uri: Input URI (simplified or complete)

        Returns:
            Expanded URI or original if no context/already complete

        Example:
            # With context
            storage = Storage(client=..., client_id="alice", session_id="sess_001")
            storage._expand_uri("@storage://file.pdf")
            # Returns: "@storage[alice]://file.pdf"

            # Without context - returns unchanged
            storage = Storage(client=...)
            storage._expand_uri("@storage://file.pdf")
            # Returns: "@storage://file.pdf"
        """
        if not uri.startswith("@"):
            # Not a URI, return as-is (might be raw resource_id)
            return uri

        if not (self.client_id or self.session_id):
            # No context available, return original
            return uri

        try:
            return URIParser.expand(uri, client_id=self.client_id, session_id=self.session_id)
        except ValueError:
            # Expansion failed (unsupported scheme or missing context)
            # Return original URI
            return uri

    def upload(self, file_path: str) -> str:
        """
        Upload file to backend storage

        Args:
            file_path: Path to local file

        Returns:
            uri: Full storage URI (e.g., "@context://images/abc123.png" for session artifacts,
                 "@storage://abc123.pdf" for persistent user files)

        Raises:
            FileNotFoundError: If file doesn't exist
            APIError: If upload fails

        Example:
            # Session context upload (temporary)
            storage = poping.Storage(client_id="alice", session_id="sess_001")
            uri = storage.upload("/path/to/image.png")
            # Returns: "@context://images/550e8400-e29b-41d4-a716-446655440000.png"

            # Persistent user file upload
            storage = poping.Storage(client_id="alice")
            uri = storage.upload("/path/to/document.pdf")
            # Returns: "@storage://550e8400-e29b-41d4-a716-446655440000.pdf"

            # Use directly in agent messages (no manual concatenation!)
            # conv.chat(f"Analyze this image: {uri}")
        """
        file_path_obj = Path(file_path)

        if not file_path_obj.exists():
            raise FileNotFoundError(f"File not found: {file_path}")

        # Session-scoped upload → use v2 session artifacts API (@context://texts/... etc.)
        if self.session_id:
            # Infer folder from MIME type (align with backend session_routes.py)
            mime_type, _ = mimetypes.guess_type(str(file_path_obj))
            mime_type = mime_type or "application/octet-stream"

            if mime_type.startswith("image/"):
                folder = "images"
            elif mime_type.startswith("audio/"):
                folder = "audios"
            elif mime_type.startswith("text/"):
                folder = "texts"
            else:
                folder = "artifacts"

            with open(file_path, "rb") as f:
                files = {"file": (file_path_obj.name, f, "application/octet-stream")}
                endpoint = f"/api/v1/sessions/{self.session_id}/artifacts"
                params = {"folder": folder}
                result = self.client._request(
                    method="POST",
                    endpoint=endpoint,
                    params=params,
                    files=files,
                )

            # ArtifactUploadResponse includes "uri" (e.g., "@context://texts/…")
            return result["uri"]

        # Persistent user/storage upload (no session) → use v2 storage routes (@storage://...)
        # Requires project_id and end_user_id (or client_id as fallback).
        if not self.project_id or not self.end_user_id:
            raise ValueError(
                "Storage.upload() outside a session requires project_id and end_user_id.\n"
                "Hint: use `with agent.session(...): conv.storage.upload(...)` for session artifacts, or\n"
                "initialize Storage with project_id and end_user_id for persistent storage."
            )

        with open(file_path, "rb") as f:
            files = {"file": (file_path_obj.name, f, "application/octet-stream")}
            endpoint = "/api/v1/storage/files"
            params = {
                "project_id": self.project_id,
                "end_user_id": self.end_user_id,
                "folder_path": "",
            }
            result = self.client._request(
                method="POST",
                endpoint=endpoint,
                params=params,
                files=files,
            )

        # FileUploadResponse includes "uri" (e.g., "@storage[alice]://file_xxx.txt")
        return result["uri"]

    def get_url(self, uri: str, expires_in: int = 3600) -> str:
        """
        Get temporary download URL for a resource

        Args:
            uri: Storage URI (e.g., "@storage://file.pdf") or raw resource_id
            expires_in: URL validity in seconds (default: 1 hour)

        Returns:
            url: Temporary download URL

        Example:
            url = storage.get_url("@storage://file.pdf", expires_in=3600)
            # With client_id context, expands to "@storage[alice]://file.pdf"
        """
        # Expand simplified URI if context available
        expanded_uri = self._expand_uri(uri)

        # Session artifacts (@context://...) → use session download endpoint to build URL
        if expanded_uri.startswith("@context://") and self.session_id:
            # The backend exposes /api/v1/sessions/{id}/artifacts/download?uri=...
            # This returns a streaming response; here we just construct a direct URL.
            # NOTE: SDK callers that need an actual URL can rely on this pattern.
            base_url = getattr(self.client, "base_url", None) or getattr(
                getattr(self.client, "_http", None), "base_url", ""
            )
            if not base_url:
                # Fallback to relative URL
                return f"/api/v1/sessions/{self.session_id}/artifacts/download?uri={expanded_uri}"
            return (
                f"{base_url.rstrip('/')}/api/v1/sessions/{self.session_id}/artifacts/download"
                f"?uri={expanded_uri}"
            )

        # Persistent storage URIs (@storage:// or @storage[...]://)
        if expanded_uri.startswith("@storage://") or expanded_uri.startswith("@storage["):
            if not self.project_id or not self.end_user_id:
                raise ValueError(
                    "get_url() for @storage URIs requires project_id and end_user_id.\n"
                    "Initialize Storage with these fields or use session-scoped @context URIs."
                )

            # For v2 storage routes, there is no dedicated signed-URL endpoint; instead
            # we construct the download URL using the file_id extracted from metadata.
            # Since the SDK only has the URI, we return the direct download endpoint
            # that the frontend also uses.
            # The backend expects file_id, not URI, so callers should prefer using
            # Storage.download() for SDK workflows. Here we return a best-effort URL.
            base_url = getattr(self.client, "base_url", None) or getattr(
                getattr(self.client, "_http", None), "base_url", ""
            )
            if not base_url:
                return "/api/v1/storage/files"
            return f"{base_url.rstrip('/')}/api/v1/storage/files"

        # Fallback: keep legacy behaviour if someone passes a raw resource_id
        response = self.client._request(
            method="GET",
            endpoint=f"/api/v1/storage/{expanded_uri}/url",
            params={"expires_in": expires_in},
        )
        return response["url"]

    def download(self, uri: str, output_path: str) -> str:
        """
        Download file from storage to local path

        Args:
            uri: Storage URI (e.g., "@storage://file.pdf") or raw resource_id
            output_path: Local path to save file

        Returns:
            output_path: Path where file was saved

        Example:
            storage.download("@storage://file.pdf", "/tmp/downloaded.pdf")
            # With client_id context, expands to "@storage[alice]://file.pdf"
        """
        # Expand simplified URI if context available
        expanded_uri = self._expand_uri(uri)

        # Session artifacts (@context://...) → use session download endpoint
        if expanded_uri.startswith("@context://") and self.session_id:
            endpoint = f"/api/v1/sessions/{self.session_id}/artifacts/download"
            resp = self.client._request(
                method="GET",
                endpoint=endpoint,
                params={"uri": expanded_uri},
                stream=True,
            )
        else:
            # Persistent storage and legacy URIs → use legacy download endpoint as fallback.
            # New user storage flows should prefer project-scoped APIs, but we keep this
            # for backward compatibility.
            resp = self.client._request(
                method="GET",
                endpoint=f"/api/v1/storage/{expanded_uri}/download",
                stream=True,
            )

        # Write to output file
        output_path_obj = Path(output_path)
        output_path_obj.parent.mkdir(parents=True, exist_ok=True)

        with open(output_path, "wb") as f:
            for chunk in resp.iter_content(chunk_size=8192):
                f.write(chunk)

        return output_path
