from __future__ import annotations

from PySide6.QtCore import Signal, Qt
from PySide6.QtGui import QAction, QKeySequence, QFont, QFontDatabase
from PySide6.QtWidgets import QToolBar


class ToolBar(QToolBar):
    boldRequested = Signal()
    italicRequested = Signal()
    underlineRequested = Signal()
    strikeRequested = Signal()
    codeRequested = Signal()
    headingRequested = Signal(int)
    bulletsRequested = Signal()
    numbersRequested = Signal()
    alignRequested = Signal(Qt.AlignmentFlag)
    historyRequested = Signal()

    def __init__(self, parent=None):
        super().__init__("Format", parent)
        self.setObjectName("Format")
        self.setToolButtonStyle(Qt.ToolButtonTextOnly)
        self._build_actions()
        self._apply_toolbar_styles()

    def _build_actions(self):
        self.actBold = QAction("Bold", self)
        self.actBold.setShortcut(QKeySequence.Bold)
        self.actBold.triggered.connect(self.boldRequested)

        self.actItalic = QAction("Italic", self)
        self.actItalic.setShortcut(QKeySequence.Italic)
        self.actItalic.triggered.connect(self.italicRequested)

        self.actUnderline = QAction("Underline", self)
        self.actUnderline.setShortcut(QKeySequence.Underline)
        self.actUnderline.triggered.connect(self.underlineRequested)

        self.actStrike = QAction("Strikethrough", self)
        self.actStrike.setShortcut("Ctrl+-")
        self.actStrike.triggered.connect(self.strikeRequested)

        self.actCode = QAction("Inline code", self)
        self.actCode.setShortcut("Ctrl+`")
        self.actCode.triggered.connect(self.codeRequested)

        # Headings
        self.actH1 = QAction("Heading 1", self)
        self.actH2 = QAction("Heading 2", self)
        self.actH3 = QAction("Heading 3", self)
        self.actNormal = QAction("Normal text", self)
        self.actH1.setShortcut("Ctrl+1")
        self.actH2.setShortcut("Ctrl+2")
        self.actH3.setShortcut("Ctrl+3")
        self.actNormal.setShortcut("Ctrl+N")
        self.actH1.triggered.connect(lambda: self.headingRequested.emit(24))
        self.actH2.triggered.connect(lambda: self.headingRequested.emit(18))
        self.actH3.triggered.connect(lambda: self.headingRequested.emit(14))
        self.actNormal.triggered.connect(lambda: self.headingRequested.emit(0))

        # Lists
        self.actBullets = QAction("Bulleted list", self)
        self.actBullets.triggered.connect(self.bulletsRequested)
        self.actNumbers = QAction("Numbered list", self)
        self.actNumbers.triggered.connect(self.numbersRequested)

        # Alignment
        self.actAlignL = QAction("Align left", self)
        self.actAlignC = QAction("Align center", self)
        self.actAlignR = QAction("Align right", self)
        self.actAlignL.triggered.connect(lambda: self.alignRequested.emit(Qt.AlignLeft))
        self.actAlignC.triggered.connect(
            lambda: self.alignRequested.emit(Qt.AlignHCenter)
        )
        self.actAlignR.triggered.connect(
            lambda: self.alignRequested.emit(Qt.AlignRight)
        )

        # History button
        self.actHistory = QAction("History", self)
        self.actHistory.triggered.connect(self.historyRequested)

        self.addActions(
            [
                self.actBold,
                self.actItalic,
                self.actUnderline,
                self.actStrike,
                self.actCode,
                self.actH1,
                self.actH2,
                self.actH3,
                self.actNormal,
                self.actBullets,
                self.actNumbers,
                self.actAlignL,
                self.actAlignC,
                self.actAlignR,
                self.actHistory,
            ]
        )

    def _apply_toolbar_styles(self):
        self._style_letter_button(self.actBold, "B", bold=True)
        self._style_letter_button(self.actItalic, "I", italic=True)
        self._style_letter_button(self.actUnderline, "U", underline=True)
        self._style_letter_button(self.actStrike, "S", strike=True)

        # Monospace look for code; use a fixed font
        code_font = QFontDatabase.systemFont(QFontDatabase.FixedFont)
        self._style_letter_button(self.actCode, "</>", custom_font=code_font)

        # Headings
        self._style_letter_button(self.actH1, "H1")
        self._style_letter_button(self.actH2, "H2")
        self._style_letter_button(self.actH3, "H3")
        self._style_letter_button(self.actNormal, "N")

        # Lists
        self._style_letter_button(self.actBullets, "•")
        self._style_letter_button(self.actNumbers, "1.")

        # Alignment
        self._style_letter_button(self.actAlignL, "L")
        self._style_letter_button(self.actAlignC, "C")
        self._style_letter_button(self.actAlignR, "R")

        # History
        self._style_letter_button(self.actHistory, "View History")

    def _style_letter_button(
        self,
        action: QAction,
        text: str,
        *,
        bold: bool = False,
        italic: bool = False,
        underline: bool = False,
        strike: bool = False,
        custom_font: QFont | None = None,
    ):
        btn = self.widgetForAction(action)
        if not btn:
            return
        btn.setText(text)
        f = custom_font if custom_font is not None else QFont(btn.font())
        if custom_font is None:
            f.setBold(bold)
            f.setItalic(italic)
            f.setUnderline(underline)
            f.setStrikeOut(strike)
        btn.setFont(f)

        # Keep accessibility/tooltip readable
        btn.setToolTip(action.text())
        btn.setAccessibleName(action.text())
