#!/usr/bin/env node
"use strict";
/**
 * automatically creates a module for any CloudFormation namespaces that do not
 * have an AWS construct library.
 */
Object.defineProperty(exports, "__esModule", { value: true });
const path = require("path");
const pkglint = require("@aws-cdk/pkglint");
const fs = require("fs-extra");
const cfnspec = require("../lib");
// don't be a prude:
/* eslint-disable no-console */
/* eslint-disable quote-props */
async function main() {
    const root = path.join(__dirname, '..', '..');
    if (path.basename(root) !== '@aws-cdk') {
        throw new Error(`Something went wrong. We expected ${root} to be the "packages/@aws-cdk" directory. Did you move me?`);
    }
    // eslint-disable-next-line @typescript-eslint/no-require-imports
    const version = require('../package.json').version;
    // iterate over all cloudformation namespaces
    for (const namespace of cfnspec.namespaces()) {
        const module = pkglint.createModuleDefinitionFromCfnNamespace(namespace);
        const lowcaseModuleName = module.moduleName.toLocaleLowerCase();
        const packagePath = path.join(root, module.moduleName);
        // we already have a module for this namesapce, move on.
        if (await fs.pathExists(packagePath)) {
            const packageJsonPath = path.join(packagePath, 'package.json');
            // eslint-disable-next-line @typescript-eslint/no-require-imports
            const packageJson = require(packageJsonPath);
            let scopes = packageJson['cdk-build'].cloudformation;
            if (typeof scopes === 'string') {
                scopes = [scopes];
            }
            if (scopes.indexOf(namespace) !== -1) {
                // V2-style module is already modeled in the root package, nothing to be done!
                continue;
            }
            else if (await fs.pathExists(path.join(root, `${module.moduleFamily}-${module.moduleBaseName}`.toLocaleLowerCase()))) {
                // V2-style package already has it's own package (legacy behavior), nothing to be done!
                continue;
            }
            else {
                // V2-style package needs to be added to it's "V1" package... Get down to business!
                console.error(`Adding ${namespace} to ${module.packageName}`);
                scopes.push(namespace);
                packageJson['cdk-build'].cloudformation = scopes;
                await fs.writeJson(packageJsonPath, packageJson, { encoding: 'utf-8', spaces: 2 });
                const indexTsPath = path.join(packagePath, 'lib', 'index.ts');
                const indexTs = [
                    (await fs.readFile(indexTsPath, { encoding: 'utf8' })).trimRight(),
                    `// ${namespace} CloudFormation Resources:`,
                    `export * from './${lowcaseModuleName}.generated';`,
                ].join('\n');
                await fs.writeFile(indexTsPath, indexTs, { encoding: 'utf8' });
                continue;
            }
        }
        async function write(relativePath, contents) {
            const fullPath = path.join(packagePath, relativePath);
            const dir = path.dirname(fullPath);
            await fs.mkdirp(dir);
            let data;
            if (typeof contents === 'string') {
                data = contents.trimLeft(); // trim first newline
            }
            else if (Array.isArray(contents)) {
                data = contents.join('\n');
            }
            else if (typeof contents === 'object') {
                data = JSON.stringify(contents, undefined, 2);
            }
            else {
                throw new Error('Invalid type of contents: ' + contents);
            }
            await fs.writeFile(fullPath, data + '\n');
        }
        console.log(`generating module for ${module.packageName}...`);
        const description = `${namespace} Construct Library`;
        await write('package.json', {
            name: module.packageName,
            version,
            description,
            main: 'lib/index.js',
            types: 'lib/index.d.ts',
            jsii: {
                outdir: 'dist',
                projectReferences: true,
                targets: {
                    dotnet: {
                        namespace: module.dotnetPackage,
                        packageId: module.dotnetPackage,
                        signAssembly: true,
                        assemblyOriginatorKeyFile: '../../key.snk',
                        iconUrl: 'https://raw.githubusercontent.com/aws/aws-cdk/master/logo/default-256-dark.png',
                    },
                    java: {
                        package: `${module.javaGroupId}.${module.javaPackage}`,
                        maven: {
                            groupId: module.javaGroupId,
                            artifactId: module.javaArtifactId,
                        },
                    },
                    python: {
                        classifiers: [
                            'Framework :: AWS CDK',
                            'Framework :: AWS CDK :: 1',
                        ],
                        distName: module.pythonDistName,
                        module: module.pythonModuleName,
                    },
                },
                metadata: {
                    jsii: {
                        rosetta: {
                            strict: true,
                        },
                    },
                },
            },
            repository: {
                type: 'git',
                url: 'https://github.com/aws/aws-cdk.git',
                directory: `packages/${module.packageName}`,
            },
            homepage: 'https://github.com/aws/aws-cdk',
            scripts: {
                build: 'cdk-build',
                watch: 'cdk-watch',
                lint: 'cdk-lint',
                test: 'cdk-test',
                integ: 'cdk-integ',
                pkglint: 'pkglint -f',
                package: 'cdk-package',
                awslint: 'cdk-awslint',
                cfn2ts: 'cfn2ts',
                'build+test': 'yarn build && yarn test',
                'build+test+package': 'yarn build+test && yarn package',
                compat: 'cdk-compat',
                gen: 'cfn2ts',
                'rosetta:extract': 'yarn --silent jsii-rosetta extract',
                'build+extract': 'yarn build && yarn rosetta:extract',
                'build+test+extract': 'yarn build+test && yarn rosetta:extract',
            },
            'cdk-build': {
                cloudformation: namespace,
                jest: true,
                env: {
                    AWSLINT_BASE_CONSTRUCT: 'true',
                },
            },
            keywords: [
                'aws',
                'cdk',
                'constructs',
                namespace,
                module.moduleName,
            ],
            author: {
                name: 'Amazon Web Services',
                url: 'https://aws.amazon.com',
                organization: true,
            },
            license: 'Apache-2.0',
            devDependencies: {
                '@aws-cdk/assertions': version,
                '@aws-cdk/cdk-build-tools': version,
                '@aws-cdk/cfn2ts': version,
                '@aws-cdk/pkglint': version,
                '@types/jest': '^26.0.22',
            },
            dependencies: {
                '@aws-cdk/core': version,
            },
            peerDependencies: {
                '@aws-cdk/core': version,
            },
            engines: {
                node: '>= 10.13.0 <13 || >=13.7.0',
            },
            stability: 'experimental',
            maturity: 'cfn-only',
            awscdkio: {
                announce: false,
            },
            publishConfig: {
                tag: 'latest',
            },
        });
        await write('.gitignore', [
            '*.js',
            '*.js.map',
            '*.d.ts',
            'tsconfig.json',
            'node_modules',
            '*.generated.ts',
            'dist',
            '.jsii',
            '',
            '.LAST_BUILD',
            '.nyc_output',
            'coverage',
            '.nycrc',
            '.LAST_PACKAGE',
            '*.snk',
            'nyc.config.js',
            '!.eslintrc.js',
            '!jest.config.js',
            'junit.xml',
        ]);
        await write('.npmignore', [
            '# Don\'t include original .ts files when doing `npm pack`',
            '*.ts',
            '!*.d.ts',
            'coverage',
            '.nyc_output',
            '*.tgz',
            '',
            'dist',
            '.LAST_PACKAGE',
            '.LAST_BUILD',
            '!*.js',
            '',
            '# Include .jsii',
            '!.jsii',
            '',
            '*.snk',
            '',
            '*.tsbuildinfo',
            '',
            'tsconfig.json',
            '',
            '.eslintrc.js',
            'jest.config.js',
            '',
            '# exclude cdk artifacts',
            '**/cdk.out',
            'junit.xml',
            'test/',
            '!*.lit.ts',
        ]);
        await write('lib/index.ts', [
            `// ${namespace} CloudFormation Resources:`,
            `export * from './${lowcaseModuleName}.generated';`,
        ]);
        await write(`test/${lowcaseModuleName}.test.ts`, [
            "import '@aws-cdk/assertions';",
            "import {} from '../lib';",
            '',
            "test('No tests are specified for this package', () => {",
            '  expect(true).toBe(true);',
            '});',
        ]);
        await pkglint.createLibraryReadme(namespace, path.join(packagePath, 'README.md'));
        await write('.eslintrc.js', [
            "const baseConfig = require('@aws-cdk/cdk-build-tools/config/eslintrc');",
            "baseConfig.parserOptions.project = __dirname + '/tsconfig.json';",
            'module.exports = baseConfig;',
        ]);
        await write('jest.config.js', [
            "const baseConfig = require('@aws-cdk/cdk-build-tools/config/jest.config');",
            'module.exports = baseConfig;',
        ]);
        await write('rosetta/default.ts-fixture', [
            "import { Construct } from 'constructs';",
            "import { Stack } from '@aws-cdk/core';",
            '',
            'class MyStack extends Stack {',
            '  constructor(scope: Construct, id: string) {',
            '    /// here',
            '  }',
            '}',
        ]);
        const templateDir = path.join(__dirname, 'template');
        for (const file of await fs.readdir(templateDir)) {
            await fs.copy(path.join(templateDir, file), path.join(packagePath, file));
        }
        await addDependencyToMegaPackage(path.join('@aws-cdk', 'cloudformation-include'), module.packageName, version, ['dependencies', 'peerDependencies']);
        await addDependencyToMegaPackage('aws-cdk-lib', module.packageName, version, ['devDependencies']);
        await addDependencyToMegaPackage('monocdk', module.packageName, version, ['devDependencies']);
        await addDependencyToMegaPackage('decdk', module.packageName, version, ['dependencies']);
    }
}
/**
 * A few of our packages (e.g., decdk, aws-cdk-lib) require a dependency on every service package.
 * This automates adding the dependency (and peer dependency) to the package.json.
 */
async function addDependencyToMegaPackage(megaPackageName, packageName, version, dependencyTypes) {
    const packageJsonPath = path.join(__dirname, '..', '..', '..', megaPackageName, 'package.json');
    const packageJson = JSON.parse(await fs.readFile(packageJsonPath, 'utf8'));
    dependencyTypes.forEach(dependencyType => {
        const unorderedDeps = {
            ...packageJson[dependencyType],
            [packageName]: version,
        };
        packageJson[dependencyType] = {};
        Object.keys(unorderedDeps).sort().forEach(k => {
            packageJson[dependencyType][k] = unorderedDeps[k];
        });
    });
    await fs.writeFile(packageJsonPath, JSON.stringify(packageJson, null, 2) + '\n');
}
main().catch(e => {
    console.error(e);
    process.exit(1);
});
//# sourceMappingURL=data:application/json;base64,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