"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.unionOf = exports.diffKeyedEntities = exports.deepEqual = void 0;
/**
 * Compares two objects for equality, deeply. The function handles arguments that are
 * +null+, +undefined+, arrays and objects. For objects, the function will not take the
 * object prototype into account for the purpose of the comparison, only the values of
 * properties reported by +Object.keys+.
 *
 * If both operands can be parsed to equivalent numbers, will return true.
 * This makes diff consistent with CloudFormation, where a numeric 10 and a literal "10"
 * are considered equivalent.
 *
 * @param lvalue the left operand of the equality comparison.
 * @param rvalue the right operand of the equality comparison.
 *
 * @returns +true+ if both +lvalue+ and +rvalue+ are equivalent to each other.
 */
function deepEqual(lvalue, rvalue) {
    if (lvalue === rvalue) {
        return true;
    }
    // CloudFormation allows passing strings into boolean-typed fields
    if (((typeof lvalue === 'string' && typeof rvalue === 'boolean') ||
        (typeof lvalue === 'boolean' && typeof rvalue === 'string')) &&
        lvalue.toString() === rvalue.toString()) {
        return true;
    }
    // allows a numeric 10 and a literal "10" to be equivalent;
    // this is consistent with CloudFormation.
    if ((typeof lvalue === 'string' || typeof rvalue === 'string') &&
        safeParseFloat(lvalue) === safeParseFloat(rvalue)) {
        return true;
    }
    if (typeof lvalue !== typeof rvalue) {
        return false;
    }
    if (Array.isArray(lvalue) !== Array.isArray(rvalue)) {
        return false;
    }
    if (Array.isArray(lvalue) /* && Array.isArray(rvalue) */) {
        if (lvalue.length !== rvalue.length) {
            return false;
        }
        for (let i = 0; i < lvalue.length; i++) {
            if (!deepEqual(lvalue[i], rvalue[i])) {
                return false;
            }
        }
        return true;
    }
    if (typeof lvalue === 'object' /* && typeof rvalue === 'object' */) {
        if (lvalue === null || rvalue === null) {
            // If both were null, they'd have been ===
            return false;
        }
        const keys = Object.keys(lvalue);
        if (keys.length !== Object.keys(rvalue).length) {
            return false;
        }
        for (const key of keys) {
            if (!rvalue.hasOwnProperty(key)) {
                return false;
            }
            if (key === 'DependsOn') {
                if (!dependsOnEqual(lvalue[key], rvalue[key])) {
                    return false;
                }
                ;
                // check differences other than `DependsOn`
                continue;
            }
            if (!deepEqual(lvalue[key], rvalue[key])) {
                return false;
            }
        }
        return true;
    }
    // Neither object, nor array: I deduce this is primitive type
    // Primitive type and not ===, so I deduce not deepEqual
    return false;
}
exports.deepEqual = deepEqual;
/**
 * Compares two arguments to DependsOn for equality.
 *
 * @param lvalue the left operand of the equality comparison.
 * @param rvalue the right operand of the equality comparison.
 *
 * @returns +true+ if both +lvalue+ and +rvalue+ are equivalent to each other.
 */
function dependsOnEqual(lvalue, rvalue) {
    // allows ['Value'] and 'Value' to be equal
    if (Array.isArray(lvalue) !== Array.isArray(rvalue)) {
        const array = Array.isArray(lvalue) ? lvalue : rvalue;
        const nonArray = Array.isArray(lvalue) ? rvalue : lvalue;
        if (array.length === 1 && deepEqual(array[0], nonArray)) {
            return true;
        }
        return false;
    }
    // allows arrays passed to DependsOn to be equivalent irrespective of element order
    if (Array.isArray(lvalue) && Array.isArray(rvalue)) {
        if (lvalue.length !== rvalue.length) {
            return false;
        }
        for (let i = 0; i < lvalue.length; i++) {
            for (let j = 0; j < lvalue.length; j++) {
                if ((!deepEqual(lvalue[i], rvalue[j])) && (j === lvalue.length - 1)) {
                    return false;
                }
                break;
            }
        }
        return true;
    }
    return false;
}
/**
 * Produce the differences between two maps, as a map, using a specified diff function.
 *
 * @param oldValue  the old map.
 * @param newValue  the new map.
 * @param elementDiff the diff function.
 *
 * @returns a map representing the differences between +oldValue+ and +newValue+.
 */
function diffKeyedEntities(oldValue, newValue, elementDiff) {
    const result = {};
    for (const logicalId of unionOf(Object.keys(oldValue || {}), Object.keys(newValue || {}))) {
        const oldElement = oldValue && oldValue[logicalId];
        const newElement = newValue && newValue[logicalId];
        result[logicalId] = elementDiff(oldElement, newElement, logicalId);
    }
    return result;
}
exports.diffKeyedEntities = diffKeyedEntities;
/**
 * Computes the union of two sets of strings.
 *
 * @param lv the left set of strings.
 * @param rv the right set of strings.
 *
 * @returns a new array containing all elemebts from +lv+ and +rv+, with no duplicates.
 */
function unionOf(lv, rv) {
    const result = new Set(lv);
    for (const v of rv) {
        result.add(v);
    }
    return new Array(...result);
}
exports.unionOf = unionOf;
/**
 * A parseFloat implementation that does the right thing for
 * strings like '0.0.0'
 * (for which JavaScript's parseFloat() returns 0).
 * We return NaN for all of these strings that do not represent numbers,
 * and so comparing them fails,
 * and doesn't short-circuit the diff logic.
 */
function safeParseFloat(str) {
    return Number(str);
}
//# sourceMappingURL=data:application/json;base64,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