"""Tests for the `AdvancedFilterBackend` class."""

from unittest.mock import Mock

from django.db import models
from django.test import TestCase
from django_filters import FilterSet

from plugins.filter import AdvancedFilterBackend, FactoryBuiltFilterSet


class SampleModel(models.Model):
    """Sample database model for testing"""

    class Meta:
        app_label = 'plugins'

    bool_field = models.BooleanField()
    char_field = models.CharField(max_length=100)
    integer_field = models.IntegerField()
    date_field = models.DateField(auto_now_add=True)
    time_field = models.TimeField(auto_now_add=True)
    datetime_field = models.DateTimeField(auto_now_add=True)
    text_field = models.TextField()
    unknown_field = models.Field()


class GetFiltersetClassMethod(TestCase):
    """Test the fetching of filterset classes for different view/querysets via the `get_filterset_class` method."""

    def test_user_provided_filterset(self) -> None:
        """Verify existing filtersets are not overwritten"""

        # Mock a view with a filterset already defined
        mock_view = Mock()
        mock_view.filterset_class = Mock(spec=FilterSet)

        result = AdvancedFilterBackend().get_filterset_class(mock_view)
        self.assertEqual(result, mock_view.filterset_class, 'User defined filterset class was overwritten')

    def test_auto_generated_filterset(self) -> None:
        """Verify automatically generated filtersets use the correct field types."""

        # Mock a view without a predefined filterset
        mock_view = Mock()
        mock_view.filterset_class = None
        mock_view.filterset_fields = None

        mock_queryset = Mock()
        mock_queryset.model = SampleModel

        backend = AdvancedFilterBackend()
        result = backend.get_filterset_class(mock_view, mock_queryset)

        # Test the factory method returned an autogenerated filterset
        self.assertTrue(issubclass(result, FactoryBuiltFilterSet))
        self.assertEqual(result.Meta.model, SampleModel, 'Factory built filterset has incorrect DB model')

        returned_fields = result.Meta.fields
        filters = backend.field_filter_map
        self.assertEqual(filters[models.BooleanField], returned_fields['bool_field'], 'bool type field has incorrect filters')
        self.assertEqual(filters[models.CharField], returned_fields['char_field'], 'char type field has incorrect filters')
        self.assertEqual(filters[models.IntegerField], returned_fields['integer_field'], 'integer type field has incorrect filters')
        self.assertEqual(filters[models.DateField], returned_fields['date_field'], 'date type field has incorrect filters')
        self.assertEqual(filters[models.TimeField], returned_fields['time_field'], 'time type field has incorrect filters')
        self.assertEqual(filters[models.DateTimeField], returned_fields['datetime_field'], 'datetime type field has incorrect filters')
        self.assertEqual(filters[models.TextField], returned_fields['text_field'], 'text type field has incorrect filters')
        self.assertNotIn('unknown_field', filters, "Field with unknown typer was not ignored")
