# Database-Driven Model Training

## Overview

This example demonstrates end-to-end ML training using database as the data source via DataFlow integration.

## Features

1. **Database-Sourced Training** - Train models directly from PostgreSQL/SQLite tables
2. **AI-Powered Feature Selection** - LLM analyzes schema and recommends features
3. **Automated Pipeline** - Extract, transform, train, and deploy
4. **Real-Time Inference** - Predictions with database context enrichment
5. **Model Versioning** - Track model metadata in database

## Workflow

```
Database → Feature Analysis (AI) → Model Training → Metadata Storage → Inference
```

### Step 1: Data Extraction
- Fetch training data from database table
- AI analyzes table schema and data samples

### Step 2: Feature Engineering
- LLM recommends optimal feature columns
- Identifies target variable
- Suggests preprocessing steps

### Step 3: Model Training
- Extract features and target from data
- Train model (simplified in example)
- Store model metadata in database

### Step 4: Real-Time Inference
- Fetch model metadata from database
- Enrich input with historical context
- Generate predictions with explanations

## Usage

### Basic Training

```python
from dataclasses import dataclass
from kailash_dataflow import DataFlow
from kaizen.integrations.dataflow import DBTrainingPipeline

@dataclass
class TrainingConfig:
    llm_provider: str = "openai"
    model: str = "gpt-4"
    temperature: float = 0.3

# Connect to database
db = DataFlow("postgresql://localhost/customer_db")

# Define schema
@db.model
class Customer:
    id: int
    age: int
    income: float
    purchases: int
    churn_risk: str

# Train model from database
trainer = DBTrainingPipeline(config=TrainingConfig(), db=db)
result = trainer.train_from_database(
    table="Customer",
    model_objective="Predict customer churn risk"
)

print(f"Model ID: {result['model_id']}")
print(f"Accuracy: {result['accuracy']}")
print(f"Features: {result['features']}")
```

### Real-Time Inference

```python
from kaizen.integrations.dataflow import InferencePipeline

pipeline = InferencePipeline(config=TrainingConfig(), db=db)

prediction = pipeline.infer_with_db_context(
    model_id=result['model_id'],
    input_data={"age": 35, "income": 75000, "purchases": 12},
    store_result=True
)

print(f"Prediction: {prediction['prediction']}")
print(f"Confidence: {prediction['confidence']}")
print(f"Explanation: {prediction['explanation']}")
```

### Automated Pipeline

```python
from kaizen.integrations.dataflow import PipelineOrchestrator

orchestrator = PipelineOrchestrator(config=TrainingConfig(), db=db)

result = orchestrator.create_pipeline(
    pipeline_name="customer_churn_pipeline",
    data_sources=["Customer", "Transactions"],
    objective="Predict customer churn with full pipeline"
)

print(f"Steps executed: {result['steps_executed']}")
print(f"Metrics: {result['metrics']}")
```

## Requirements

```bash
pip install kailash[dataflow]
# or
pip install kailash-dataflow
```

## Configuration

Set environment variables:
```bash
export DATABASE_URL="postgresql://user:pass@localhost/db"
export OPENAI_API_KEY="your-key-here"
```

## Database Schema

The example creates these tables:
- `Customer` - Training data
- `model_metadata` - Model versioning and tracking
- `inference_results` - Inference history and auditing
- `pipeline_metrics` - Pipeline performance tracking

## Key Concepts

### Database-Driven AI
Unlike traditional ML pipelines that extract data to files, this approach:
- Keeps data in database (no export/import)
- Uses database as single source of truth
- Stores model metadata alongside data
- Enables real-time inference with context

### AI-Powered Feature Engineering
The LLM analyzes:
- Table schema (column names and types)
- Sample data (first 100 rows)
- Model objective (business goal)

And recommends:
- Which columns to use as features
- Target variable to predict
- Required preprocessing steps
- Optimal model configuration

### Context-Enriched Inference
Each prediction is enriched with:
- Historical data from database
- Related entity information
- User behavior patterns
- Temporal trends

## Performance

- **Training**: Handles 10K+ rows efficiently
- **Inference**: <100ms with database lookup
- **Storage**: Model metadata stored in milliseconds
- **Scalability**: PostgreSQL-backed for production

## Production Considerations

1. **Model Versioning** - Track all models in database
2. **Audit Trail** - Store all inferences for compliance
3. **Performance Monitoring** - Track accuracy over time
4. **Data Privacy** - Keep sensitive data in database
5. **Rollback** - Easy model version rollback via metadata

## Next Steps

- Add feature preprocessing pipeline
- Implement model evaluation metrics
- Add hyperparameter tuning
- Create model deployment workflow
- Add A/B testing capabilities
