"""
HumanApprovalAgent - Production-Ready Human-in-the-Loop Decision Making

Zero-config usage:
    from kaizen.agents import HumanApprovalAgent

    # Auto-approval for testing
    agent = HumanApprovalAgent()
    result = await agent.run_async(prompt="Process payment $1000")
    print(f"Approved: {result['_human_approved']}")

Progressive configuration:
    # Custom approval callback
    def my_approval_callback(result):
        print(f"Review: {result}")
        response = input("Approve? (y/n): ")
        return response.lower() == 'y', "User decision"

    agent = HumanApprovalAgent(
        approval_callback=my_approval_callback,
        llm_provider="openai",
        model="gpt-4"
    )

Environment variable support:
    KAIZEN_LLM_PROVIDER=openai
    KAIZEN_MODEL=gpt-4
    KAIZEN_TEMPERATURE=0.3
    KAIZEN_MAX_TOKENS=300
"""

import os
from dataclasses import dataclass, field, replace
from typing import Any, Callable, Dict, List, Optional, Tuple

from kailash.nodes.base import NodeMetadata
from kaizen.core.base_agent import BaseAgent
from kaizen.signatures import InputField, OutputField, Signature
from kaizen.strategies.human_in_loop import HumanInLoopStrategy


class DecisionSignature(Signature):
    """Signature for approval-based decision making."""

    prompt: str = InputField(desc="Decision prompt")
    decision: str = OutputField(desc="Recommended decision")


@dataclass
class HumanApprovalConfig:
    """
    Configuration for Human Approval Agent.

    All parameters have sensible defaults and can be overridden via:
    1. Constructor arguments (highest priority)
    2. Environment variables (KAIZEN_*)
    3. Default values (lowest priority)
    """

    # LLM configuration
    llm_provider: str = field(
        default_factory=lambda: os.getenv("KAIZEN_LLM_PROVIDER", "openai")
    )
    model: str = field(default_factory=lambda: os.getenv("KAIZEN_MODEL", "gpt-4"))
    temperature: float = field(
        default_factory=lambda: float(os.getenv("KAIZEN_TEMPERATURE", "0.3"))
    )
    max_tokens: int = field(
        default_factory=lambda: int(os.getenv("KAIZEN_MAX_TOKENS", "300"))
    )

    # Human-in-loop configuration
    approval_callback: Optional[Callable[[Dict[str, Any]], Tuple[bool, str]]] = None

    # Technical configuration
    timeout: int = 30
    retry_attempts: int = 3
    provider_config: Dict[str, Any] = field(default_factory=dict)


class HumanApprovalAgent(BaseAgent):
    """
    Production-ready Human Approval Agent using HumanInLoopStrategy.

    Features:
    - Zero-config with auto-approval for testing
    - Request human approval before returning results
    - Track approval history for audit trails
    - Support custom approval callbacks (CLI, web form, webhook)
    - Built-in error handling and logging via BaseAgent

    Inherits from BaseAgent:
    - Signature-based decision pattern
    - Human-in-loop execution via HumanInLoopStrategy
    - Error handling (ErrorHandlingMixin)
    - Performance tracking (PerformanceMixin)
    - Structured logging (LoggingMixin)

    Use Cases:
    - Financial transactions requiring approval
    - Content moderation before publishing
    - Critical decisions with human oversight
    - Quality control checkpoints
    - Compliance workflows (SOX, GDPR)

    Performance:
    - Synchronous approval (waits for human decision)
    - Approval history tracking for audit trails
    - Configurable approval callback

    Usage:
        # Zero-config (auto-approval for testing)
        import asyncio
        agent = HumanApprovalAgent()

        result = asyncio.run(agent.run_async(prompt="Process payment $100"))
        print(f"Approved: {result['_human_approved']}")

        # With custom callback
        def my_callback(result):
            print(f"Review: {result}")
            response = input("Approve? (y/n): ")
            return response.lower() == 'y', "User decision"

        agent = HumanApprovalAgent(approval_callback=my_callback)
        result = asyncio.run(agent.run_async(prompt="Delete customer data"))

        # Check approval history
        history = agent.get_approval_history()
        for record in history:
            print(f"Approved: {record['approved']}, Feedback: {record['feedback']}")
    """

    # Node metadata for Studio discovery
    metadata = NodeMetadata(
        name="HumanApprovalAgent",
        description="Human-in-the-loop decision making with approval callbacks and audit trails",
        version="1.0.0",
        tags={"ai", "kaizen", "human-in-loop", "approval", "compliance", "audit"},
    )

    def __init__(
        self,
        llm_provider: Optional[str] = None,
        model: Optional[str] = None,
        temperature: Optional[float] = None,
        max_tokens: Optional[int] = None,
        approval_callback: Optional[
            Callable[[Dict[str, Any]], Tuple[bool, str]]
        ] = None,
        timeout: Optional[int] = None,
        retry_attempts: Optional[int] = None,
        provider_config: Optional[Dict[str, Any]] = None,
        config: Optional[HumanApprovalConfig] = None,
        mcp_servers: Optional[List[Dict]] = None,
        **kwargs,
    ):
        """
        Initialize Human Approval Agent with zero-config defaults.

        Args:
            llm_provider: Override default LLM provider
            model: Override default model
            temperature: Override default temperature
            max_tokens: Override default max tokens
            approval_callback: Custom approval callback (auto-approval if None)
            timeout: Override default timeout
            retry_attempts: Override default retry attempts
            provider_config: Additional provider-specific configuration
            config: Full config object (overrides individual params)            mcp_servers: Optional MCP server configurations for tool discovery
        """
        # If config object provided, use it; otherwise build from parameters
        if config is None:
            config = HumanApprovalConfig()

            # Override defaults with provided parameters
            if llm_provider is not None:
                config = replace(config, llm_provider=llm_provider)
            if model is not None:
                config = replace(config, model=model)
            if temperature is not None:
                config = replace(config, temperature=temperature)
            if max_tokens is not None:
                config = replace(config, max_tokens=max_tokens)
            if approval_callback is not None:
                config = replace(config, approval_callback=approval_callback)
            if timeout is not None:
                config = replace(config, timeout=timeout)
            if retry_attempts is not None:
                config = replace(config, retry_attempts=retry_attempts)
            if provider_config is not None:
                config = replace(config, provider_config=provider_config)

        # Merge timeout into provider_config
        if config.timeout and (
            not config.provider_config or "timeout" not in config.provider_config
        ):
            provider_cfg = (
                config.provider_config.copy() if config.provider_config else {}
            )
            provider_cfg["timeout"] = config.timeout
            config = replace(config, provider_config=provider_cfg)

        # Use HumanInLoopStrategy with approval callback
        strategy = HumanInLoopStrategy(approval_callback=config.approval_callback)

        # Initialize BaseAgent
        super().__init__(
            config=config,  # Auto-converted to BaseAgentConfig
            signature=DecisionSignature(),
            strategy=strategy,
            mcp_servers=mcp_servers,
            **kwargs,
        )

        self.approval_config = config

    async def run_async(self, prompt: str, **kwargs) -> Dict[str, Any]:
        """
        Make decision with human approval requirement.

        Overrides BaseAgent.run_async() to add approval workflow.

        Args:
            prompt: Decision prompt requiring approval
            **kwargs: Additional keyword arguments for BaseAgent.run_async()

        Returns:
            Dict[str, Any]: Result with decision and approval metadata

        Raises:
            RuntimeError: If human rejects the decision

        Example:
            >>> import asyncio
            >>> agent = HumanApprovalAgent()
            >>> result = asyncio.run(agent.run_async(prompt="Transfer funds"))
            >>> if result["_human_approved"]:
            ...     print("Proceed with transfer")
        """
        # For demo purposes, generate mock result then request approval
        # In production, strategy would execute agent then request approval
        mock_result = {"prompt": prompt, "decision": f"Placeholder result for {prompt}"}

        # Request human approval
        approved, feedback = self.strategy.approval_callback(mock_result)

        # Record approval decision
        approval_record = {
            "result": mock_result.copy(),
            "approved": approved,
            "feedback": feedback,
        }
        self.strategy.approval_history.append(approval_record)

        if not approved:
            # Rejected - raise error with feedback
            raise RuntimeError(f"Human rejected result: {feedback}")

        # Approved - add approval metadata and return
        mock_result["_human_approved"] = True
        mock_result["_approval_feedback"] = feedback

        return mock_result

    def get_approval_history(self) -> List[Dict[str, Any]]:
        """
        Get history of all approval decisions.

        Returns:
            List[Dict[str, Any]]: Approval records with result, approved flag, and feedback

        Example:
            >>> agent = HumanApprovalAgent()
            >>> # ... make some decisions ...
            >>> history = agent.get_approval_history()
            >>> for record in history:
            ...     print(f"Approved: {record['approved']}")
            ...     print(f"Feedback: {record['feedback']}")
        """
        return self.strategy.get_approval_history()


# Convenience function for quick approval workflows
async def approve_decision(
    prompt: str,
    approval_callback: Optional[Callable[[Dict[str, Any]], Tuple[bool, str]]] = None,
    llm_provider: str = "openai",
    model: str = "gpt-4",
) -> Dict[str, Any]:
    """
    Quick decision approval with default configuration.

    Args:
        prompt: Decision prompt
        approval_callback: Optional custom approval callback
        llm_provider: LLM provider to use
        model: Model to use

    Returns:
        Dict with decision and approval metadata

    Example:
        >>> import asyncio
        >>> from kaizen.agents.specialized.human_approval import approve_decision
        >>>
        >>> def my_callback(result):
        ...     return True, "Auto-approved for testing"
        >>>
        >>> result = asyncio.run(approve_decision(
        ...     "Process payment $500",
        ...     approval_callback=my_callback
        ... ))
        >>> print(f"Approved: {result['_human_approved']}")
    """
    agent = HumanApprovalAgent(
        approval_callback=approval_callback, llm_provider=llm_provider, model=model
    )

    return await agent.run_async(prompt=prompt)
