"""
Test RAGResearchAgent - Production-Ready Library Agent

Tests zero-config initialization, progressive configuration,
vector store integration, memory support, and RAG-specific features.

Written BEFORE implementation (TDD).
"""

import os

import pytest

# ============================================================================
# TEST CLASS 1: Initialization (REQUIRED - 8 tests)
# ============================================================================


class TestRAGResearchAgentInitialization:
    """Test agent initialization patterns."""

    def test_zero_config_initialization(self):
        """Test agent works with zero configuration (most important test)."""
        from kaizen.agents.specialized.rag_research import RAGResearchAgent

        # Should work with no parameters
        agent = RAGResearchAgent()

        assert agent is not None
        assert hasattr(agent, "rag_config")
        assert hasattr(agent, "run")
        assert hasattr(agent, "vector_store")

    def test_zero_config_uses_environment_variables(self):
        """Test that zero-config reads from environment variables."""
        from kaizen.agents.specialized.rag_research import RAGResearchAgent

        # Set environment variables
        os.environ["KAIZEN_LLM_PROVIDER"] = "anthropic"
        os.environ["KAIZEN_MODEL"] = "claude-3-sonnet"
        os.environ["KAIZEN_TEMPERATURE"] = "0.5"
        os.environ["KAIZEN_MAX_TOKENS"] = "2000"

        try:
            agent = RAGResearchAgent()

            # Should use environment values
            assert agent.rag_config.llm_provider == "anthropic"
            assert agent.rag_config.model == "claude-3-sonnet"
            assert agent.rag_config.temperature == 0.5
            assert agent.rag_config.max_tokens == 2000
        finally:
            # Clean up
            del os.environ["KAIZEN_LLM_PROVIDER"]
            del os.environ["KAIZEN_MODEL"]
            del os.environ["KAIZEN_TEMPERATURE"]
            del os.environ["KAIZEN_MAX_TOKENS"]

    def test_progressive_configuration_model_only(self):
        """Test progressive configuration - override model only."""
        from kaizen.agents.specialized.rag_research import RAGResearchAgent

        agent = RAGResearchAgent(model="gpt-3.5-turbo")

        assert agent.rag_config.model == "gpt-3.5-turbo"
        # Other values should be defaults
        assert agent.rag_config.llm_provider == "openai"  # default

    def test_progressive_configuration_multiple_params(self):
        """Test progressive configuration - override multiple parameters."""
        from kaizen.agents.specialized.rag_research import RAGResearchAgent

        agent = RAGResearchAgent(
            llm_provider="anthropic",
            model="claude-3-opus",
            temperature=0.7,
            max_tokens=2000,
            top_k_documents=5,
            similarity_threshold=0.5,
        )

        assert agent.rag_config.llm_provider == "anthropic"
        assert agent.rag_config.model == "claude-3-opus"
        assert agent.rag_config.temperature == 0.7
        assert agent.rag_config.max_tokens == 2000
        assert agent.rag_config.top_k_documents == 5
        assert agent.rag_config.similarity_threshold == 0.5

    def test_full_config_object_initialization(self):
        """Test initialization with full config object."""
        from kaizen.agents.specialized.rag_research import RAGConfig, RAGResearchAgent

        config = RAGConfig(
            llm_provider="openai",
            model="gpt-4-turbo",
            temperature=0.2,
            max_tokens=1800,
            timeout=60,
            top_k_documents=5,
            similarity_threshold=0.4,
            embedding_model="all-mpnet-base-v2",
        )

        agent = RAGResearchAgent(config=config)

        assert agent.rag_config.llm_provider == "openai"
        assert agent.rag_config.model == "gpt-4-turbo"
        assert agent.rag_config.temperature == 0.2
        assert agent.rag_config.max_tokens == 1800
        assert agent.rag_config.timeout == 60
        assert agent.rag_config.top_k_documents == 5
        assert agent.rag_config.similarity_threshold == 0.4
        assert agent.rag_config.embedding_model == "all-mpnet-base-v2"

    def test_config_parameter_overrides_defaults(self):
        """Test that constructor parameters override config defaults."""
        from kaizen.agents.specialized.rag_research import RAGResearchAgent

        # Parameter should override default
        # Note: Use valid embedding model name to avoid HuggingFace errors
        agent = RAGResearchAgent(
            top_k_documents=7,
            similarity_threshold=0.6,
            embedding_model="all-mpnet-base-v2",
        )

        assert agent.rag_config.top_k_documents == 7
        assert agent.rag_config.similarity_threshold == 0.6
        assert agent.rag_config.embedding_model == "all-mpnet-base-v2"

    def test_default_configuration_values(self):
        """Test that defaults are set correctly."""
        from kaizen.agents.specialized.rag_research import RAGResearchAgent

        agent = RAGResearchAgent()

        # LLM defaults
        assert agent.rag_config.llm_provider == "openai"
        assert agent.rag_config.model == "gpt-3.5-turbo"
        assert isinstance(agent.rag_config.temperature, float)
        assert isinstance(agent.rag_config.max_tokens, int)

        # RAG-specific defaults
        assert agent.rag_config.top_k_documents == 3
        assert agent.rag_config.similarity_threshold == 0.3
        assert agent.rag_config.embedding_model == "all-MiniLM-L6-v2"

        # Technical defaults
        assert agent.rag_config.timeout == 30
        assert agent.rag_config.retry_attempts == 3
        assert isinstance(agent.rag_config.provider_config, dict)

    def test_timeout_merged_into_provider_config(self):
        """Test that timeout is merged into provider_config."""
        from kaizen.agents.specialized.rag_research import RAGResearchAgent

        agent = RAGResearchAgent(timeout=60)

        # Timeout should be in provider_config
        assert "timeout" in agent.rag_config.provider_config
        assert agent.rag_config.provider_config["timeout"] == 60


# ============================================================================
# TEST CLASS 2: Execution (REQUIRED - 12 tests)
# ============================================================================


class TestRAGResearchAgentExecution:
    """Test agent execution and run method."""

    def test_run_method_exists(self):
        """Test that research convenience method exists."""
        from kaizen.agents.specialized.rag_research import RAGResearchAgent

        agent = RAGResearchAgent()

        assert hasattr(agent, "run")
        assert callable(getattr(agent, "research"))

    def test_run_returns_dict(self):
        """Test that run method returns a dictionary."""
        from kaizen.agents.specialized.rag_research import RAGResearchAgent

        agent = RAGResearchAgent()
        result = agent.run(query="What is machine learning?")

        assert isinstance(result, dict)

    def test_run_has_expected_output_fields(self):
        """Test that output contains expected signature fields."""
        from kaizen.agents.specialized.rag_research import RAGResearchAgent

        agent = RAGResearchAgent(llm_provider="mock")
        result = agent.run(query="What is deep learning?")

        # Check for signature output fields
        assert "answer" in result
        assert "sources" in result
        assert "confidence" in result
        assert "relevant_excerpts" in result

    def test_run_accepts_query_parameter(self):
        """Test that run method accepts query parameter."""
        from kaizen.agents.specialized.rag_research import RAGResearchAgent

        agent = RAGResearchAgent()

        # Should accept query parameter
        result = agent.run(query="What is NLP?")

        assert result is not None
        assert isinstance(result, dict)

    def test_run_method_integration(self):
        """Test that agent.run() method works (inherited from BaseAgent)."""
        from kaizen.agents.specialized.rag_research import RAGResearchAgent

        agent = RAGResearchAgent()

        # Direct call to BaseAgent.run()
        # Note: run() expects the enhanced query, not the raw query
        result = agent.run(
            query="Based on the following documents, answer the query.\n\nQuery: What is AI?"
        )

        assert isinstance(result, dict)

    def test_execution_with_different_queries(self):
        """Test execution with various query types."""
        from kaizen.agents.specialized.rag_research import RAGResearchAgent

        agent = RAGResearchAgent(llm_provider="mock")

        # Test with different queries
        test_cases = [
            "What is machine learning?",
            "Explain deep learning fundamentals",
            "How does NLP work?",
            "What are computer vision applications?",
        ]

        for query in test_cases:
            result = agent.research(query)
            assert isinstance(result, dict)
            assert "answer" in result
            assert "sources" in result

    def test_execution_performance(self):
        """Test that execution completes in reasonable time."""
        import time

        from kaizen.agents.specialized.rag_research import RAGResearchAgent

        agent = RAGResearchAgent()

        start = time.time()
        result = agent.run(query="What is AI?")
        duration = time.time() - start

        # Should complete in less than 30 seconds
        assert duration < 30
        assert result is not None

    def test_vector_retrieval_works(self):
        """Test that vector retrieval returns documents."""
        from kaizen.agents.specialized.rag_research import RAGResearchAgent

        agent = RAGResearchAgent()
        result = agent.run(query="machine learning")

        # Should have sources from retrieval
        assert "sources" in result
        assert isinstance(result["sources"], list)

    def test_source_attribution_works(self):
        """Test that sources are properly attributed."""
        from kaizen.agents.specialized.rag_research import RAGResearchAgent

        agent = RAGResearchAgent()
        result = agent.run(query="deep learning")

        # Should have sources
        assert "sources" in result
        assert len(result["sources"]) >= 0  # May be 0 if no relevant docs

        # Should have excerpts
        assert "relevant_excerpts" in result
        assert isinstance(result["relevant_excerpts"], list)

    def test_confidence_scoring_works(self):
        """Test that confidence scores are provided."""
        from kaizen.agents.specialized.rag_research import RAGResearchAgent

        agent = RAGResearchAgent(llm_provider="mock")
        result = agent.run(query="neural networks")

        # Should have confidence score
        assert "confidence" in result
        # Confidence should be a number between 0 and 1
        if result["confidence"] is not None:
            assert isinstance(result["confidence"], (int, float))
            assert 0 <= result["confidence"] <= 1

    def test_retrieval_quality_metric(self):
        """Test that retrieval quality is tracked."""
        from kaizen.agents.specialized.rag_research import RAGResearchAgent

        agent = RAGResearchAgent()
        result = agent.run(query="computer vision")

        # Should have retrieval quality metric if documents were retrieved
        if result.get("sources"):
            assert "retrieval_quality" in result
            assert isinstance(result["retrieval_quality"], (int, float))

    def test_session_id_support(self):
        """Test that session_id parameter is accepted."""
        from kaizen.agents.specialized.rag_research import RAGResearchAgent

        agent = RAGResearchAgent()

        # Should accept session_id for memory tracking
        result = agent.run(query="test query", session_id="test-session-123")

        assert isinstance(result, dict)


# ============================================================================
# TEST CLASS 3: Configuration (REQUIRED - 8 tests)
# ============================================================================


class TestRAGResearchAgentConfiguration:
    """Test configuration class and behavior."""

    def test_config_class_exists(self):
        """Test that configuration class exists."""
        from kaizen.agents.specialized.rag_research import RAGConfig

        assert RAGConfig is not None

    def test_config_is_dataclass(self):
        """Test that config uses dataclass decorator."""
        import dataclasses

        from kaizen.agents.specialized.rag_research import RAGConfig

        assert dataclasses.is_dataclass(RAGConfig)

    def test_config_has_required_llm_fields(self):
        """Test that config has required LLM fields."""
        from kaizen.agents.specialized.rag_research import RAGConfig

        config = RAGConfig()

        assert hasattr(config, "llm_provider")
        assert hasattr(config, "model")
        assert hasattr(config, "temperature")
        assert hasattr(config, "max_tokens")

    def test_config_has_required_technical_fields(self):
        """Test that config has required technical fields."""
        from kaizen.agents.specialized.rag_research import RAGConfig

        config = RAGConfig()

        assert hasattr(config, "timeout")
        assert hasattr(config, "retry_attempts")
        assert hasattr(config, "provider_config")

    def test_config_has_rag_specific_fields(self):
        """Test that config has RAG-specific fields."""
        from kaizen.agents.specialized.rag_research import RAGConfig

        config = RAGConfig()

        assert hasattr(config, "top_k_documents")
        assert hasattr(config, "similarity_threshold")
        assert hasattr(config, "embedding_model")

    def test_config_environment_variable_defaults(self):
        """Test that config reads from environment variables."""
        from kaizen.agents.specialized.rag_research import RAGConfig

        os.environ["KAIZEN_MODEL"] = "test-model"
        os.environ["KAIZEN_TOP_K"] = "5"

        try:
            config = RAGConfig()
            assert config.model == "test-model"
            assert config.top_k_documents == 5
        finally:
            del os.environ["KAIZEN_MODEL"]
            del os.environ["KAIZEN_TOP_K"]

    def test_config_can_be_instantiated_with_custom_values(self):
        """Test that config accepts custom values."""
        from kaizen.agents.specialized.rag_research import RAGConfig

        config = RAGConfig(
            llm_provider="custom_provider",
            model="custom_model",
            temperature=0.123,
            max_tokens=999,
            top_k_documents=7,
            similarity_threshold=0.6,
            embedding_model="all-mpnet-base-v2",  # Use valid model
        )

        assert config.llm_provider == "custom_provider"
        assert config.model == "custom_model"
        assert config.temperature == 0.123
        assert config.max_tokens == 999
        assert config.top_k_documents == 7
        assert config.similarity_threshold == 0.6
        assert config.embedding_model == "all-mpnet-base-v2"

    def test_config_provider_config_is_dict(self):
        """Test that provider_config is initialized as dict."""
        from kaizen.agents.specialized.rag_research import RAGConfig

        config = RAGConfig()

        assert isinstance(config.provider_config, dict)


# ============================================================================
# TEST CLASS 4: Signature (REQUIRED - 5 tests)
# ============================================================================


class TestRAGResearchAgentSignature:
    """Test signature definition and structure."""

    def test_signature_class_exists(self):
        """Test that signature class exists."""
        from kaizen.agents.specialized.rag_research import RAGSignature

        assert RAGSignature is not None

    def test_signature_inherits_from_base(self):
        """Test that signature inherits from Signature base class."""
        from kaizen.agents.specialized.rag_research import RAGSignature
        from kaizen.signatures import Signature

        assert issubclass(RAGSignature, Signature)

    def test_signature_has_input_fields(self):
        """Test that signature has defined input fields."""
        from kaizen.agents.specialized.rag_research import RAGSignature

        sig = RAGSignature()

        # Check for input field
        assert hasattr(sig, "query")

    def test_signature_has_output_fields(self):
        """Test that signature has defined output fields."""
        from kaizen.agents.specialized.rag_research import RAGSignature

        sig = RAGSignature()

        # Check for output fields
        assert hasattr(sig, "answer")
        assert hasattr(sig, "sources")
        assert hasattr(sig, "confidence")
        assert hasattr(sig, "relevant_excerpts")

    def test_signature_has_docstring(self):
        """Test that signature has comprehensive docstring."""
        from kaizen.agents.specialized.rag_research import RAGSignature

        assert RAGSignature.__doc__ is not None
        assert len(RAGSignature.__doc__) > 20


# ============================================================================
# TEST CLASS 5: Error Handling (REQUIRED - 5 tests)
# ============================================================================


class TestRAGResearchAgentErrorHandling:
    """Test error handling and edge cases."""

    def test_empty_input_handling(self):
        """Test handling of empty query."""
        from kaizen.agents.specialized.rag_research import RAGResearchAgent

        agent = RAGResearchAgent()

        # Should handle empty input gracefully
        result = agent.run(query="")

        assert isinstance(result, dict)
        # Should have error indicator
        assert "error" in result
        assert result["error"] == "INVALID_INPUT"

    def test_whitespace_only_input_handling(self):
        """Test handling of whitespace-only query."""
        from kaizen.agents.specialized.rag_research import RAGResearchAgent

        agent = RAGResearchAgent()

        # Should handle whitespace input gracefully
        result = agent.run(query="   \t\n   ")

        assert isinstance(result, dict)
        assert "error" in result
        assert result["error"] == "INVALID_INPUT"

    def test_none_input_handling(self):
        """Test handling of None input."""
        from kaizen.agents.specialized.rag_research import RAGResearchAgent

        agent = RAGResearchAgent()

        # Should handle None input gracefully
        try:
            result = agent.research(None)
            # If it doesn't raise, check for error
            assert isinstance(result, dict)
        except (TypeError, AttributeError):
            # Acceptable to raise error for None
            pass

    def test_no_documents_in_vector_store(self):
        """Test handling when vector store has no documents."""
        from kaizen.agents.specialized.rag_research import RAGResearchAgent
        from kaizen.retrieval.vector_store import SimpleVectorStore

        # Create empty vector store
        empty_store = SimpleVectorStore()

        agent = RAGResearchAgent(vector_store=empty_store)
        result = agent.run(query="test query")

        assert isinstance(result, dict)
        # Should handle gracefully
        assert "error" in result or result.get("sources") == []

    def test_invalid_config_handling(self):
        """Test handling of invalid configuration values."""
        from kaizen.agents.specialized.rag_research import RAGResearchAgent

        # Test with invalid similarity threshold (negative)
        try:
            agent = RAGResearchAgent(similarity_threshold=-0.5)
            # If it doesn't raise, it should handle gracefully
            assert (
                agent.rag_config.similarity_threshold >= 0
                or agent.rag_config.similarity_threshold == -0.5
            )
        except ValueError:
            # Acceptable to raise ValueError for invalid config
            pass


# ============================================================================
# TEST CLASS 6: Documentation (REQUIRED - 4 tests)
# ============================================================================


class TestRAGResearchAgentDocumentation:
    """Test docstrings and documentation completeness."""

    def test_agent_class_has_docstring(self):
        """Test that agent class has comprehensive docstring."""
        from kaizen.agents.specialized.rag_research import RAGResearchAgent

        assert RAGResearchAgent.__doc__ is not None
        assert len(RAGResearchAgent.__doc__) > 100

    def test_run_method_has_docstring(self):
        """Test that run method has docstring."""
        from kaizen.agents.specialized.rag_research import RAGResearchAgent

        assert RAGResearchAgent.run.__doc__ is not None
        assert len(RAGResearchAgent.run.__doc__) > 50

    def test_config_class_has_docstring(self):
        """Test that config class has docstring."""
        from kaizen.agents.specialized.rag_research import RAGConfig

        assert RAGConfig.__doc__ is not None

    def test_helper_methods_have_docstrings(self):
        """Test that helper methods have docstrings."""
        from kaizen.agents.specialized.rag_research import RAGResearchAgent

        # Check add_document method
        assert RAGResearchAgent.add_document.__doc__ is not None

        # Check get_document_count method
        assert RAGResearchAgent.get_document_count.__doc__ is not None

        # Check clear_documents method
        assert RAGResearchAgent.clear_documents.__doc__ is not None


# ============================================================================
# TEST CLASS 7: Type Hints (REQUIRED - 2 tests)
# ============================================================================


class TestRAGResearchAgentTypeHints:
    """Test type hint completeness."""

    def test_run_method_has_type_hints(self):
        """Test that run method has type hints."""
        import inspect

        from kaizen.agents.specialized.rag_research import RAGResearchAgent

        sig = inspect.signature(RAGResearchAgent.research)

        # Check return type hint
        assert sig.return_annotation != inspect.Parameter.empty

        # Check parameter type hints
        params_with_hints = 0
        total_params = 0

        for param_name, param in sig.parameters.items():
            if param_name != "self":
                total_params += 1
                if param.annotation != inspect.Parameter.empty:
                    params_with_hints += 1

        # At least 80% of parameters should have type hints
        if total_params > 0:
            hint_percentage = params_with_hints / total_params
            assert hint_percentage >= 0.8

    def test_init_method_has_type_hints(self):
        """Test that __init__ has type hints."""
        import inspect

        from kaizen.agents.specialized.rag_research import RAGResearchAgent

        sig = inspect.signature(RAGResearchAgent.__init__)

        # Check parameter type hints (most should have hints)
        params_with_hints = 0
        total_params = 0

        for param_name, param in sig.parameters.items():
            if param_name not in ["self", "kwargs"]:
                total_params += 1
                if param.annotation != inspect.Parameter.empty:
                    params_with_hints += 1

        # At least 80% of parameters should have type hints
        if total_params > 0:
            hint_percentage = params_with_hints / total_params
            assert hint_percentage >= 0.8


# ============================================================================
# TEST CLASS 8: Vector Store Integration (RAG-specific - 8 tests)
# ============================================================================


class TestRAGResearchAgentVectorStore:
    """Test vector store integration (RAG-specific)."""

    def test_vector_store_created_by_default(self):
        """Test that vector store is created if not provided."""
        from kaizen.agents.specialized.rag_research import RAGResearchAgent

        agent = RAGResearchAgent()

        assert agent.vector_store is not None

    def test_vector_store_can_be_provided(self):
        """Test that custom vector store can be provided."""
        from kaizen.agents.specialized.rag_research import RAGResearchAgent
        from kaizen.retrieval.vector_store import SimpleVectorStore

        custom_store = SimpleVectorStore(embedding_model="custom-model")
        agent = RAGResearchAgent(vector_store=custom_store)

        assert agent.vector_store is custom_store

    def test_vector_store_uses_correct_embedding_model(self):
        """Test that vector store uses configured embedding model."""
        from kaizen.agents.specialized.rag_research import RAGResearchAgent

        agent = RAGResearchAgent(embedding_model="all-mpnet-base-v2")

        # Vector store should use the configured embedding model
        # Note: SimpleVectorStore might not expose embedding_model publicly
        # This test verifies the config is set correctly
        assert agent.rag_config.embedding_model == "all-mpnet-base-v2"

    def test_documents_can_be_retrieved(self):
        """Test that documents can be retrieved from vector store."""
        from kaizen.agents.specialized.rag_research import RAGResearchAgent

        agent = RAGResearchAgent()

        # Should have default documents loaded
        count = agent.get_document_count()
        assert count >= 0  # May be 0 or have default documents

    def test_similarity_threshold_works(self):
        """Test that similarity threshold is applied."""
        from kaizen.agents.specialized.rag_research import RAGResearchAgent

        # High threshold should filter more aggressively
        agent_strict = RAGResearchAgent(similarity_threshold=0.9)
        result_strict = agent_strict.research("unrelated random query xyz123")

        # Should have few or no results with high threshold
        assert "sources" in result_strict

    def test_top_k_retrieval_works(self):
        """Test that top_k parameter limits results."""
        from kaizen.agents.specialized.rag_research import RAGResearchAgent

        agent = RAGResearchAgent(top_k_documents=2)

        result = agent.run(query="machine learning")

        # Should retrieve at most top_k documents
        if result.get("sources"):
            assert len(result["sources"]) <= 2

    def test_sample_documents_loaded_by_default(self):
        """Test that sample documents are loaded by default."""
        from kaizen.agents.specialized.rag_research import RAGResearchAgent

        agent = RAGResearchAgent()

        # Should have documents in vector store
        count = agent.get_document_count()
        assert count > 0  # Default documents should be loaded

    def test_add_document_method_works(self):
        """Test that documents can be added to vector store."""
        from kaizen.agents.specialized.rag_research import RAGResearchAgent

        agent = RAGResearchAgent()
        initial_count = agent.get_document_count()

        # Add a document
        agent.add_document(
            doc_id="test_doc",
            title="Test Document",
            content="This is a test document about testing.",
        )

        # Count should increase
        new_count = agent.get_document_count()
        assert new_count == initial_count + 1


# ============================================================================
# TEST CLASS 9: Memory Support (REQUIRED - 3 tests)
# ============================================================================


class TestRAGResearchAgentMemory:
    """Test memory integration."""

    def test_memory_disabled_by_default(self):
        """Test that memory is disabled by default (opt-in)."""
        from kaizen.agents.specialized.rag_research import RAGResearchAgent

        agent = RAGResearchAgent()

        # Memory should be None by default
        assert agent.memory is None

    def test_memory_enabled_with_memory_config(self):
        """Test that memory can be enabled via memory_config."""
        from kaizen.agents.specialized.rag_research import RAGResearchAgent

        memory_config = {"enabled": True, "top_k": 5, "similarity_threshold": 0.7}

        agent = RAGResearchAgent(memory_config=memory_config)

        # Memory should be initialized
        assert agent.memory is not None

    def test_vector_memory_used_when_enabled(self):
        """Test that VectorMemory is used when enabled."""
        from kaizen.agents.specialized.rag_research import RAGResearchAgent
        from kaizen.memory.vector import VectorMemory

        memory_config = {"enabled": True, "top_k": 5, "similarity_threshold": 0.7}

        agent = RAGResearchAgent(memory_config=memory_config)

        # Should use VectorMemory
        assert isinstance(agent.memory, VectorMemory)


# ============================================================================
# TEST CLASS 10: Helper Methods (RAG-specific - 3 tests)
# ============================================================================


class TestRAGResearchAgentHelperMethods:
    """Test helper methods for document management."""

    def test_get_document_count_method(self):
        """Test get_document_count returns correct count."""
        from kaizen.agents.specialized.rag_research import RAGResearchAgent

        agent = RAGResearchAgent()

        count = agent.get_document_count()
        assert isinstance(count, int)
        assert count >= 0

    def test_clear_documents_method(self):
        """Test clear_documents removes all documents."""
        from kaizen.agents.specialized.rag_research import RAGResearchAgent

        agent = RAGResearchAgent()

        # Clear documents
        agent.clear_documents()

        # Count should be 0
        count = agent.get_document_count()
        assert count == 0

    def test_add_document_with_all_parameters(self):
        """Test add_document accepts all required parameters."""
        from kaizen.agents.specialized.rag_research import RAGResearchAgent

        agent = RAGResearchAgent()

        # Should accept all parameters
        agent.add_document(
            doc_id="test123", title="Test Title", content="Test content goes here"
        )

        # Document should be added
        count = agent.get_document_count()
        assert count > 0


# ============================================================================
# TEST CLASS 11: BaseAgent Integration (REQUIRED - 2 tests)
# ============================================================================


class TestRAGResearchAgentBaseAgentIntegration:
    """Test integration with BaseAgent."""

    def test_agent_inherits_from_base_agent(self):
        """Test RAGResearchAgent inherits from BaseAgent."""
        from kaizen.agents.specialized.rag_research import RAGResearchAgent
        from kaizen.core.base_agent import BaseAgent

        agent = RAGResearchAgent()

        assert isinstance(agent, BaseAgent)

    def test_agent_uses_async_single_shot_strategy(self):
        """Test that agent uses MultiCycleStrategy by default."""
        from kaizen.agents.specialized.rag_research import RAGResearchAgent
        from kaizen.strategies.multi_cycle import MultiCycleStrategy

        agent = RAGResearchAgent(llm_provider="mock")

        # Should use MultiCycleStrategy (default for BaseAgent)
        assert isinstance(agent.strategy, MultiCycleStrategy)


if __name__ == "__main__":
    pytest.main([__file__, "-v", "--tb=short"])
