"""
Tests for Multi-Agent Shared Memory Example.

This module tests the shared-insights example which demonstrates multi-agent
collaboration via SharedMemoryPool from Phase 2.

Test Coverage:
- Agent initialization with shared memory
- ResearcherAgent writes findings to shared memory
- AnalystAgent reads and analyzes findings
- SynthesizerAgent reads all insights
- Insight writing and reading flow
- Attention filtering (tags, importance, exclude_own)
- Full collaboration workflow
- Statistics tracking

Agents Tested:
- ResearcherAgent: Conducts research, writes findings
- AnalystAgent: Reads findings, performs analysis, writes insights
- SynthesizerAgent: Reads all insights, creates synthesis

Author: Kaizen Framework Team
Created: 2025-10-02 (Phase 4, Task 4I.3)
Reference: Week 3 Phase 2 SharedMemoryPool implementation
"""

# Standardized example loading
from example_import_helper import import_example_module

# Load shared-insights example
_module = import_example_module("examples/2-multi-agent/shared-insights")
ResearcherAgent = _module.ResearcherAgent
AnalystAgent = _module.AnalystAgent
SynthesizerAgent = _module.SynthesizerAgent
research_collaboration_workflow = _module.research_collaboration_workflow

from kaizen.core.config import BaseAgentConfig
from kaizen.memory.shared_memory import SharedMemoryPool


class TestAgentInitialization:
    """Test agent initialization with shared memory."""

    def test_researcher_agent_initializes_with_shared_memory(self):
        """Test ResearcherAgent initializes with shared memory."""
        pool = SharedMemoryPool()
        config = BaseAgentConfig(llm_provider="mock", model="gpt-3.5-turbo")

        agent = ResearcherAgent(config, pool, agent_id="researcher_1")

        assert agent.shared_memory is pool
        assert agent.agent_id == "researcher_1"
        assert hasattr(agent, "run")

    def test_analyst_agent_initializes_with_shared_memory(self):
        """Test AnalystAgent initializes with shared memory."""
        pool = SharedMemoryPool()
        config = BaseAgentConfig(llm_provider="mock", model="gpt-3.5-turbo")

        agent = AnalystAgent(config, pool, agent_id="analyst_1")

        assert agent.shared_memory is pool
        assert agent.agent_id == "analyst_1"
        assert hasattr(agent, "run")

    def test_synthesizer_agent_initializes_with_shared_memory(self):
        """Test SynthesizerAgent initializes with shared memory."""
        pool = SharedMemoryPool()
        config = BaseAgentConfig(llm_provider="mock", model="gpt-3.5-turbo")

        agent = SynthesizerAgent(config, pool, agent_id="synthesizer_1")

        assert agent.shared_memory is pool
        assert agent.agent_id == "synthesizer_1"
        assert hasattr(agent, "synthesize")


class TestInsightWriting:
    """Test insight writing to shared memory."""

    def test_researcher_writes_findings_to_shared_memory(self):
        """Test ResearcherAgent writes findings to shared memory."""
        pool = SharedMemoryPool()
        config = BaseAgentConfig(llm_provider="mock", model="gpt-3.5-turbo")
        agent = ResearcherAgent(config, pool, agent_id="researcher_1")

        # Research a topic
        agent.run(query="Python")

        # Check insight was written
        insights = pool.read_all()
        assert len(insights) > 0

        # Find researcher's insight
        researcher_insights = [i for i in insights if i["agent_id"] == "researcher_1"]
        assert len(researcher_insights) > 0
        assert "research" in researcher_insights[0]["tags"]
        assert "Python" in researcher_insights[0]["tags"]

    def test_insight_has_correct_format(self):
        """Test insights have correct format with all required fields."""
        pool = SharedMemoryPool()
        config = BaseAgentConfig(llm_provider="mock", model="gpt-3.5-turbo")
        agent = ResearcherAgent(config, pool, agent_id="researcher_1")

        agent.run(query="Machine Learning")

        insights = pool.read_all()
        assert len(insights) > 0

        insight = insights[0]
        # Required fields
        assert "agent_id" in insight
        assert "content" in insight
        assert "tags" in insight
        assert "importance" in insight
        assert "segment" in insight
        assert "timestamp" in insight

        # Values
        assert insight["agent_id"] == "researcher_1"
        assert isinstance(insight["content"], str)
        assert isinstance(insight["tags"], list)
        assert 0.0 <= insight["importance"] <= 1.0

    def test_multiple_insights_accumulate_in_pool(self):
        """Test multiple insights accumulate in pool."""
        pool = SharedMemoryPool()
        config = BaseAgentConfig(llm_provider="mock", model="gpt-3.5-turbo")
        agent = ResearcherAgent(config, pool, agent_id="researcher_1")

        # Research multiple topics
        agent.run(query="Topic A")
        agent.run(query="Topic B")
        agent.run(query="Topic C")

        insights = pool.read_all()
        assert len(insights) >= 3


class TestInsightReading:
    """Test insight reading from shared memory."""

    def test_analyst_reads_researcher_insights(self):
        """Test AnalystAgent reads researcher's insights."""
        pool = SharedMemoryPool()
        config = BaseAgentConfig(llm_provider="mock", model="gpt-3.5-turbo")

        # Researcher writes
        researcher = ResearcherAgent(config, pool, agent_id="researcher_1")
        researcher.research("Python")

        # Analyst reads
        analyst = AnalystAgent(config, pool, agent_id="analyst_1")
        analyst.analyze("Python")

        # Analyst should have written analysis based on research
        insights = pool.read_all()
        analyst_insights = [i for i in insights if i["agent_id"] == "analyst_1"]
        assert len(analyst_insights) > 0
        assert "analysis" in analyst_insights[0]["tags"]

    def test_filtering_by_tags_works(self):
        """Test filtering insights by tags works correctly."""
        pool = SharedMemoryPool()
        config = BaseAgentConfig(llm_provider="mock", model="gpt-3.5-turbo")

        researcher = ResearcherAgent(config, pool, agent_id="researcher_1")
        researcher.research("Python")
        researcher.research("Java")

        # Read only Python-related insights
        python_insights = pool.read_relevant(
            agent_id="analyst_1", tags=["Python"], exclude_own=True
        )

        # Should filter correctly
        for insight in python_insights:
            assert "Python" in insight["tags"]

    def test_exclude_own_parameter_works(self):
        """Test exclude_own parameter correctly filters out own insights."""
        pool = SharedMemoryPool()
        config = BaseAgentConfig(llm_provider="mock", model="gpt-3.5-turbo")

        # Write insight
        researcher = ResearcherAgent(config, pool, agent_id="researcher_1")
        researcher.research("Python")

        # Read with exclude_own=True
        insights_excluded = pool.read_relevant(
            agent_id="researcher_1", tags=["research"], exclude_own=True
        )

        # Should be empty (own insights excluded)
        assert len(insights_excluded) == 0

        # Read with exclude_own=False
        insights_included = pool.read_relevant(
            agent_id="researcher_1", tags=["research"], exclude_own=False
        )

        # Should include own insights
        assert len(insights_included) > 0


class TestCollaborationWorkflow:
    """Test full collaboration workflow."""

    def test_full_workflow_executes(self):
        """Test full research → analysis → synthesis workflow executes."""
        result = research_collaboration_workflow("Artificial Intelligence")

        # Should have results from all phases
        assert "research" in result
        assert "analysis" in result
        assert "synthesis" in result
        assert "stats" in result

    def test_insights_flow_between_agents(self):
        """Test insights correctly flow between agents."""
        pool = SharedMemoryPool()
        config = BaseAgentConfig(llm_provider="mock", model="gpt-3.5-turbo")

        # Create agents
        researcher = ResearcherAgent(config, pool, agent_id="researcher_1")
        analyst = AnalystAgent(config, pool, agent_id="analyst_1")
        synthesizer = SynthesizerAgent(config, pool, agent_id="synthesizer_1")

        # Execute pipeline
        researcher.research("Machine Learning")
        analyst.analyze("Machine Learning")
        synthesizer.synthesize("Machine Learning")

        # Check pool has insights from all agents
        insights = pool.read_all()
        agent_ids = [i["agent_id"] for i in insights]

        assert "researcher_1" in agent_ids
        assert "analyst_1" in agent_ids

    def test_statistics_reflect_all_operations(self):
        """Test statistics correctly reflect all operations."""
        result = research_collaboration_workflow("Data Science")

        stats = result["stats"]
        assert "insight_count" in stats
        assert "agent_count" in stats
        assert "tag_distribution" in stats
        assert "segment_distribution" in stats

        # Should have multiple insights
        assert stats["insight_count"] > 0

        # Should have multiple agents
        assert stats["agent_count"] >= 2


class TestSharedMemoryFeatures:
    """Test shared memory features."""

    def test_attention_filtering_by_tags(self):
        """Test attention filtering by tags."""
        pool = SharedMemoryPool()
        config = BaseAgentConfig(llm_provider="mock", model="gpt-3.5-turbo")

        researcher = ResearcherAgent(config, pool, agent_id="researcher_1")
        researcher.research("Python")
        researcher.research("JavaScript")

        # Filter by Python tag
        python_insights = pool.read_relevant(
            agent_id="analyst_1", tags=["Python"], exclude_own=True
        )

        # All should have Python tag
        for insight in python_insights:
            assert "Python" in insight["tags"]

    def test_attention_filtering_by_importance(self):
        """Test attention filtering by importance threshold."""
        pool = SharedMemoryPool()

        # Write insights with different importance
        pool.write_insight(
            {
                "agent_id": "agent_1",
                "content": "Low importance",
                "tags": ["test"],
                "importance": 0.3,
                "segment": "test",
            }
        )
        pool.write_insight(
            {
                "agent_id": "agent_1",
                "content": "High importance",
                "tags": ["test"],
                "importance": 0.9,
                "segment": "test",
            }
        )

        # Filter by importance
        high_importance = pool.read_relevant(
            agent_id="agent_2", min_importance=0.7, exclude_own=True
        )

        assert len(high_importance) == 1
        assert high_importance[0]["importance"] >= 0.7

    def test_attention_filtering_by_segment(self):
        """Test attention filtering by segment."""
        pool = SharedMemoryPool()

        # Write insights with different segments
        pool.write_insight(
            {
                "agent_id": "agent_1",
                "content": "Research finding",
                "tags": ["test"],
                "importance": 0.8,
                "segment": "findings",
            }
        )
        pool.write_insight(
            {
                "agent_id": "agent_1",
                "content": "Analysis result",
                "tags": ["test"],
                "importance": 0.8,
                "segment": "analysis",
            }
        )

        # Filter by segment
        analysis_insights = pool.read_relevant(
            agent_id="agent_2", segments=["analysis"], exclude_own=True
        )

        assert len(analysis_insights) == 1
        assert analysis_insights[0]["segment"] == "analysis"


class TestSessionIsolation:
    """Test session isolation between different topics."""

    def test_different_topics_do_not_interfere(self):
        """Test different topics maintain separate insight contexts."""
        pool = SharedMemoryPool()
        config = BaseAgentConfig(llm_provider="mock", model="gpt-3.5-turbo")

        researcher = ResearcherAgent(config, pool, agent_id="researcher_1")

        # Research different topics
        researcher.research("Python")
        researcher.research("Java")

        # Read Python-only insights
        python_insights = pool.read_relevant(
            agent_id="analyst_1", tags=["Python"], exclude_own=True
        )

        # Read Java-only insights
        java_insights = pool.read_relevant(
            agent_id="analyst_1", tags=["Java"], exclude_own=True
        )

        # Should be separate
        python_tags = [i["tags"] for i in python_insights]
        java_tags = [i["tags"] for i in java_insights]

        # Verify isolation
        for tags in python_tags:
            assert "Python" in tags

        for tags in java_tags:
            assert "Java" in tags


class TestThreadSafety:
    """Test thread safety of concurrent agent execution."""

    def test_concurrent_agent_execution(self):
        """Test concurrent agent execution is thread-safe."""
        import threading

        pool = SharedMemoryPool()
        config = BaseAgentConfig(llm_provider="mock", model="gpt-3.5-turbo")

        def research_task(agent_id, topic):
            agent = ResearcherAgent(config, pool, agent_id=agent_id)
            agent.research(topic)

        # Create threads
        threads = []
        for i in range(5):
            thread = threading.Thread(
                target=research_task, args=(f"researcher_{i}", f"Topic_{i}")
            )
            threads.append(thread)

        # Start all threads
        for thread in threads:
            thread.start()

        # Wait for completion
        for thread in threads:
            thread.join()

        # Should have 5 insights (thread-safe)
        insights = pool.read_all()
        assert len(insights) == 5

        # All agents should be unique
        agent_ids = [i["agent_id"] for i in insights]
        assert len(set(agent_ids)) == 5
