"""
Tests for Supervisor-Worker Multi-Agent Pattern.

This module tests the supervisor-worker example which demonstrates centralized
coordination with task delegation using SharedMemoryPool.

Test Coverage:
- Supervisor task breakdown and delegation
- Worker task execution
- Result aggregation
- Parallel execution
- Error handling
- Task queue management
- Worker availability
- Load balancing

Pattern:
SupervisorAgent receives request → breaks into tasks → delegates to workers
→ workers execute tasks → workers write results to SharedMemoryPool
→ supervisor aggregates results → return final result

Agents Tested:
- SupervisorAgent: Breaks requests into tasks, delegates, aggregates
- WorkerAgent: Executes assigned tasks independently
- CoordinatorAgent: Monitors progress, handles conflicts

Author: Kaizen Framework Team
Created: 2025-10-02 (Phase 5, Task 5E.1)
Reference: Phase 4 shared-insights example
"""

# Standardized example loading
from example_import_helper import import_example_module

# Load supervisor-worker example
_module = import_example_module("examples/2-multi-agent/supervisor-worker")
SupervisorAgent = _module.SupervisorAgent
WorkerAgent = _module.WorkerAgent
CoordinatorAgent = _module.CoordinatorAgent
supervisor_worker_workflow = _module.supervisor_worker_workflow

from kaizen.core.config import BaseAgentConfig
from kaizen.memory.shared_memory import SharedMemoryPool


class TestSupervisorTaskDelegation:
    """Test supervisor task breakdown and delegation."""

    def test_supervisor_breaks_request_into_tasks(self):
        """Test supervisor breaks request into multiple tasks."""
        pool = SharedMemoryPool()
        config = BaseAgentConfig(llm_provider="mock", model="gpt-3.5-turbo")

        supervisor = SupervisorAgent(config, pool, agent_id="supervisor_1")

        # Delegate request
        tasks = supervisor.delegate("Process 3 documents")

        # Should create multiple tasks
        assert isinstance(tasks, list)
        assert len(tasks) > 0

        # Each task should have required fields
        for task in tasks:
            assert "task_id" in task
            assert "description" in task
            assert "assigned_to" in task

    def test_supervisor_writes_tasks_to_shared_memory(self):
        """Test supervisor writes tasks to shared memory for workers."""
        pool = SharedMemoryPool()
        config = BaseAgentConfig(llm_provider="mock", model="gpt-3.5-turbo")

        supervisor = SupervisorAgent(config, pool, agent_id="supervisor_1")

        # Delegate request
        supervisor.delegate("Process documents")

        # Verify tasks written to shared memory
        tasks_in_pool = pool.read_relevant(
            agent_id="worker_1", tags=["task", "pending"], exclude_own=False
        )

        assert len(tasks_in_pool) > 0

        # Tasks should have correct tags and segment
        for insight in tasks_in_pool:
            assert "task" in insight["tags"]
            assert "pending" in insight["tags"]
            assert insight["segment"] == "tasks"

    def test_supervisor_assigns_tasks_to_available_workers(self):
        """Test supervisor assigns tasks to available workers."""
        pool = SharedMemoryPool()
        config = BaseAgentConfig(llm_provider="mock", model="gpt-3.5-turbo")

        supervisor = SupervisorAgent(config, pool, agent_id="supervisor_1")

        # Register workers
        WorkerAgent(config, pool, agent_id="worker_1")
        WorkerAgent(config, pool, agent_id="worker_2")

        # Delegate request
        tasks = supervisor.delegate(
            "Process 4 documents", available_workers=["worker_1", "worker_2"]
        )

        # Tasks should be assigned to available workers
        assigned_workers = [task["assigned_to"] for task in tasks]
        for worker_id in assigned_workers:
            assert worker_id in ["worker_1", "worker_2"]


class TestWorkerTaskExecution:
    """Test worker task execution."""

    def test_worker_reads_assigned_tasks_from_shared_memory(self):
        """Test worker reads tasks assigned to them."""
        pool = SharedMemoryPool()
        config = BaseAgentConfig(llm_provider="mock", model="gpt-3.5-turbo")

        # Supervisor creates task
        supervisor = SupervisorAgent(config, pool, agent_id="supervisor_1")
        supervisor.delegate("Process document", available_workers=["worker_1"])

        # Worker reads tasks
        worker = WorkerAgent(config, pool, agent_id="worker_1")
        tasks = worker.get_assigned_tasks()

        # Should find assigned tasks
        assert len(tasks) > 0

    def test_worker_executes_task_and_writes_result(self):
        """Test worker executes task and writes result to shared memory."""
        pool = SharedMemoryPool()
        config = BaseAgentConfig(llm_provider="mock", model="gpt-3.5-turbo")

        worker = WorkerAgent(config, pool, agent_id="worker_1")

        # Execute task
        task = {
            "task_id": "task_1",
            "description": "Process document A",
            "assigned_to": "worker_1",
        }
        result = worker.execute_task(task)

        # Result should be returned
        assert "task_id" in result
        assert "result" in result
        assert result["task_id"] == "task_1"

        # Result should be written to shared memory
        results_in_pool = pool.read_relevant(
            agent_id="supervisor_1", tags=["result", "completed"], exclude_own=False
        )

        assert len(results_in_pool) > 0

    def test_worker_marks_task_as_completed(self):
        """Test worker marks task as completed in shared memory."""
        pool = SharedMemoryPool()
        config = BaseAgentConfig(llm_provider="mock", model="gpt-3.5-turbo")

        worker = WorkerAgent(config, pool, agent_id="worker_1")

        # Execute task
        task = {
            "task_id": "task_1",
            "description": "Process document A",
            "assigned_to": "worker_1",
        }
        worker.execute_task(task)

        # Check task marked as completed
        completed_tasks = pool.read_relevant(
            agent_id="supervisor_1", tags=["completed"], exclude_own=False
        )

        assert len(completed_tasks) > 0
        # Find the specific task
        task_results = [
            t
            for t in completed_tasks
            if t.get("metadata", {}).get("task_id") == "task_1"
        ]
        assert len(task_results) > 0

    def test_multiple_workers_execute_tasks_independently(self):
        """Test multiple workers execute tasks independently."""
        pool = SharedMemoryPool()
        config = BaseAgentConfig(llm_provider="mock", model="gpt-3.5-turbo")

        # Create workers
        worker1 = WorkerAgent(config, pool, agent_id="worker_1")
        worker2 = WorkerAgent(config, pool, agent_id="worker_2")

        # Execute tasks
        task1 = {
            "task_id": "task_1",
            "description": "Task 1",
            "assigned_to": "worker_1",
        }
        task2 = {
            "task_id": "task_2",
            "description": "Task 2",
            "assigned_to": "worker_2",
        }

        result1 = worker1.execute_task(task1)
        result2 = worker2.execute_task(task2)

        # Both should complete
        assert result1["task_id"] == "task_1"
        assert result2["task_id"] == "task_2"

        # Both should write to shared memory
        all_results = pool.read_all()
        worker_ids = [r["agent_id"] for r in all_results]
        assert "worker_1" in worker_ids
        assert "worker_2" in worker_ids


class TestResultAggregation:
    """Test supervisor result aggregation."""

    def test_supervisor_aggregates_worker_results(self):
        """Test supervisor aggregates results from workers."""
        pool = SharedMemoryPool()
        config = BaseAgentConfig(llm_provider="mock", model="gpt-3.5-turbo")

        supervisor = SupervisorAgent(config, pool, agent_id="supervisor_1")

        # Simulate worker results
        pool.write_insight(
            {
                "agent_id": "worker_1",
                "content": "Result 1",
                "tags": ["result", "completed", "request_1"],
                "importance": 0.8,
                "segment": "results",
                "metadata": {"task_id": "task_1"},
            }
        )
        pool.write_insight(
            {
                "agent_id": "worker_2",
                "content": "Result 2",
                "tags": ["result", "completed", "request_1"],
                "importance": 0.8,
                "segment": "results",
                "metadata": {"task_id": "task_2"},
            }
        )

        # Aggregate results
        aggregated = supervisor.aggregate_results("request_1")

        # Should combine all results
        assert "final_result" in aggregated
        assert "task_results" in aggregated
        assert len(aggregated["task_results"]) == 2

    def test_supervisor_waits_for_all_tasks_to_complete(self):
        """Test supervisor waits for all tasks before aggregating."""
        pool = SharedMemoryPool()
        config = BaseAgentConfig(llm_provider="mock", model="gpt-3.5-turbo")

        supervisor = SupervisorAgent(config, pool, agent_id="supervisor_1")

        # Create tasks
        tasks = supervisor.delegate(
            "Process 2 documents", available_workers=["worker_1"]
        )

        # Simulate only one result
        pool.write_insight(
            {
                "agent_id": "worker_1",
                "content": "Result 1",
                "tags": ["result", "completed"],
                "importance": 0.8,
                "segment": "results",
                "metadata": {"task_id": tasks[0]["task_id"]},
            }
        )

        # Check completion status
        is_complete = supervisor.check_all_tasks_completed(
            tasks[0].get("request_id", "request_1")
        )

        # Should not be complete (only 1 of 2 tasks done)
        assert is_complete == False


class TestCoordinatorAgent:
    """Test coordinator monitoring and conflict resolution."""

    def test_coordinator_monitors_worker_progress(self):
        """Test coordinator monitors worker progress."""
        pool = SharedMemoryPool()
        config = BaseAgentConfig(llm_provider="mock", model="gpt-3.5-turbo")

        coordinator = CoordinatorAgent(config, pool, agent_id="coordinator_1")

        # Simulate worker activity
        pool.write_insight(
            {
                "agent_id": "worker_1",
                "content": "Working on task 1",
                "tags": ["progress", "task_1"],
                "importance": 0.5,
                "segment": "progress",
            }
        )

        # Monitor progress
        progress = coordinator.monitor_progress()

        # Should return progress information
        assert "active_workers" in progress
        assert "pending_tasks" in progress
        assert "completed_tasks" in progress

    def test_coordinator_handles_worker_conflicts(self):
        """Test coordinator handles conflicts between workers."""
        pool = SharedMemoryPool()
        config = BaseAgentConfig(llm_provider="mock", model="gpt-3.5-turbo")

        coordinator = CoordinatorAgent(config, pool, agent_id="coordinator_1")

        # Simulate conflict: two workers assigned same task
        pool.write_insight(
            {
                "agent_id": "worker_1",
                "content": "Processing task_1",
                "tags": ["conflict", "task_1"],
                "importance": 0.9,
                "segment": "conflicts",
                "metadata": {"task_id": "task_1"},
            }
        )
        pool.write_insight(
            {
                "agent_id": "worker_2",
                "content": "Processing task_1",
                "tags": ["conflict", "task_1"],
                "importance": 0.9,
                "segment": "conflicts",
                "metadata": {"task_id": "task_1"},
            }
        )

        # Resolve conflict
        resolution = coordinator.resolve_conflicts()

        # Should detect and resolve conflicts
        assert "conflicts_found" in resolution
        assert "resolutions" in resolution


class TestParallelExecution:
    """Test parallel task execution."""

    def test_workers_execute_tasks_in_parallel(self):
        """Test workers can execute tasks in parallel."""
        import threading

        pool = SharedMemoryPool()
        config = BaseAgentConfig(llm_provider="mock", model="gpt-3.5-turbo")

        def worker_task(worker_id, task_id):
            worker = WorkerAgent(config, pool, agent_id=worker_id)
            task = {
                "task_id": task_id,
                "description": f"Task {task_id}",
                "assigned_to": worker_id,
            }
            worker.execute_task(task)

        # Create threads
        threads = []
        for i in range(3):
            thread = threading.Thread(
                target=worker_task, args=(f"worker_{i}", f"task_{i}")
            )
            threads.append(thread)

        # Start all threads
        for thread in threads:
            thread.start()

        # Wait for completion
        for thread in threads:
            thread.join()

        # Should have 3 results
        results = pool.read_all()
        assert len(results) == 3


class TestErrorHandling:
    """Test error handling in supervisor-worker pattern."""

    def test_supervisor_handles_worker_failure(self):
        """Test supervisor handles worker failure gracefully."""
        pool = SharedMemoryPool()
        config = BaseAgentConfig(llm_provider="mock", model="gpt-3.5-turbo")

        supervisor = SupervisorAgent(config, pool, agent_id="supervisor_1")

        # Simulate worker failure
        pool.write_insight(
            {
                "agent_id": "worker_1",
                "content": "Task failed",
                "tags": ["error", "failed"],
                "importance": 0.9,
                "segment": "errors",
                "metadata": {"task_id": "task_1", "error": "Processing error"},
            }
        )

        # Check for failures
        failures = supervisor.check_failures("request_1")

        # Should detect failure
        assert len(failures) > 0

    def test_supervisor_reassigns_failed_tasks(self):
        """Test supervisor can reassign failed tasks to other workers."""
        pool = SharedMemoryPool()
        config = BaseAgentConfig(llm_provider="mock", model="gpt-3.5-turbo")

        supervisor = SupervisorAgent(config, pool, agent_id="supervisor_1")

        # Failed task
        failed_task = {
            "task_id": "task_1",
            "description": "Failed task",
            "assigned_to": "worker_1",
        }

        # Reassign to different worker
        reassigned = supervisor.reassign_task(failed_task, new_worker="worker_2")

        # Should reassign successfully
        assert reassigned["assigned_to"] == "worker_2"


class TestFullWorkflow:
    """Test full supervisor-worker workflow."""

    def test_full_workflow_executes(self):
        """Test full workflow from request to final result."""
        result = supervisor_worker_workflow("Process 3 documents")

        # Should have complete result
        assert "request" in result
        assert "tasks" in result
        assert "results" in result
        assert "final_result" in result
        assert "stats" in result

    def test_workflow_with_multiple_workers(self):
        """Test workflow with multiple workers."""
        result = supervisor_worker_workflow("Process 5 documents", num_workers=3)

        # Should distribute work across workers
        assert "tasks" in result
        assert len(result["tasks"]) > 0

        # Stats should show multiple workers
        stats = result["stats"]
        assert stats["agent_count"] >= 3  # supervisor + workers

    def test_workflow_statistics_accurate(self):
        """Test workflow statistics are accurate."""
        result = supervisor_worker_workflow("Process documents")

        stats = result["stats"]

        # Should have accurate counts
        assert "insight_count" in stats
        assert "agent_count" in stats
        assert stats["insight_count"] > 0
        assert stats["agent_count"] > 0
