"""Constants for the Sound Shell Model module"""

import typing as tp

import numpy as np

from pttools import bubble

#: Default number of xi points used in bubble profiles
NXIDEFAULT: int = 2000
#: Default number of T-tilde values for bubble lifetime distribution integration
NTDEFAULT: int = 10000
#: Default number of wavevectors used in the velocity convolution integrations.
# This should be at least as large as the default number of GW frequencies.
N_Z_LOOKUP_DEFAULT: int = 10000
NptType = np.ndarray[tuple[int], np.int_] | tuple[int, int, int]
NPTDEFAULT: NptType = (NXIDEFAULT, NTDEFAULT, N_Z_LOOKUP_DEFAULT)
Y_DEFAULT: np.ndarray[tuple[int], np.float64] = np.logspace(-1, 3, 1000)  # type: ignore

# It seems that NPTDEFAULT should be something like NXIDEFAULT/(2.pi), otherwise one
# gets a GW power spectrum which drifts up at high k.
#
# The maximum trustworthy k is approx NXIDEFAULT/(2.pi)
#
# NTDEFAULT can be left as it is, or even reduced to 100

#: Default dimensionless wavenumber above which to use approximation for sin_transform, sin_transform_approx.
# TODO: check that this can actually be a float
Z_ST_THRESH: float = 50

#: Default wavenumber overlap for matching sin_transform_approx
DZ_ST_BLEND: float = np.pi

#: Maximum in bubble lifetime distribution integration
T_TILDE_MAX: float = 20.0
#: Minimum in bubble lifetime distribution integration
T_TILDE_MIN: float = 0.01

#: Default nucleation parameters
DEFAULT_NUC_PARM: tuple[int] = (1,)

#: Default sound speed
CS0: tp.Final[np.float64] = bubble.CS0

#: Default mean adiabatic index
GAMMA: float = 4/3

#: Gravitational constant in GeV
G: float = 1.22e19**(-2)
