"""Data types and operations from UTM API (USS->DSS and USS->USS) 1.0.0 OpenAPI"""

# This file is autogenerated; do not modify manually!

from __future__ import annotations

from enum import Enum

from implicitdict import ImplicitDict, StringBasedDateTime

from uas_standards.utils import Operation

API_VERSION = "1.0.0"
"""Version of UTM API (USS->DSS and USS->USS) OpenAPI specification from which the objects in this package were generated."""

UUIDv4Format = str
"""String whose format matches a version-4 UUID according to RFC 4122."""


UUIDv7Format = str
"""String whose format matches a version-7 UUID according to RFC 9562."""


EntityID = UUIDv4Format


EntityOVN = str
"""A token associated with a particular UTM Entity+version created by the DSS upon creation or modification of an Entity reference and provided to the client creating or modifying the Entity reference.  The EntityOVN is stored privately by the DSS and then compared against entries in a Key provided to mutate the airspace.  The EntityOVN is also provided by the client whenever that client transmits the full information of the Entity (either via GET, or via a subscription notification)."""


EntityVersion = int
"""Numeric version of this entity which increments upon each change in the entity, regardless of whether any field of the entity changes.  A USS with the details of this entity when it was at a particular version does not need to retrieve the details again until the version changes."""


SubscriptionID = UUIDv4Format


Key = list[EntityOVN]


class TimeFormat(str, Enum):
    RFC3339 = "RFC3339"


class Time(ImplicitDict):
    value: StringBasedDateTime
    """RFC3339-formatted time/date string.  The time zone must be 'Z'."""

    format: TimeFormat = TimeFormat.RFC3339


class RadiusUnits(str, Enum):
    """FIXM-compatible units.  Only meters ("M") are acceptable for UTM."""

    M = "M"


class Radius(ImplicitDict):
    value: float
    """Distance from the centerpoint of a circular area, along the WGS84 ellipsoid."""

    units: RadiusUnits = RadiusUnits.M
    """FIXM-compatible units.  Only meters ("M") are acceptable for UTM."""


class AltitudeReference(str, Enum):
    """A code indicating the reference for a vertical distance. See AIXM 5.1 and FIXM 4.2.0. Currently, UTM only allows WGS84 with no immediate plans to allow other options. FIXM and AIXM allow for 'SFC' which is equivalent to AGL."""

    W84 = "W84"


class AltitudeUnits(str, Enum):
    """The reference quantities used to express the value of altitude. See FIXM 4.2. Currently, UTM only allows meters with no immediate plans to allow other options."""

    M = "M"


class Altitude(ImplicitDict):
    value: float
    """The numeric value of the altitude. Note that min and max values are added as a sanity check. As use cases evolve and more options are made available in terms of units of measure or reference systems, these bounds may be re-evaluated."""

    reference: AltitudeReference = AltitudeReference.W84
    """A code indicating the reference for a vertical distance. See AIXM 5.1 and FIXM 4.2.0. Currently, UTM only allows WGS84 with no immediate plans to allow other options. FIXM and AIXM allow for 'SFC' which is equivalent to AGL."""

    units: AltitudeUnits = AltitudeUnits.M
    """The reference quantities used to express the value of altitude. See FIXM 4.2. Currently, UTM only allows meters with no immediate plans to allow other options."""


Latitude = float
"""Degrees of latitude north of the equator, with reference to the WGS84 ellipsoid."""


Longitude = float
"""Degrees of longitude east of the Prime Meridian, with reference to the WGS84 ellipsoid."""


class LatLngPoint(ImplicitDict):
    """Point on the earth's surface."""

    lng: Longitude

    lat: Latitude


class Circle(ImplicitDict):
    """A circular area on the surface of the earth."""

    center: LatLngPoint | None

    radius: Radius | None


class ErrorResponse(ImplicitDict):
    """Human-readable string returned when an error occurs as a result of a USS - DSS transaction."""

    message: str | None
    """Human-readable message indicating what error occurred and/or why."""


SubscriptionNotificationIndex = int
"""Tracks the notifications sent for a subscription so the subscriber can detect missed notifications more easily."""


UssBaseURL = str
"""The base URL of a USS implementation of part or all of the USS-USS API. Per the USS-USS API, the full URL of a particular resource can be constructed by appending, e.g., `/uss/v1/{resource}/{id}` to this base URL. Accordingly, this URL may not have a trailing '/' character."""


class OperationalIntentState(str, Enum):
    """State of an operational intent.
    'Accepted': Operational intent is created and shared, but not yet in use; see standard text for more details.
    The create or update request for this operational intent reference must include a Key containing all OVNs for all relevant Entities.
    'Activated': Operational intent is in active use; see standard text for more details.
    The create or update request for this operational intent reference must include a Key containing all OVNs for all relevant Entities.
    'Nonconforming': UA is temporarily outside its volumes, but the situation is expected to be recoverable; see standard text for more details.
    In this state, the `/uss/v1/operational_intents/{entityid}/telemetry` USS-USS endpoint should respond, if available, to queries from USS peers.  The create or update request for this operational intent may omit a Key in this case because the operational intent is being adjusted as flown and cannot necessarily deconflict.
    'Contingent': UA is considered unrecoverably unable to conform with its coordinate operational intent; see standard text for more details.
    This state must transition to Ended.  In this state, the `/uss/v1/operational_intents/{entityid}/telemetry` USS-USS endpoint should respond, if available, to queries from USS peers.  The create or update request for this operational intent may omit a Key in this case because the operational intent is being adjusted as flown and cannot necessarily deconflict.
    """

    Accepted = "Accepted"
    Activated = "Activated"
    Nonconforming = "Nonconforming"
    Contingent = "Contingent"


OperationalIntentUssBaseURL = UssBaseURL


ConstraintUssBaseURL = UssBaseURL


Priority = int
"""Ordinal priority of the operational intent, as defined by the regulator.  Operational intents with lesser values are lower priority than all operational intents with greater values.  A lower-priority operational intent may not create a conflict with a higher-priority operational intent.  A higher-priority operational intent may create a conflict with a lower-priority operational intent.  The regulator specifies whether an operational intent may create a conflict with other operational intents of the same priority."""


class PositionAccuracyVertical(str, Enum):
    """Vertical error that is likely to be present in this reported position. This is the GVA enumeration from ADS-B, plus some finer values for UAS."""

    VAUnknown = "VAUnknown"
    VA150mPlus = "VA150mPlus"
    VA150m = "VA150m"
    VA45m = "VA45m"
    VA25m = "VA25m"
    VA10m = "VA10m"
    VA3m = "VA3m"
    VA1m = "VA1m"


class PositionAccuracyHorizontal(str, Enum):
    """Horizontal error that is likely to be present in this reported position. This is the NACp enumeration from ADS-B, plus 1m for a more complete range for UAS."""

    HAUnknown = "HAUnknown"
    HA10NMPlus = "HA10NMPlus"
    HA10NM = "HA10NM"
    HA4NM = "HA4NM"
    HA2NM = "HA2NM"
    HA1NM = "HA1NM"
    HA05NM = "HA05NM"
    HA03NM = "HA03NM"
    HA01NM = "HA01NM"
    HA005NM = "HA005NM"
    HA30m = "HA30m"
    HA10m = "HA10m"
    HA3m = "HA3m"
    HA1m = "HA1m"


class Position(ImplicitDict):
    """Location of the vehicle (UAS) as reported for UTM. Note: 'accuracy' values are required when extrapolated field is true."""

    longitude: Longitude | None

    latitude: Latitude | None

    accuracy_h: PositionAccuracyHorizontal | None

    accuracy_v: PositionAccuracyVertical | None

    extrapolated: bool | None = False
    """True if this position was generated primarily by computation rather than primarily from a direct instrument measurement."""

    altitude: Altitude | None


class VelocityUnitsSpeed(str, Enum):
    MetersPerSecond = "MetersPerSecond"


class Velocity(ImplicitDict):
    speed: float
    """Ground speed in meters/second."""

    units_speed: VelocityUnitsSpeed = VelocityUnitsSpeed.MetersPerSecond

    track: float | None = 0
    """Direction of flight expressed as a "True North-based" ground track angle. This value is provided in degrees East of North with a minimum resolution of 1 degree. A value of 360 indicates invalid, no value, or unknown."""


class UssAvailabilityState(str, Enum):
    """A USS is presumed to be in the Unknown state absent indication otherwise by a USS with availability arbitration scope.  Upon determination via availability arbitration, a USS is Down when it does not respond appropriately, and a Down USS may not perform the following operations in the DSS:
    * Create an operational intent in the Accepted or Activated states
    * Modify an operational intent whose new or unchanged state is Accepted or Activated
    * Delete an operational intent
    A USS in the Unknown state possesses all the capabilities, within the DSS, of a USS in the Normal state.
    """

    Unknown = "Unknown"
    Normal = "Normal"
    Down = "Down"


class SetUssAvailabilityStatusParameters(ImplicitDict):
    old_version: str = ""
    """Version of USS's availability to change, for consistent read-modify-write operations and consistent retry behavior."""

    availability: UssAvailabilityState


class ExchangeRecordRecorderRole(str, Enum):
    """A coded value that indicates the role of the logging USS: 'Client' (initiating a request to a remote USS) or 'Server' (handling a request from a remote USS)"""

    Client = "Client"
    Server = "Server"


class ExchangeRecord(ImplicitDict):
    """Details of a request/response data exchange."""

    url: str
    """Full URL of request."""

    method: str
    """HTTP verb used by requestor (e.g., "PUT," "GET," etc.)"""

    headers: list[str] | None = []
    """Set of headers associated with request or response. Requires 'Authorization:' field (at a minimum)"""

    recorder_role: ExchangeRecordRecorderRole
    """A coded value that indicates the role of the logging USS: 'Client' (initiating a request to a remote USS) or 'Server' (handling a request from a remote USS)"""

    request_time: Time
    """The time at which the request was sent/received."""

    request_body: str | None = ""
    """Base64-encoded body content sent/received as a request."""

    response_time: Time | None
    """The time at which the response was sent/received."""

    response_body: str | None = ""
    """Base64-encoded body content sent/received in response to request."""

    response_code: int | None = 0
    """HTTP response code sent/received in response to request."""

    problem: str | None
    """'Human-readable description of the problem with the exchange, if any.'"""


class ErrorReport(ImplicitDict):
    """A report informing a server of a communication problem."""

    report_id: str | None
    """ID assigned by the server receiving the report.  Not populated when submitting a report."""

    exchange: ExchangeRecord
    """The request (by this USS) and response associated with the error."""


class PlanningRecord(ImplicitDict):
    """A record of a single attempt to (successfully or unsuccessfully) create or modify an operational intent."""

    time: Time
    """Time that this planning event occurred"""

    ovns: list[EntityOVN] = []
    """OVNs the planning USS was aware of when it was planning the operational intent"""

    missing_operational_intents: list[EntityID] | None = []
    """List of missing operational intents (for planning attempts that were denied by the DSS with code 409)"""

    missing_constraints: list[EntityID] | None = []
    """List of missing constraints (for planning attempts that were denied by the DSS with code 409)"""

    operational_intent_id: EntityID | None
    """ID of the operational intent being planned"""

    problem: str | None
    """A free text description of the problem(s) encountered during this planning attempt."""


class UserNotificationRecordNotificationTriggeringEvent(str, Enum):
    """Requirement ID that pertains to the given notification"""

    GEN0400 = "GEN0400"
    GEN0405 = "GEN0405"
    SCD0090 = "SCD0090"
    SCD0095 = "SCD0095"
    ACM0010 = "ACM0010"
    CMSA0115 = "CMSA0115"
    CMSA0300 = "CMSA0300"
    CSTP0005 = "CSTP0005"
    CSTP0010 = "CSTP0010"
    CSTP0020 = "CSTP0020"
    CSTP0025 = "CSTP0025"
    CSTP0030 = "CSTP0030"
    CSTP0035 = "CSTP0035"


class UserNotificationRecord(ImplicitDict):
    """User notification record."""

    triggering_event_time: Time
    """Time of the notification triggering event"""

    notification_time: Time
    """Time at which the user was notified"""

    notification_details: str | None
    """Description of information that was provided to the user, as per the referenced notification_triggering_event requirement"""

    notification_triggering_event: UserNotificationRecordNotificationTriggeringEvent
    """Requirement ID that pertains to the given notification"""


class UserInputRecordInputTriggeringEvent(str, Enum):
    """Requirement ID that pertains to the given notification"""

    OPIN0040 = "OPIN0040"
    CMSA0010 = "CMSA0010"
    CMSA0025 = "CMSA0025"
    CMSA0100 = "CMSA0100"
    CMSA0105 = "CMSA0105"
    CMSA0110 = "CMSA0110"
    CMSA0200 = "CMSA0200"
    CMSA0205 = "CMSA0205"
    CMSA0210 = "CMSA0210"
    CMSA0215 = "CMSA0215"


class UserInputRecord(ImplicitDict):
    """User input record"""

    triggering_event_time: Time
    """Time in which user input was received by the USS"""

    operational_intent_id: EntityID
    """ID of the operational_intent ID pertaining to the user input"""

    input_triggering_event: UserInputRecordInputTriggeringEvent
    """Requirement ID that pertains to the given notification"""

    input_details: str | None
    """Description of the information that was provided by the user, as per the referenced input_triggering_event requirement"""


class OperatorAssociation(ImplicitDict):
    """Association between an operational intent and the operator of that operational intent"""

    operational_intent_id: EntityID
    """ID of operational intent to which this association pertains"""

    operator_id: str
    """Unique identifier of the operator responsible for the operational intent"""


class ConstraintProviderAssociation(ImplicitDict):
    """Association between a constraint and the constraint provider responsible for that constraint"""

    constraint_id: EntityID
    """ID of constraint to which this association pertains"""

    constraint_provider_id: str
    """Unique identifier of the constraint provider responsible for the constraint"""


class GeoZoneAdditionalProperties(ImplicitDict):
    """Indicates that exemptions from the national or European regulations are allowed in the UAS Zone, that will be detailed via the "message" property."""


CodeZoneIdentifierType = str
"""a string of maximum 7 characters that uniquely identifies the area within a geographical scope.
NOTE (1): This shall not include the country identifier, which is a separate attribute of the UAS Zone.
NOTE (2): The length of this data type is limited to 7 characters for compatibility with ARINC 424 and AIXM, where an airspace designator may have maximum 10 characters. The 10 characters are the result of concatenating the UAS Zone attributes for country and identifier.
"""


CodeCountryISOType = str
"""A 3 letter identifier of a country or territory using the ISO 3166-1 alpha-3 standard.
NOTE: >-
   The ISO 3-letter country codes come with the following advantages:
      - allow to distinguish between remote territories and mainland
      - are unique, unlike the ICAO Country codes where the same State
        could have two or more codes
      - are also used in military standards, such as NATO STANAG 1059
        INT, which come with well document additions that might be also
        useful for UAS areas.
"""


class CodeZoneType(str, Enum):
    """A coded list of values which allows indicating that the definition of a UAS Zone is specifically customised for a particular UAS or operator."""

    COMMON = "COMMON"
    CUSTOMIZED = "CUSTOMIZED"
    PROHIBITED = "PROHIBITED"
    REQ_AUTHORISATION = "REQ_AUTHORISATION"
    CONDITIONAL = "CONDITIONAL"
    NO_RESTRICTION = "NO_RESTRICTION"


ConditionExpressionType = str
"""A coded expression that provides information about what is authorised / forbidden in a zone that has conditional access.
By difference with the “Message” field per zone, this coded expression is made to be interopreted by the UAS while the “Message” is to interpreted by the remote pilot.
NOTE: the maximum field length is 10 000 characters.
---------------------- Condition definition language ---------------- • A list of relevant characteristics (CHARTYPE) has first to be established per state, and their finite list of acceptable values (CHARVAL)
• Each chartype and charval fields are defined by a limited set of characters
• A public document shall give the definitions of each, and provide the reference to legal or technical characteristics implied
• The Geozone editor per state can use these characteristics, with the dedicated condition language defined below, to define exact conditions per zone
• Each UAS Geofencing function shall be loaded with the corresponding chararacteristic status of the UAS for the intended flight, so as to be able to apply the conditions , either to generate alerts or to limit the flight
• If the value of a given characteristic of the condition equation is not defined in the UAS, the UAS Geofencing function should inform the pilot in Geoawareness alerting or consider that the zone is forbidden, by default in automatic Geofencing.
A conditional expression shall be of the following type:
• The UAS is PERMITTED XOR PROHIBITED (exclusive choice) to fly in this zone at this time IF (Characteristic1) CHARTYPE1 = (Value1) CHARVAL1 AND
CHARTYPE 2 = CHARVAL 2 AND ... AND End IF
OR (...)
...
End OR
• Only the fields in bold need to be edited in the character string, separated by”/”. Others are implicit.
Examples of CHARTYPE and CHARVALUE:
• CHARTYPE: operator type; Acceptable CHARVAL values: Military/Police/Firefighting
• CHARTYPE: Operator ID (registration number); Acceptable CHARVAL values: as per registration format
• CHARTYPE: Operation type: A1 as per EASA Open Types or S1 (National standard Scenario 1), STS01 (EASA Specific standard scenario) or ...
• CHARTYPE: UTM operation type: Planned/Unplanned,
• CHARTYPE: passengers on board: yes /no Note that it is possible in each national catalog of chartype and charval items, to define complex categories of operation/drone /equipment. Example: In nation A, we may have a type “drone level” with values Low, Medium, High. Each level corresponds to a defined set of required UAS performance/operation features/ operator qualification etc. This avoids to code a complex combination in the geozone database. This conditional expression can also be used to code a prohibition of image capture in a zone.
Example: PERMITTED/IMAGE CAPTURE=NO/NOISE

  CLASS=A/OR/OPERATOR=POLICE

Meaning: >-
   the fight is permitted in this zone at that time if No image is
captured (removed or deactivated) and if noise class = class A (following a known classification) or if the UAS operator is the Police
"""


CodeRestrictionType = str
"""An indication if flying in the zone is conditional, forbidden or unrestricted."""


class CodeZoneReasonType(str, Enum):
    """A coded indication of a reason that justifies the existence of an UAS Zone"""

    AIR_TRAFFIC = "AIR_TRAFFIC"
    SENSITIVE = "SENSITIVE"
    PRIVACY = "PRIVACY"
    POPULATION = "POPULATION"
    NATURE = "NATURE"
    NOISE = "NOISE"
    FOREIGN_TERRITORY = "FOREIGN_TERRITORY"
    EMERGENCY = "EMERGENCY"
    OTHER = "OTHER"


CodeUSpaceClassType = str
"""A coded identifier for a category or class of the zone applying a "USpace concept".
NOTE: >-
   In the current model version, there is no specific list of values.
For example, the “X”, “Y”, “Z” types of zones as per SESAR JU Corus project on USpace concept of operation could be used in a future version. Until a precise list of values is defined, this data type will be considered as string of characters of maximum 100 characters.
"""


CodeYesNoType = str
"""A coded value that indicates a choice between a positive (yes) or a negative (no) applicability.

Acceptable values:
* True
* False
"""


class CodeAuthorityRole(str, Enum):
    """A coded list of values indicating the role that an authority has in relation with the UAS zone."""

    AUTHORIZATION = "AUTHORIZATION"
    NOTIFICATION = "NOTIFICATION"
    INFORMATION = "INFORMATION"


TextShortType = str
"""A free text with a maximum length of 200 characters"""


class Polygon(ImplicitDict):
    """An enclosed area on the earth. The bounding edges of this polygon are defined to be the shortest paths between connected vertices.  This means, for instance, that the edge between two points both defined at a particular latitude is not generally contained at that latitude. The winding order must be interpreted as the order which produces the smaller area. The path between two vertices is defined to be the shortest possible path between those vertices. Edges may not cross. Vertices may not be duplicated.  In particular, the final polygon vertex must not be identical to the first vertex."""

    vertices: list[LatLngPoint]


class Volume3D(ImplicitDict):
    """A three-dimensional geographic volume consisting of a vertically-extruded shape. Exactly one outline must be specified."""

    outline_circle: Circle | None
    """A circular geographic shape on the surface of the earth."""

    outline_polygon: Polygon | None
    """A polygonal geographic shape on the surface of the earth."""

    altitude_lower: Altitude | None
    """Minimum bounding altitude of this volume. Must be less than altitude_upper, if specified."""

    altitude_upper: Altitude | None
    """Maximum bounding altitude of this volume. Must be greater than altitude_lower, if specified."""


class Volume4D(ImplicitDict):
    """Contiguous block of geographic spacetime."""

    volume: Volume3D

    time_start: Time | None
    """Beginning time of this volume. Must be before time_end."""

    time_end: Time | None
    """End time of this volume. Must be after time_start."""


class SubscriptionState(ImplicitDict):
    """State of subscription which is causing a notification to be sent."""

    subscription_id: SubscriptionID

    notification_index: SubscriptionNotificationIndex


class QuerySubscriptionParameters(ImplicitDict):
    """Parameters for a request to find subscriptions matching the provided criteria."""

    area_of_interest: Volume4D | None


SubscriptionUssBaseURL = UssBaseURL


class OperationalIntentReference(ImplicitDict):
    """The high-level information of a planned or active operational intent with the URL of a USS to query for details.  Note: 'ovn' is returned ONLY to the USS that created the operational intent but NEVER to other USS instances."""

    id: EntityID

    manager: str
    """Created by the DSS based on creating client's ID (via access token).  Used internal to the DSS for restricting mutation and deletion operations to manager.  Used by USSs to reject operational intent update notifications originating from a USS that does not manage the operational intent."""

    uss_availability: UssAvailabilityState

    version: EntityVersion

    state: OperationalIntentState

    ovn: EntityOVN | None
    """Opaque version number of this operational intent.  Populated only when the OperationalIntentReference is managed by the USS retrieving or providing it.  Not populated when the OperationalIntentReference is not managed by the USS retrieving or providing it (instead, the USS must obtain the OVN from the details retrieved from the managing USS)."""

    time_start: Time
    """Beginning time of operational intent."""

    time_end: Time
    """End time of operational intent."""

    uss_base_url: OperationalIntentUssBaseURL

    subscription_id: SubscriptionID
    """The ID of the subscription that is ensuring the operational intent manager receives relevant airspace updates."""


class ImplicitSubscriptionParameters(ImplicitDict):
    """Information necessary to create a subscription to serve a single operational intent's notification needs."""

    uss_base_url: SubscriptionUssBaseURL
    """The base URL of a USS implementation of the parts of the USS-USS API necessary for receiving the notifications that the operational intent must be aware of.  This includes, at least, notifications for relevant changes in operational intents."""

    notify_for_constraints: bool | None = False
    """True if this operational intent's subscription should trigger notifications when constraints change. Otherwise, changes in constraints should not trigger notifications.  The scope utm.constraint_processing is required to set this flag true, and a USS performing the constraint processing role should set this flag true."""


class GetOperationalIntentReferenceResponse(ImplicitDict):
    """Response to DSS request for the OperationalIntentReference with the given ID."""

    operational_intent_reference: OperationalIntentReference


class QueryOperationalIntentReferenceParameters(ImplicitDict):
    """Parameters for a request to find OperationalIntentReferences matching the provided criteria."""

    area_of_interest: Volume4D | None


class QueryOperationalIntentReferenceResponse(ImplicitDict):
    """Response to DSS query for OperationalIntentReferences in an area of interest."""

    operational_intent_references: list[OperationalIntentReference] = []
    """OperationalIntentReferences in the area of interest."""


class ConstraintReference(ImplicitDict):
    """A ConstraintReference (area in which a constraint is present, along with other high-level information, but no details).  The DSS reports only these references and clients must exchange details and additional information peer-to-peer."""

    id: EntityID

    manager: str
    """Created by the DSS based on creating client's ID (via access token).  Used internal to the DSS for restricting mutation and deletion operations to manager.  Used by USSs to reject constraint update notifications originating from a USS that does not manage the constraint."""

    uss_availability: UssAvailabilityState

    version: EntityVersion

    ovn: EntityOVN | None
    """Opaque version number of this constraint.  Populated only when the ConstraintReference is managed by the USS retrieving or providing it.  Not populated when the ConstraintReference is not managed by the USS retrieving or providing it (instead, the USS must obtain the OVN from the details retrieved from the managing USS)."""

    time_start: Time

    time_end: Time

    uss_base_url: ConstraintUssBaseURL


class PutConstraintReferenceParameters(ImplicitDict):
    """Parameters for a request to create/update a ConstraintReference in the DSS."""

    extents: list[Volume4D]
    """Spacetime extents that bound this constraint.
    The end time may not be in the past.
    All volumes of the constraint must be encompassed in these extents. However, these extents do not need to match the precise volumes of the constraint; a single bounding extent may be provided instead, for instance.
    """

    uss_base_url: ConstraintUssBaseURL


class GetConstraintReferenceResponse(ImplicitDict):
    """Response to DSS request for the ConstraintReference with the given ID."""

    constraint_reference: ConstraintReference


class QueryConstraintReferenceParameters(ImplicitDict):
    """Parameters for a request to find ConstraintReferences matching the provided criteria."""

    area_of_interest: Volume4D | None


class QueryConstraintReferencesResponse(ImplicitDict):
    """Response to DSS query for ConstraintReferences in an area of interest."""

    constraint_references: list[ConstraintReference] = []
    """ConstraintReferences in the area of interest."""


class AirspaceConflictResponse(ImplicitDict):
    """Data provided when an airspace conflict was encountered."""

    message: str | None
    """Human-readable message indicating what error occurred and/or why."""

    missing_operational_intents: list[OperationalIntentReference] | None = []
    """List of operational intent references for which current proof of knowledge was not provided.  If this field is present and contains elements, the calling USS should query the details URLs for these operational intents to obtain their details and correct OVNs.  The OVNs can be used to update the key, at which point the USS may retry this call."""

    missing_constraints: list[ConstraintReference] | None = []
    """List of constraint references for which current proof of knowledge was not provided.  If this field is present and contains elements, the calling USS should query the details URLs for these constraints to obtain their details and correct OVNs.  The OVNs can be used to update the key, at which point the USS may retry this call."""


class OperationalIntentDetails(ImplicitDict):
    """Details of a UTM operational intent. Note that this data is not stored in the DSS; only with the clients."""

    volumes: list[Volume4D] | None = []
    """Volumes that wholly contain the operational intent while being as small as practical.
    Start and end times, as well as lower and upper altitudes, are required for each volume. The end time may not be in the past.
    Required with at least one item when the operational intent is Accepted, Activated, or Nonconforming.
    May not contain any items when the operational intent is Contingent.
    """

    off_nominal_volumes: list[Volume4D] | None = []
    """Volumes that contain the anticipated area of non-conformance while the aircraft is in the Nonconforming or Contingent states.
    Start and end times, as well as lower and upper altitudes, are required for each volume. The end time may not be in the past.
    Required with at least one item when the operational intent is Nonconforming or Contingent.
    May not contain any items when the operational intent is Accepted or Activated.
    """

    priority: Priority | None


class OperationalIntent(ImplicitDict):
    """Full description of a UTM operational intent."""

    reference: OperationalIntentReference

    details: OperationalIntentDetails


class PutOperationalIntentDetailsParameters(ImplicitDict):
    """Parameters of a message informing of detailed information for a peer operational intent. Pushed (by a client, not the DSS) directly to clients with subscriptions when another client makes a change to airspace within a cell with a subscription."""

    operational_intent_id: EntityID
    """ID of operational intent that has changed."""

    operational_intent: OperationalIntent | None
    """Full information about the operational intent that has changed.  If this field is omitted, the operational intent was deleted.  The `ovn` field in the nested `reference` must be populated."""

    subscriptions: list[SubscriptionState]
    """Subscription(s) prompting this notification."""


class GetOperationalIntentDetailsResponse(ImplicitDict):
    """Response to peer request for the details of operational intent with the given ID."""

    operational_intent: OperationalIntent


class VehicleTelemetry(ImplicitDict):
    """Vehicle position, altitude, and velocity."""

    time_measured: Time

    position: Position | None

    velocity: Velocity | None


class UssAvailabilityStatus(ImplicitDict):
    uss: str
    """Client ID (matching their `sub` in access tokens) of the USS to which this availability applies."""

    availability: UssAvailabilityState


class UssAvailabilityStatusResponse(ImplicitDict):
    version: str
    """Current version of USS's availability.  Used to change USS's availability."""

    status: UssAvailabilityStatus


class PositionRecord(ImplicitDict):
    """A record of vehicle telemetry information received by this USS (typically for conformance monitoring)."""

    time_received: Time
    """Time that this position data was received by the USS"""

    telemetry: VehicleTelemetry


class OperationalIntentPositions(ImplicitDict):
    """A record of position data gathered through the course of an operational intent"""

    positions: list[PositionRecord] | None = []

    operational_intent_id: EntityID
    """ID of the operational intent associated with `positions`"""


class Authority(ImplicitDict):
    """A relevant authority that is in charge for authorising, being notified or providing information for UAS operations in the UAS zone.
    Rule: >-
       at least one of the following shall be specified - siteURL, email,
    phone.
    """

    name: TextShortType | None
    """The official name of a public or private authority"""

    service: TextShortType | None
    """The name of a specific department or service within the organisation"""

    contact_name: TextShortType | None
    """The name or role of a specific person that needs to be contacted within the organisation"""

    site_url: TextShortType | None
    """The URL of the public internet site through which the organisation may be contacted
    Note: in the data coding format, this might be further constrained in order to ensure a valid URL format.
    """

    email: TextShortType | None
    """The e-mail address by which the organisation may be contacted.
    Note: in the data coding format, this might be further constrained in order to ensure a valid e-mail format.
    """

    phone: TextShortType | None
    """A phone number at which the organisation may be contacted"""

    purpose: CodeAuthorityRole | None
    """The role of the Authority in relation with the zone."""

    interval_before: str | None
    """The minimal time interval required between notification or authorization request and starting to operate in the zone, in the format PnnDTnnHnnM (ISO 8601)."""


class SubscriberToNotify(ImplicitDict):
    """Subscriber to notify of a change in the airspace.  This is provided by the DSS to a client changing the airspace, and it is the responsibility of that client to send a notification to the specified USS according to the change made to the airspace."""

    subscriptions: list[SubscriptionState]
    """Subscription(s) prompting this notification."""

    uss_base_url: SubscriptionUssBaseURL


class Subscription(ImplicitDict):
    """Specification of a geographic area that a client is interested in on an ongoing basis (e.g., "planning area")."""

    id: SubscriptionID

    version: str
    """Version of the subscription that the DSS changes every time a USS changes the subscription.  The DSS incrementing the notification_index does not constitute a change that triggers a new version.  A USS must specify this version when modifying an existing subscription to ensure consistency in read-modify-write operations and distributed systems."""

    notification_index: SubscriptionNotificationIndex

    time_start: Time | None
    """If set, this subscription will not receive notifications involving airspace changes entirely before this time."""

    time_end: Time | None
    """If set, this subscription will not receive notifications involving airspace changes entirely after this time."""

    uss_base_url: SubscriptionUssBaseURL

    notify_for_operational_intents: bool | None = False
    """If true, trigger notifications when operational intents are created, updated, or deleted.  Otherwise, changes in operational intents should not trigger notifications.  The scope utm.strategic_coordination is required to set this flag true."""

    notify_for_constraints: bool | None = False
    """If true, trigger notifications when constraints are created, updated, or deleted.  Otherwise, changes in constraints should not trigger notifications.  The scope utm.constraint_processing is required to set this flag true."""

    implicit_subscription: bool | None = False
    """True if this subscription was implicitly created by the DSS via the creation of an operational intent, and should therefore be deleted by the DSS when that operational intent is deleted."""

    dependent_operational_intents: list[EntityID] | None = []
    """List of IDs for operational intents that are dependent on this subscription."""


class QuerySubscriptionsResponse(ImplicitDict):
    """Response to DSS query for subscriptions in a particular geographic area."""

    subscriptions: list[Subscription] = []
    """Subscriptions that overlap the specified geographic area."""


class GetSubscriptionResponse(ImplicitDict):
    """Response to DSS request for the subscription with the given id."""

    subscription: Subscription


class PutSubscriptionParameters(ImplicitDict):
    """Parameters for a request to create/update a subscription in the DSS.  At least one form of notifications must be requested."""

    extents: Volume4D
    """Spacetime extents of the volume to subscribe to.
    This subscription will automatically be deleted after its end time if it has not been refreshed by then. If end time is not specified, the value will be chosen automatically by the DSS. If start time is not specified, it will default to the time the request is processed. The end time may not be in the past.
    Note that some Entities triggering notifications may lie entirely outside the requested area.
    """

    uss_base_url: SubscriptionUssBaseURL

    notify_for_operational_intents: bool | None = False
    """If true, trigger notifications when operational intents are created, updated, or deleted.  Otherwise, changes in operational intents should not trigger notifications.  The scope utm.strategic_coordination is required to set this flag true."""

    notify_for_constraints: bool | None = False
    """If true, trigger notifications when constraints are created, updated, or deleted.  Otherwise, changes in constraints should not trigger notifications.  The scope utm.constraint_processing is required to set this flag true."""


class PutSubscriptionResponse(ImplicitDict):
    """Response for a request to create or update a subscription."""

    subscription: Subscription

    operational_intent_references: list[OperationalIntentReference] | None = []
    """Operational intents in or near the subscription area at the time of creation/update, if `notify_for_operational_intents` is true."""

    constraint_references: list[ConstraintReference] | None = []
    """Constraints in or near the subscription area at the time of creation/update, if `notify_for_constraints` is true."""


class DeleteSubscriptionResponse(ImplicitDict):
    """Response for a successful request to delete a subscription."""

    subscription: Subscription


class PutOperationalIntentReferenceParameters(ImplicitDict):
    """Parameters for a request to create an OperationalIntentReference in the DSS. A subscription to changes overlapping this volume may be implicitly created, but this can be overridden by providing the (optional) 'subscription_id' to use. Note: The implicit subscription is managed by the DSS, not the USS."""

    extents: list[Volume4D]
    """Spacetime extents that bound this operational intent.
    Start and end times, as well as lower and upper altitudes, are required for each volume. The end time may not be in the past. All volumes, both nominal and off-nominal, must be encompassed in these extents. However, these extents do not need to match the precise volumes of the operational intent; a single bounding extent may be provided instead, for instance.
    """

    key: Key | None
    """Proof that the USS creating or mutating this operational intent was aware of the current state of the airspace, with the expectation that this operational intent is therefore deconflicted from all relevant features in the airspace.  This field is not required when declaring an operational intent Nonconforming or Contingent, or when there are no relevant Entities in the airspace, but is otherwise required. OVNs for constraints are required if and only if the USS managing this operational intent is performing the constraint processing role, which is indicated by whether the subscription associated with this operational intent triggers notifications for constraints.  The key does not need to contain the OVN for the operational intent being updated."""

    state: OperationalIntentState

    uss_base_url: OperationalIntentUssBaseURL

    subscription_id: EntityID | None
    """The ID of an existing subscription that the USS will use to keep the operator informed about updates to relevant airspace information. If this field is not provided when the operational intent is in the Activated, Nonconforming, or Contingent state, then the `new_subscription` field must be provided in order to provide notification capability for the operational intent.  The subscription specified by this ID must cover at least the area over which this operational intent is conducted, and it must provide notifications for operational intents."""

    new_subscription: ImplicitSubscriptionParameters | None
    """If an existing subscription is not specified in `subscription_id`, and the operational intent is in the Activated, Nonconforming, or Contingent state, then this field must be populated.  When this field is populated, an implicit subscription will be created and associated with this operational intent, and will generally be deleted automatically upon the deletion of this operational intent."""

    requested_ovn_suffix: UUIDv7Format | None
    """This optional field not part of the original F3548 standard API allows a USS to request a specific OVN when creating or updating an operational intent. When creating an operational intent, this enables a USS to immediately publish the operational intent details with the expected OVN. When updating an operational intent, this enables a USS to immediately make available this new version of the operational intent details if specifically requested by the remote USS. The USS must still wait for the DSS receipt to actually publish the new operational intent details. This allows USSs to obtain correct operational intent details even if the DSS takes a long time to respond and/or the USS processing  it.
    The requested suffix must be a UUIDv7 string containing a timestamp of the current time. If the suffix is invalid, and notably if the time is too far in the past or the future, the request will be rejected. If the suffix is valid, the DSS will set the OVN of the operational intent to be `{entityid}_{requested_ovn_suffix}`. If no suffix is set, the DSS will proceed as specified by the standard.
    """


class ChangeOperationalIntentReferenceResponse(ImplicitDict):
    """Response to a request to create, update, or delete an OperationalIntentReference in the DSS."""

    subscribers: list[SubscriberToNotify] = []
    """DSS subscribers that this client now has the obligation to notify of the operational intent changes just made.  This client must call POST for each provided URL according to the USS-USS `/uss/v1/operational_intents` path API.  The client's own subscriptions will also be included in this list."""

    operational_intent_reference: OperationalIntentReference


class ChangeConstraintReferenceResponse(ImplicitDict):
    """Response to a request to create, update, or delete a ConstraintReference. in the DSS."""

    subscribers: list[SubscriberToNotify] = []
    """DSS subscribers that this client now has the obligation to notify of the constraint changes just made.  This client must call POST for each provided URL according to the USS-USS `/uss/v1/constraints` path API.  The client's own subscriptions will also be included in this list."""

    constraint_reference: ConstraintReference


class GetOperationalIntentTelemetryResponse(ImplicitDict):
    """Response to a peer request for telemetry of an off-nominal operational intent."""

    operational_intent_id: EntityID
    """ID of the operational intent which the vehicle reporting telemetry is flying."""

    telemetry: VehicleTelemetry | None

    next_telemetry_opportunity: Time | None
    """The next telemetry similar to this telemetry is not expected to be available until at or after this time, so the polling USS should generally not poll the endpoint providing this response data again until at or after that time.  If this field is omitted, then there is no current expectation of new telemetry becoming available."""


class USSLogSet(ImplicitDict):
    """The set of log data fulfilling this standard's Logging requirements."""

    messages: list[ExchangeRecord] | None = []
    """Outgoing messages sent to other USSs and the DSS, and incoming messages received from other USSs, including instances where an expected response to a request is not received."""

    operator_notifications: list[UserNotificationRecord] | None = []
    """Instances of operator notifications as specifically required within this standard."""

    operator_inputs: list[UserInputRecord] | None = []
    """Instances of operator input as specifically required within this standard."""

    operator_associations: list[OperatorAssociation] | None = []
    """For a USS that manages operational intents, associations of an operator with operational intents that transitioned to the Accepted state."""

    planning_attempts: list[PlanningRecord] | None = []
    """For a USS that manages operational intents, instances where an operational intent could not be planned or replanned due to conflicts with other operational intents or constraints."""

    operational_intent_positions: list[OperationalIntentPositions] | None = []
    """For a USS performing conformance monitoring, all position data used for conformance monitoring that is ingested from the UA."""

    constraint_provider_associations: list[ConstraintProviderAssociation] | None = []
    """For a USS that performs constraint management, associations of an authorized constraint provider with all constraints that transition to the valid state."""


class GeoZone(ImplicitDict):
    """An airspace of defined dimensions, above the land areas or territorial waters of a State, within which a particular restriction or condition for UAS flights applies."""

    identifier: CodeZoneIdentifierType
    """A string of characters that uniquely identifies the UAS Zone within the State/Territory identified by the country attribute.
    Note - The UAS Zone is uniquely identified worldwide by the combination of the country and the identifier attributes
    """

    country: CodeCountryISOType
    """The State that has the authority to declare the zone.
    Note - There will be no Zone belonging to two States. Not necessary to code the information that two zones are "in neighboring States" or "related".
    """

    zone_authority: list[Authority]

    name: TextShortType | None
    """A free text name by which the zone may be known by the public or by the UAS community."""

    type: CodeZoneType
    """An indication whether the Zone is provided with its common definition or with a customised definition, for a particular user."""

    restriction: CodeRestrictionType
    """An indication if flying in the zone is conditional, forbidden or unrestricted."""

    restriction_conditions: list[ConditionExpressionType] | None
    """An indication of the conditions under which the zone can be used"""

    region: int | None
    """Where applicable, identifies a region inside a State where the UAS Zone is located.
    Note 1) identified with a digit between 0-65535 (16 bit), corresponding to a list of regions pre-defined for each State.
    Note 2) this attribute is intended to facilitate extracting sub-sets of data, for specific regions
    """

    reason: list[CodeZoneReasonType] | None
    """A coded indication for the reason that led to the establishment of the zone."""

    other_reason_info: str | None
    """A free text description of the reason that led to the establishment of the zone, when not covered by a pre-defined coded value."""

    regulation_exemption: CodeYesNoType | None
    """This is an extension point. It allows adding additional attributes of national interest through this element."""

    u_space_class: CodeUSpaceClassType | None
    """A code that identifies the category or class of the zone applying a "USpace concept".
    Note: Two (draft) classifications exist, one from Eurocontrol and one from CORUS. Therefore, two instances of this attribute are expected, one from each sub-list. This might be later replaced with separate attributes and separate lists of values.
    """

    message: TextShortType | None
    """A message to be displayed to the user of the zone, typically on the RPS for the Remote Pilot, to make him/her aware about specific information associated with the zone (typically when it is not only a restriction to fly in the zone, thus not only an alert or an automatic limitation, for example : “image capture prohibited in this zone”, “frequent strong winds in this zone”, “no landing or take-off in this zone”). This message is also used to indicate exemptions from regulation in a zone (see below). Several information can be grouped in a message, separated by a “/”."""

    additional_properties: GeoZoneAdditionalProperties | None
    """Indicates that exemptions from the national or European regulations are allowed in the UAS Zone, that will be detailed via the "message" property."""


class ConstraintDetails(ImplicitDict):
    """Details of a UTM constraint. Note that this data is not stored in the DSS; only with the clients."""

    volumes: list[Volume4D]
    """Volumes that wholly contain the constraint while being as small as practical.
    The end time may not be in the past.
    """

    type: str | None
    """Type of airspace feature this constraint represents."""

    geozone: GeoZone | None
    """If this constraint is an ED-269 compliant geo zone, the details about that geo zone."""


class Constraint(ImplicitDict):
    """Full specification of a UTM constraint."""

    reference: ConstraintReference

    details: ConstraintDetails


class PutConstraintDetailsParameters(ImplicitDict):
    """Parameters of a message informing of new full information for a constraint.  Pushed (by a client, not the DSS) directly to clients with subscriptions when another client makes a change to airspace within a cell with a subscription."""

    constraint_id: EntityID
    """ID of constraint that has changed."""

    constraint: Constraint | None
    """Full information about the constraint that has changed.  If this field is omitted, the constraint was deleted.  The `ovn` field in the nested `reference` must be populated."""

    subscriptions: list[SubscriptionState]
    """Subscription(s) prompting this notification."""


class GetConstraintDetailsResponse(ImplicitDict):
    """Response to peer request for the details of operational intent with the given ID."""

    constraint: Constraint


class OperationID(str, Enum):
    QueryOperationalIntentReferences = "queryOperationalIntentReferences"
    GetOperationalIntentReference = "getOperationalIntentReference"
    CreateOperationalIntentReference = "createOperationalIntentReference"
    UpdateOperationalIntentReference = "updateOperationalIntentReference"
    DeleteOperationalIntentReference = "deleteOperationalIntentReference"
    QueryConstraintReferences = "queryConstraintReferences"
    GetConstraintReference = "getConstraintReference"
    CreateConstraintReference = "createConstraintReference"
    UpdateConstraintReference = "updateConstraintReference"
    DeleteConstraintReference = "deleteConstraintReference"
    QuerySubscriptions = "querySubscriptions"
    GetSubscription = "getSubscription"
    CreateSubscription = "createSubscription"
    UpdateSubscription = "updateSubscription"
    DeleteSubscription = "deleteSubscription"
    MakeDssReport = "makeDssReport"
    SetUssAvailability = "setUssAvailability"
    GetUssAvailability = "getUssAvailability"
    GetOperationalIntentDetails = "getOperationalIntentDetails"
    GetOperationalIntentTelemetry = "getOperationalIntentTelemetry"
    NotifyOperationalIntentDetailsChanged = "notifyOperationalIntentDetailsChanged"
    GetConstraintDetails = "getConstraintDetails"
    NotifyConstraintDetailsChanged = "notifyConstraintDetailsChanged"
    MakeUssReport = "makeUssReport"
    GetLogSet = "getLogSet"


OPERATIONS: dict[OperationID, Operation] = {
    OperationID.QueryOperationalIntentReferences: Operation(
        id="queryOperationalIntentReferences",
        path="/dss/v1/operational_intent_references/query",
        verb="POST",
        request_body_type=QueryOperationalIntentReferenceParameters,
        response_body_type={
            200: QueryOperationalIntentReferenceResponse,
            400: ErrorResponse,
            401: ErrorResponse,
            403: ErrorResponse,
            413: ErrorResponse,
            429: ErrorResponse,
        },
    ),
    OperationID.GetOperationalIntentReference: Operation(
        id="getOperationalIntentReference",
        path="/dss/v1/operational_intent_references/{entityid}",
        verb="GET",
        request_body_type=None,
        response_body_type={
            200: GetOperationalIntentReferenceResponse,
            400: ErrorResponse,
            401: ErrorResponse,
            403: ErrorResponse,
            404: ErrorResponse,
            429: ErrorResponse,
        },
    ),
    OperationID.CreateOperationalIntentReference: Operation(
        id="createOperationalIntentReference",
        path="/dss/v1/operational_intent_references/{entityid}",
        verb="PUT",
        request_body_type=PutOperationalIntentReferenceParameters,
        response_body_type={
            201: ChangeOperationalIntentReferenceResponse,
            400: ErrorResponse,
            401: ErrorResponse,
            403: ErrorResponse,
            409: AirspaceConflictResponse,
            412: ErrorResponse,
            413: ErrorResponse,
            429: ErrorResponse,
        },
    ),
    OperationID.UpdateOperationalIntentReference: Operation(
        id="updateOperationalIntentReference",
        path="/dss/v1/operational_intent_references/{entityid}/{ovn}",
        verb="PUT",
        request_body_type=PutOperationalIntentReferenceParameters,
        response_body_type={
            200: ChangeOperationalIntentReferenceResponse,
            400: ErrorResponse,
            401: ErrorResponse,
            403: ErrorResponse,
            409: AirspaceConflictResponse,
            412: ErrorResponse,
            413: ErrorResponse,
            429: ErrorResponse,
        },
    ),
    OperationID.DeleteOperationalIntentReference: Operation(
        id="deleteOperationalIntentReference",
        path="/dss/v1/operational_intent_references/{entityid}/{ovn}",
        verb="DELETE",
        request_body_type=None,
        response_body_type={
            200: ChangeOperationalIntentReferenceResponse,
            400: ErrorResponse,
            401: ErrorResponse,
            403: ErrorResponse,
            404: ErrorResponse,
            409: ErrorResponse,
            412: ErrorResponse,
            429: ErrorResponse,
        },
    ),
    OperationID.QueryConstraintReferences: Operation(
        id="queryConstraintReferences",
        path="/dss/v1/constraint_references/query",
        verb="POST",
        request_body_type=QueryConstraintReferenceParameters,
        response_body_type={
            200: QueryConstraintReferencesResponse,
            400: ErrorResponse,
            401: ErrorResponse,
            403: ErrorResponse,
            413: ErrorResponse,
            429: ErrorResponse,
        },
    ),
    OperationID.GetConstraintReference: Operation(
        id="getConstraintReference",
        path="/dss/v1/constraint_references/{entityid}",
        verb="GET",
        request_body_type=None,
        response_body_type={
            200: GetConstraintReferenceResponse,
            400: ErrorResponse,
            401: ErrorResponse,
            403: ErrorResponse,
            404: ErrorResponse,
            429: ErrorResponse,
        },
    ),
    OperationID.CreateConstraintReference: Operation(
        id="createConstraintReference",
        path="/dss/v1/constraint_references/{entityid}",
        verb="PUT",
        request_body_type=PutConstraintReferenceParameters,
        response_body_type={
            201: ChangeConstraintReferenceResponse,
            400: ErrorResponse,
            401: ErrorResponse,
            403: ErrorResponse,
            409: ErrorResponse,
            413: ErrorResponse,
            429: ErrorResponse,
        },
    ),
    OperationID.UpdateConstraintReference: Operation(
        id="updateConstraintReference",
        path="/dss/v1/constraint_references/{entityid}/{ovn}",
        verb="PUT",
        request_body_type=PutConstraintReferenceParameters,
        response_body_type={
            200: ChangeConstraintReferenceResponse,
            400: ErrorResponse,
            401: ErrorResponse,
            403: ErrorResponse,
            409: ErrorResponse,
            413: ErrorResponse,
            429: ErrorResponse,
        },
    ),
    OperationID.DeleteConstraintReference: Operation(
        id="deleteConstraintReference",
        path="/dss/v1/constraint_references/{entityid}/{ovn}",
        verb="DELETE",
        request_body_type=None,
        response_body_type={
            200: ChangeConstraintReferenceResponse,
            400: ErrorResponse,
            401: ErrorResponse,
            403: ErrorResponse,
            404: ErrorResponse,
            409: ErrorResponse,
            429: ErrorResponse,
        },
    ),
    OperationID.QuerySubscriptions: Operation(
        id="querySubscriptions",
        path="/dss/v1/subscriptions/query",
        verb="POST",
        request_body_type=QuerySubscriptionParameters,
        response_body_type={
            200: QuerySubscriptionsResponse,
            400: ErrorResponse,
            401: ErrorResponse,
            403: ErrorResponse,
            413: ErrorResponse,
            429: ErrorResponse,
        },
    ),
    OperationID.GetSubscription: Operation(
        id="getSubscription",
        path="/dss/v1/subscriptions/{subscriptionid}",
        verb="GET",
        request_body_type=None,
        response_body_type={
            200: GetSubscriptionResponse,
            400: ErrorResponse,
            401: ErrorResponse,
            403: ErrorResponse,
            404: ErrorResponse,
            429: ErrorResponse,
        },
    ),
    OperationID.CreateSubscription: Operation(
        id="createSubscription",
        path="/dss/v1/subscriptions/{subscriptionid}",
        verb="PUT",
        request_body_type=PutSubscriptionParameters,
        response_body_type={
            200: PutSubscriptionResponse,
            400: ErrorResponse,
            401: ErrorResponse,
            403: ErrorResponse,
            409: ErrorResponse,
            429: ErrorResponse,
        },
    ),
    OperationID.UpdateSubscription: Operation(
        id="updateSubscription",
        path="/dss/v1/subscriptions/{subscriptionid}/{version}",
        verb="PUT",
        request_body_type=PutSubscriptionParameters,
        response_body_type={
            200: PutSubscriptionResponse,
            400: ErrorResponse,
            401: ErrorResponse,
            403: ErrorResponse,
            409: ErrorResponse,
            429: ErrorResponse,
        },
    ),
    OperationID.DeleteSubscription: Operation(
        id="deleteSubscription",
        path="/dss/v1/subscriptions/{subscriptionid}/{version}",
        verb="DELETE",
        request_body_type=None,
        response_body_type={
            200: DeleteSubscriptionResponse,
            400: ErrorResponse,
            401: ErrorResponse,
            403: ErrorResponse,
            404: ErrorResponse,
            409: ErrorResponse,
            429: ErrorResponse,
        },
    ),
    OperationID.MakeDssReport: Operation(
        id="makeDssReport",
        path="/dss/v1/reports",
        verb="POST",
        request_body_type=ErrorReport,
        response_body_type={
            201: ErrorReport,
            400: ErrorResponse,
            401: ErrorResponse,
            403: ErrorResponse,
            429: ErrorResponse,
        },
    ),
    OperationID.SetUssAvailability: Operation(
        id="setUssAvailability",
        path="/dss/v1/uss_availability/{uss_id}",
        verb="PUT",
        request_body_type=SetUssAvailabilityStatusParameters,
        response_body_type={
            200: UssAvailabilityStatusResponse,
            400: ErrorResponse,
            401: ErrorResponse,
            403: ErrorResponse,
            429: ErrorResponse,
        },
    ),
    OperationID.GetUssAvailability: Operation(
        id="getUssAvailability",
        path="/dss/v1/uss_availability/{uss_id}",
        verb="GET",
        request_body_type=None,
        response_body_type={
            200: UssAvailabilityStatusResponse,
            400: ErrorResponse,
            401: ErrorResponse,
            403: ErrorResponse,
            429: ErrorResponse,
        },
    ),
    OperationID.GetOperationalIntentDetails: Operation(
        id="getOperationalIntentDetails",
        path="/uss/v1/operational_intents/{entityid}",
        verb="GET",
        request_body_type=None,
        response_body_type={
            200: GetOperationalIntentDetailsResponse,
            400: ErrorResponse,
            401: ErrorResponse,
            403: ErrorResponse,
            404: ErrorResponse,
            429: ErrorResponse,
        },
    ),
    OperationID.GetOperationalIntentTelemetry: Operation(
        id="getOperationalIntentTelemetry",
        path="/uss/v1/operational_intents/{entityid}/telemetry",
        verb="GET",
        request_body_type=None,
        response_body_type={
            200: GetOperationalIntentTelemetryResponse,
            400: ErrorResponse,
            401: ErrorResponse,
            403: ErrorResponse,
            404: ErrorResponse,
            409: ErrorResponse,
            412: ErrorResponse,
            429: ErrorResponse,
        },
    ),
    OperationID.NotifyOperationalIntentDetailsChanged: Operation(
        id="notifyOperationalIntentDetailsChanged",
        path="/uss/v1/operational_intents",
        verb="POST",
        request_body_type=PutOperationalIntentDetailsParameters,
        response_body_type={
            204: None,
            400: ErrorResponse,
            401: ErrorResponse,
            403: ErrorResponse,
            409: ErrorResponse,
            429: ErrorResponse,
        },
    ),
    OperationID.GetConstraintDetails: Operation(
        id="getConstraintDetails",
        path="/uss/v1/constraints/{entityid}",
        verb="GET",
        request_body_type=None,
        response_body_type={
            200: GetConstraintDetailsResponse,
            400: ErrorResponse,
            401: ErrorResponse,
            403: ErrorResponse,
            404: ErrorResponse,
            429: ErrorResponse,
        },
    ),
    OperationID.NotifyConstraintDetailsChanged: Operation(
        id="notifyConstraintDetailsChanged",
        path="/uss/v1/constraints",
        verb="POST",
        request_body_type=PutConstraintDetailsParameters,
        response_body_type={
            204: None,
            400: ErrorResponse,
            401: ErrorResponse,
            403: ErrorResponse,
            409: ErrorResponse,
            429: ErrorResponse,
        },
    ),
    OperationID.MakeUssReport: Operation(
        id="makeUssReport",
        path="/uss/v1/reports",
        verb="POST",
        request_body_type=ErrorReport,
        response_body_type={
            201: ErrorReport,
            400: ErrorResponse,
            401: ErrorResponse,
            403: ErrorResponse,
            429: ErrorResponse,
        },
    ),
    OperationID.GetLogSet: Operation(
        id="getLogSet",
        path="/uss/v1/log_sets/{log_set_id}",
        verb="GET",
        request_body_type=None,
        response_body_type={
            200: USSLogSet,
        },
    ),
}
