"""This module is used to normalize audio files in one or more directories.
"""
import logging

from pydub import AudioSegment, effects, utils

from djtools.configs.config import BaseConfig
from djtools.utils.helpers import get_local_tracks


logger = logging.getLogger(__name__)


def normalize(config: BaseConfig):
    """Gets local tracks and normalizes them.

    Tracks will be overwritten and have a headroom equal to
    NORMALIZE_AUDIO_HEADROOM.

    Args:
        config: Configuration object.

    Raises:
        RuntimeError: Must have local tracks to normalize.
    """
    folder_tracks = get_local_tracks(config)
    if not folder_tracks:
        raise RuntimeError(
            "There are no local tracks; make sure LOCAL_DIRS has one or "
            "more directories containing one or more tracks"
        )

    for track in [
        track for tracks in folder_tracks.values() for track in tracks
    ]:
        audio = AudioSegment.from_file(track)

        if abs(audio.max_dBFS + config.NORMALIZE_AUDIO_HEADROOM) > 0.001:
            logger.info(
                f"{track} has a max dB of {audio.max_dBFS}, normalizing to "
                f"have a headroom of {config.NORMALIZE_AUDIO_HEADROOM}..."
            )
            audio = effects.normalize(
                audio, headroom=config.NORMALIZE_AUDIO_HEADROOM
            )
            audio.export(
                track,
                tags=utils.mediainfo(track).get("TAG", {}),
                bitrate=config.AUDIO_BITRATE,
                format=config.AUDIO_FORMAT,
            )
            continue

        logger.info(
            f"{track} already has a headroom of "
            f"{config.NORMALIZE_AUDIO_HEADROOM}"
        )
