# Required libraries
import numpy as np
import re
import mendeleev
import h5py
import scipy.interpolate as interpolate
from .SpecWizard_Elements import Elements

elements  = [('Hydrogen',mendeleev.H.atomic_weight,'H0+'), ('Helium',mendeleev.He.atomic_weight,'He1+')]
iondir    = './HM12Tables'

ElementNames = ['Hydrogen', 'Helium', 'Oxygen', 'Carbon', 'Silicon']
atomfile     = "/cosma/home/dphlss/tt/atom.dat"       # ascii file containing physical parameters of transitions, from VpFit
elements     = Elements()

class IonizationBalance:
    """
    Computes the fractional abundance of a given ion using interpolation tables generated by Cloudy,
    assuming photo-ionization equilibrium.
    """    
    def __init__(self, iondir=iondir):
        """
        Initializes the IonizationBalance class.

        Args:
            iondir (str): Directory containing Cloudy interpolation tables.
        """

        self.iondir    = iondir  # directory that contains cloudy interpolation tables
        
    def Info(self):
        """
        Provides information about the class functionality.

        Prints:
            A description of the purpose and functionality of the IonizationBalance class.
        """
                #
        print("This class computes the fractional abundance of a given ion, by interpolating\
               a table genetrated by Cloudy - assuming photo-ionization equilibrium")

 
    def IonAbundance(self, redshift=0.0, nH_density=1.0, temperature=1e4, ionname = 'H I'):
        """
        Computes the fractional abundance of a specified ion.

        Args:
            redshift (float): Current redshift.
            nH_density (float): Proper hydrogen density in particles/cm^3.
            temperature (float): Gas temperature in Kelvin.
            ionname (str): Name of the ion (e.g., 'H I' for neutral hydrogen).

        Returns:
            float: Logarithm (base 10) of the fractional abundance, n_ion/n_element.
            -1: If the interpolation table cannot be read.

        Raises:
            ValueError: If interpolation fails due to missing or out-of-bound values.
        """
        # OPen the file for interpolation (Create this file using the Cloudy notebook)
        fname = self.iondir + '/' + ionname + '.hdf5'       

        # read the table
        try:
            (table_z, table_LogTs, table_LognHs), table = self.ReadIonTable(fname)
        except:
            print("error: could not read table {} ".format(fname))
            return -1
        
        TInterpol  = np.log10(temperature)
        nHInterpol = np.log10(nH_density)
        zInterpol  = redshift
        pts        = np.column_stack((zInterpol, TInterpol, nHInterpol))
        result     = interpolate.interpn((table_z, table_LogTs, table_LognHs), table, pts, method='linear', bounds_error=False, fill_value=None)
        return result
    
    def ReadIonTable(self, fname="./HM12Tables/H I.hdf5"):
        """
        Reads the HDF5 file containing Cloudy-generated ionization tables.

        Args:
            fname (str): Path to the HDF5 file.

        Returns:
            tuple: A tuple containing:
                - (np.ndarray) z: Tabulated redshift values.
                - (np.ndarray) LogT: Tabulated log temperature values (in K).
                - (np.ndarray) LognH: Tabulated log hydrogen number densities (in particles/cm^3).
                - (np.ndarray) LogAbundance: Logarithmic abundance of the ion relative to the element.

        Raises:
            FileNotFoundError: If the specified file cannot be found.
            IOError: If there is an error reading the HDF5 file.
        """
        hf  = h5py.File(fname,  "r")
        # cloudy parameters
        UVB   = hf.attrs.get("UVB")
        compo = hf.attrs.get("Composition")
        # tables
        z            = hf['z'][...]
        LogT         = hf['LogT'][...]
        LognH        = hf['LognH'][...]
        LogAbundance = hf['LogAbundance'][...]
        #
        hf.close()
        #
        return ((z, LogT, LognH), LogAbundance)    

    