# Methods for ensuring data reaches a remote host in its entirity
# stdlib
from copy import deepcopy
import ipaddress
import socket
from collections import deque
from typing import Any, Deque, Dict, Tuple, Union
# libs
from paramiko import AutoAddPolicy, SSHClient, SSHException
# local
from cloudcix.rcc.channel_codes import CHANNEL_SUCCESS, CONNECTION_ERROR, VALIDATION_ERROR
from cloudcix.rcc.response import RESPONSE_DICT


IPAddress = Union[ipaddress.IPv4Address, ipaddress.IPv6Address]


def comms_ssh(host_ip: str, payload: str, username: str = 'administrator', timeout: int = 4) -> Dict[str, Any]:
    """
    Run a command on a remote host over an SSH connection
    Raises `CouldNotConnectException` if an SSH connection cannot be established
    :param host_ip: The IP address where the command should be run
    :param payload: The shell commands to run on the remote host
    :param username: The user that will run the commands on the remote host
    :param timeout: How long the client can attempt to connect to the remote host
    :return: A dictionary with the following properties:
        channel_code:
            description: A code representing if suuccessful connection was made.
        channel_error:
            description: Optional string value representing an error raised validating the sent params.
        channel_message:
            description: String message representing if connection to host was successful.
        payload_code:
            description: Optional integer value representing the Linux exit code of the sent payload.
        payload_error:
            description: Optional string value representing  the standard error generated by the payload.
        payload_message:
            description: Optional string value representing the standard output of the payload.
    """
    response = deepcopy(RESPONSE_DICT)

    try:
        ip = ipaddress.ip_address(host_ip)
    except ValueError as e:
        response['channel_code'] = VALIDATION_ERROR
        response['channel_message'] = f'Could not parse sent `host_ip` value {host_ip}'
        response['channel_error'] = str(e)
        return response

    client, err = get_client(ip, username, timeout)
    if err is not None:
        response['channel_code'] = CONNECTION_ERROR
        response['channel_message'] = f'Could not eastablish a SSH connection to {host_ip} for username {username}.'
        response['channel_error'] = err
        return response

    response['channel_code'] = CHANNEL_SUCCESS
    response['channel_message'] = f'Connection established to IP {host_ip}'

    exit_code, out, err = _deploy_paramiko(client, payload)
    client.close()

    response['payload_code'] = exit_code
    response['payload_message'] = out
    response['payload_error'] = err

    return response


def get_client(ip: IPAddress, username: str, timeout: int) -> SSHClient:
    """
    Obtain a paramiko.SSHClient connected to the given `ip`
    :param ip: An IPAddress to connect to
    :param username: The remote user to login as
    :param timeout: How long the client can attempt to connect to the remote host
    :return: An ssh client
    """
    client = SSHClient()
    client.set_missing_host_key_policy(AutoAddPolicy())
    if 6 == ip.version:
        proto = socket.AF_INET6
    else:
        proto = socket.AF_INET
    try:
        sock = socket.socket(proto, socket.SOCK_STREAM)
        # Try connecting to the host using SSH keys
        sock.settimeout(timeout)
        sock.connect((str(ip), 22))
        client.connect(
            hostname=str(ip),
            username=username,
            timeout=timeout,
            sock=sock,
        )
    except (SSHException, OSError) as e:
        return '', str(e)
    except Exception as e:
        return '', f'An unknown exception occurred: {e}'

    return client, None


def _deploy_paramiko(client: SSHClient, payload: str) -> Tuple[int, str, str]:
    """
    Deploy the given `payload` to the SSH host accessible via the supplied `client`
    :param client: A paramiko.Client instance that is connected to the host
    :param payload: The command(s) to run on the host
    :return: The Linux exit code, stdout and stderr of running the payload on the remote host
    """

    _, stdout, stderr = client.exec_command(payload)

    # Read from stdout and stderr until the connection is closed
    channel = stdout.channel
    output_fragments: Deque[str] = deque()
    error_fragments: Deque[str] = deque()
    while True:
        # Check the status of the connection, and read any remaining output
        closed = channel.closed
        while channel.recv_ready():
            output_fragments.append(stdout.read().decode())  # Read the entirety of stdout

        while channel.recv_stderr_ready():
            error_fragments.append(stderr.read().decode())  # Read the entirety of stderr

        if closed:
            exit_code = channel.recv_exit_status()
            break

    output = ''.join(output_fragments)
    error = ''.join(error_fragments)

    return exit_code, output, error
