import dataclasses
from dataclasses import field
from tkinter import Widget
from tkinter.ttk import Button
from typing import Type, Optional, Literal, List

from pyguiadapterlite._messages import messages as msgs
from pyguiadapterlite.components.dialog import StringInputDialog
from pyguiadapterlite.types.lists._common import (
    BaseStringListValue,
    BaseStringListValueWidget,
    BaseStringListBox,
)
from pyguiadapterlite.utils import show_warning


@dataclasses.dataclass(frozen=True)
class StringListValue(BaseStringListValue):

    add_button: bool = True
    """是否显示添加按钮"""

    add_button_text: str = field(default_factory=lambda: msgs().MSG_ADD_BUTTON_TEXT)
    """添加按钮文本"""

    add_method: Literal["append", "prepend"] = "append"
    """添加方法，append表示在列表尾部添加，prepend表示在列表头部添加"""

    strip: bool = False
    """是否去除输入字符串两端的空格"""

    accept_empty: bool = True
    """是否接受空字符串"""

    empty_string_message: str = field(
        default_factory=lambda: msgs().MSG_EMPTY_STRING_WARNING
    )
    """空字符串警告信息"""

    accept_duplicates: bool = True
    """是否接受重复项"""

    duplicate_message: str = field(
        default_factory=lambda: msgs().MSG_DUPLICATE_ITEMS_WARNING
    )
    """重复项警告信息"""

    multi_selection_message: str = field(
        default_factory=lambda: msgs().MSG_MULTIPLE_SELECTION_WARNING
    )
    """多选警告信息"""

    add_item_dialog_title: str = field(
        default_factory=lambda: msgs().MSG_ADD_ITEM_DIALOG_TITLE
    )
    """添加项对话框标题"""

    add_item_dialog_label_text: str = field(
        default_factory=lambda: msgs().MSG_ADD_ITEM_DIALOG_LABEL_TEXT
    )
    """添加项对话框标签文本"""

    edit_item_dialog_title: str = field(
        default_factory=lambda: msgs().MSG_EDIT_ITEM_DIALOG_TITLE
    )
    """编辑项对话框标题"""

    edit_item_dialog_label_text: str = field(
        default_factory=lambda: msgs().MSG_EDIT_ITEM_DIALOG_LABEL_TEXT
    )
    """编辑项对话框标签文本"""

    @classmethod
    def target_widget_class(cls) -> Type["StringListValueWidget"]:
        return StringListValueWidget


class StringListBox(BaseStringListBox):
    def __init__(self, parent: "StringListValueWidget", **kwargs):
        self._add_button: Optional[Button] = None
        self._config: StringListValue = parent.config

        super().__init__(parent, **kwargs)

    def on_edit(self, indexes: List[int]):
        if not indexes:
            show_warning(
                self._config.no_selection_message, parent=self.winfo_toplevel()
            )
            return
        if len(indexes) > 1:
            show_warning(
                self._config.multi_selection_message, parent=self.winfo_toplevel()
            )
            return
        index = indexes[0]
        current_value = self._listview.real.get(index)
        dialog = StringInputDialog(
            self,
            title=self._config.edit_item_dialog_title,
            label_text=self._config.edit_item_dialog_label_text,
            initial_value=str(current_value),
        )
        if dialog.is_cancelled():
            return
        new_value = dialog.result
        if new_value == current_value:
            return
        new_value = self._process_input(new_value)
        if new_value is None:
            return
        self._listview.real.set(index, new_value)

    def _process_input(self, input_str: str) -> Optional[str]:
        value = str(input_str)
        if self._config.strip:
            value = value.strip()

        if not self._config.accept_empty and value.strip() == "":
            show_warning(
                self._config.empty_string_message, parent=self.winfo_toplevel()
            )
            return None

        if not self._config.accept_duplicates and self._listview.real.contains(value):
            show_warning(self._config.duplicate_message, parent=self.winfo_toplevel())
            return None

        return value

    def _on_add_item(self):
        input_dialog = StringInputDialog(
            self,
            title=self._config.add_item_dialog_title,
            label_text=self._config.add_item_dialog_label_text,
        )
        if input_dialog.is_cancelled():
            return
        value = input_dialog.result
        if value is None:
            return

        value = self._process_input(value)
        if value is None:
            return

        if self._config.add_method == "append":
            self._listview.real.append(value)
        else:
            self._listview.real.prepend(value)

    def _create_buttons(self):
        super()._create_buttons()
        if self._config.add_button:
            self._add_button = Button(
                self.buttons_area,
                text=self._config.add_button_text,
                command=self._on_add_item,
            )
            self._add_button.pack(side="right", padx=2, pady=2)


class StringListValueWidget(BaseStringListValueWidget):
    ConfigClass = StringListValue

    def __init__(
        self,
        parent: Widget,
        parameter_name: str,
        config: StringListValue,
    ):
        super().__init__(StringListBox, parent, parameter_name, config)

    @property
    def config(self) -> StringListValue:
        return self._config
