import dataclasses
from pathlib import Path
from tkinter import Menu as TkMenu, BooleanVar, messagebox, filedialog
from tkinter import Tk, Toplevel
from typing import Tuple, Optional, Any, Union, List, Dict, Type

from pyguiadapterlite._messages import messages as msgs
from pyguiadapterlite.components.dialog import BaseDialog
from pyguiadapterlite.components.menus import Menu, Separator, Action
from pyguiadapterlite.utils import _warning, _exception, _error


class ParameterGroupNameNotAvailableError(Exception):
    pass


@dataclasses.dataclass(frozen=True)
class BaseWindowConfig(object):
    title: str = ""
    """窗口标题"""

    icon: Optional[str] = None
    """窗口图标路径，ico格式"""

    size: Tuple[int, int] = (800, 605)
    """窗口大小"""

    position: Tuple[Optional[int], Optional[int]] = (None, None)
    """窗口位置"""

    always_on_top: bool = False
    """窗口是否置顶"""

    menus: Optional[List[Union[Menu, Separator]]] = dataclasses.field(
        default_factory=list
    )
    """窗口菜单"""


class BaseWindow(object):
    def __init__(self, parent: Union[Tk, Toplevel], config: BaseWindowConfig, **kwargs):
        self._parent = parent
        self._config = config
        size = config.size
        position = config.position
        self._parent.geometry(
            f"{size[0]}x{size[1]}{f'+{position[0]}+{position[1]}' if position[0] and position[1] else ''}"
        )
        self._parent.title(config.title)
        icon = config.icon
        if icon:
            icon = Path(icon)
            if not icon.is_file():
                _warning(f"icon file `{icon}` not found, using default icon.")
            else:
                self._parent.iconbitmap(icon.as_posix())

        self._menu_bar: Optional[TkMenu] = None
        self._exclusive_groups: Dict[int, List[Action]] = {}

        self.create_main_area()
        self.create_left_area()
        self.create_right_area()
        self.create_bottom_area()
        self.create_main_menu()
        self.create_status_bar()

        self._parent.protocol("WM_DELETE_WINDOW", self.on_close)
        self.set_always_on_top(config.always_on_top)

    @property
    def parent(self) -> Union[Tk, Toplevel]:
        return self._parent

    @property
    def config(self) -> BaseWindowConfig:
        return self._config

    @property
    def menus(self) -> List[Menu]:
        if not self._config.menus:
            return []
        return [menu for menu in self._config.menus if isinstance(menu, Menu)]

    def create_main_area(self) -> Any:
        pass

    def create_bottom_area(self) -> Any:
        pass

    def create_left_area(self) -> Any:
        pass

    def create_right_area(self) -> Any:
        pass

    def create_status_bar(self):
        pass

    def create_main_menu(self) -> None:
        if not self._config.menus:
            return
        # 创建主菜单栏
        menu_bar = TkMenu(self._parent)
        # 遍历菜单配置并创建菜单
        menus = self._config.menus or []
        for menu_item in menus:
            if isinstance(menu_item, Separator):
                # 在菜单栏中添加分隔符（如果有需要的话）
                # 注意：在顶层菜单栏中通常不直接添加分隔符，所以这里跳过
                continue
            elif isinstance(menu_item, Menu):
                # 创建子菜单
                submenu = self._create_submenu(menu_bar, menu_item)
                # 将子菜单添加到菜单栏
                menu_bar.add_cascade(label=menu_item.title, menu=submenu)
        # 将菜单栏设置到窗口
        self._parent.config(menu=menu_bar)
        self._menu_bar = menu_bar

    def _create_submenu(self, parent_menu: TkMenu, menu_config: Menu) -> TkMenu:
        """
        创建子菜单（递归处理嵌套菜单）
        """
        # 创建菜单，设置 tearoff 属性
        menu = TkMenu(parent_menu, tearoff=1 if menu_config.tear_off_enabled else 0)
        for i, action in enumerate(menu_config.actions):
            if isinstance(action, Separator):
                menu.add_separator()
            elif isinstance(action, Menu):
                # 递归创建子菜单
                submenu = self._create_submenu(menu, action)
                menu.add_cascade(label=action.title, menu=submenu)
            elif isinstance(action, Action):
                self._add_action(action, menu_config, menu)
            else:
                raise ValueError(f"unknown menu item type: {type(action)}")
        return menu

    def _add_action(self, action: Action, menu_config: Menu, menu: TkMenu):
        # 添加动作菜单项
        # 创建菜单项命令
        command = lambda action_=action: self._handle_action_triggered(action_)
        # 根据动作类型创建不同的菜单项
        if menu_config.exclusive and action.checkable:
            # 排他复选框
            self._create_checkable_menu(menu, action, command=command)
            exclusive_group_id = id(menu_config)
            exclusive_group = self._exclusive_groups.get(exclusive_group_id, None)
            if exclusive_group is None:
                self._exclusive_groups[exclusive_group_id] = [action]
            else:
                exclusive_group.append(action)
            action.add_to_exclusive_group(exclusive_group_id)
            self._handle_exclusive_group_changed(action)
        elif action.checkable:
            # 普通复选框
            self._create_checkable_menu(menu, action, command=command)
        else:
            # 普通菜单项
            menu.add_command(label=action.text, command=command)
        # 设置快捷键
        if action.shortcut:
            shortcut = action.shortcut.strip().replace("Ctrl", "Control")
            # 绑定快捷键到窗口
            self._bind_shortcut(shortcut, command)
        # 设置启用状态
        if not action.enabled:
            menu.entryconfig(menu.index("end"), state="disabled")

    @staticmethod
    def _create_checkable_menu(menu: TkMenu, item: Action, command):
        # 回退到普通复选框
        action_checked_var = BooleanVar()
        action_checked_var.set(item.initial_checked)
        menu.add_checkbutton(
            label=item.text,
            command=command,
            variable=action_checked_var,
        )
        item.bind_checked_var(action_checked_var)

    def _handle_action_triggered(self, action: Action) -> None:
        """
        处理动作触发
        """
        if not action.on_triggered:
            return
        try:
            action.on_triggered(self, action)
            self._handle_exclusive_group_changed(action)
        except Exception as e:
            _exception(e, f"error executing action '{action.text}'")

    def _handle_exclusive_group_changed(self, action: Action) -> None:
        if not action.checkable:
            return
        exclusive_group_id = action.get_exclusive_group_id()
        if exclusive_group_id is None:
            return
        exclusive_group = self._exclusive_groups.get(exclusive_group_id, None)
        if exclusive_group is None:
            _error(
                f"exclusive group with id '{exclusive_group_id}'  not found for action '{action.text}'"
            )
            return
        action_checked = action.is_checked()
        for other_action in exclusive_group:
            if other_action is action:
                continue
            if action_checked:
                other_action.set_checked(False)

    def _bind_shortcut(self, shortcut: str, command) -> None:
        """
        绑定快捷键
        """
        try:
            # 简单的快捷键绑定实现
            # 实际实现可能需要更复杂的快捷键解析
            self._parent.bind(f"<{shortcut}>", lambda event: command())
        except Exception as e:
            _exception(e, f"failed to bind shortcut '{shortcut}'")

    def move_to_center(self):
        """
        将窗口居中显示（自动获取窗口尺寸）
        """
        self.hide()

        # 更新窗口以确保获取正确的尺寸
        self._parent.update()

        # 获取窗口尺寸
        width = self._parent.winfo_width()
        height = self._parent.winfo_height()

        # 如果尺寸无效，使用请求尺寸
        if width <= 1 or height <= 1:
            width = self._parent.winfo_reqwidth()
            height = self._parent.winfo_reqheight()

        # 根据目标控件的位置设置提示框位置（对DPI缩放进行了适配）
        if hasattr(self._parent, "DPI_scaling"):
            scaling = self._parent.DPI_scaling
        else:
            scaling = 1.0

        width = int(width / scaling)
        height = int(height / scaling)

        # 使用Tkinter内置的窗口放置功能
        x = (self._parent.winfo_screenwidth() // 2) - (width // 2)
        y = (self._parent.winfo_screenheight() // 2) - (height // 2)

        # 直接设置几何位置
        self._parent.geometry(f"{width}x{height}+{x}+{y}")

        self.show()

    def hide(self):
        self._parent.withdraw()

    def show(self):
        self._parent.deiconify()

    def set_always_on_top(self, on: bool):
        self._parent.wm_attributes("-topmost", on)

    def close(self):
        self.on_close()

    def on_close(self):
        if self._parent:
            self._parent.destroy()
            self._parent = None

    def show_custom_dialog(
        self, dialog_class: Type[BaseDialog], title: str, **dialog_kwargs
    ) -> Any:
        dialog = dialog_class(title=title, parent=self.parent, **dialog_kwargs)
        if dialog.is_cancelled():
            return None
        return dialog.result

    def show_sub_window(
        self,
        window_class: Type["BaseWindow"],
        config: Optional[BaseWindowConfig],
        modal: bool = False,
        **kwargs,
    ):
        sub_window_toplevel = Toplevel(self.parent)
        window = window_class(parent=sub_window_toplevel, config=config, **kwargs)
        if modal:
            window.parent.grab_set()
            self._parent.wait_window(window._parent)
            return

    def show_information(self, message: str, title: Optional[str] = None, **options):
        title = title or msgs().MSG_INFO_TITLE
        options = options or {}
        options["parent"] = self.parent
        messagebox.showinfo(title=title, message=message, **options)

    def show_warning(self, message: str, title: Optional[str] = None, **options):
        title = title or msgs().MSG_WARNING_TITLE
        options = options or {}
        options["parent"] = self.parent
        messagebox.showwarning(title=title, message=message, **options)

    def show_error(self, message: str, title: Optional[str] = None, **options):
        title = title or msgs().MSG_ERROR_TITLE
        options = options or {}
        options["parent"] = self.parent
        messagebox.showerror(title=title, message=message, **options)

    def show_question(
        self, message: str, title: Optional[str] = None, **options
    ) -> str:
        title = title or msgs().MSG_QUESTION_TITLE
        options = options or {}
        options["parent"] = self.parent
        return messagebox.askquestion(title=title, message=message, **options)

    def ask_yes_no(self, message: str, title: Optional[str] = None, **options) -> bool:
        title = title or msgs().MSG_QUESTION_TITLE
        options = options or {}
        options["parent"] = self.parent
        return messagebox.askyesno(title=title, message=message, **options)

    def ask_ok_cancel(
        self, message: str, title: Optional[str] = None, **options
    ) -> bool:
        title = title or msgs().MSG_QUESTION_TITLE
        options = options or {}
        options["parent"] = self.parent
        return messagebox.askokcancel(title=title, message=message, **options)

    def ask_retry_cancel(
        self, message: str, title: Optional[str] = None, **options
    ) -> bool:
        title = title or msgs().MSG_QUESTION_TITLE
        options = options or {}
        options["parent"] = self.parent
        return messagebox.askretrycancel(title=title, message=message, **options)

    def ask_yes_no_cancel(
        self, message: str, title: Optional[str] = None, **options
    ) -> Optional[str]:
        title = title or msgs().MSG_QUESTION_TITLE
        options = options or {}
        options["parent"] = self.parent
        return messagebox.askyesnocancel(title=title, message=message, **options)

    def select_open_file(
        self,
        title: Optional[str] = None,
        filetypes: Optional[list] = None,
        initialdir: Optional[str] = None,
        initialfile: Optional[str] = None,
        defaultextension: Optional[str] = None,
    ) -> Optional[str]:
        msgs_ = msgs()
        title = title or msgs_.MSG_OPEN_FILE_DIALOG_TITLE
        filetypes = filetypes or [(msgs_.MSG_FILE_FILTER_ALL, "*.*")]
        path = filedialog.askopenfilename(
            parent=self.parent,
            title=title,
            filetypes=filetypes,
            initialdir=initialdir,
            initialfile=initialfile,
            defaultextension=defaultextension,
        )
        if not path:
            return None
        return path

    def select_save_file(
        self,
        title: Optional[str] = None,
        filetypes: Optional[list] = None,
        initialdir: Optional[str] = None,
        initialfile: Optional[str] = None,
        defaultextension: Optional[str] = None,
        confirmoverwrite: bool = True,
    ) -> Optional[str]:
        msgs_ = msgs()
        title = title or msgs_.MSG_SAVE_FILE_DIALOG_TITLE
        filetypes = filetypes or [(msgs_.MSG_FILE_FILTER_ALL, "*.*")]
        path = filedialog.asksaveasfilename(
            parent=self.parent,
            title=title,
            confirmoverwrite=confirmoverwrite,
            filetypes=filetypes,
            initialdir=initialdir,
            initialfile=initialfile,
            defaultextension=defaultextension,
        )
        if not path:
            return None
        return path

    def select_open_files(
        self,
        title: Optional[str] = None,
        filetypes: List[str] = None,
        initialdir: Optional[str] = None,
        initialfile: Optional[str] = None,
        defaultextension: Optional[str] = None,
    ) -> Tuple[str, ...]:
        msgs_ = msgs()
        title = title or msgs_.MSG_OPEN_FILE_DIALOG_TITLE
        filetypes = filetypes or [(msgs_.MSG_FILE_FILTER_ALL, "*.*")]

        paths = filedialog.askopenfilenames(
            parent=self.parent,
            title=title,
            filetypes=filetypes,
            initialdir=initialdir,
            initialfile=initialfile,
            defaultextension=defaultextension,
        )
        if not paths:
            return ()
        return paths

    def select_directory(
        self,
        title: Optional[str] = None,
        initialdir: Optional[str] = None,
        mustexist: Optional[bool] = None,
    ) -> Optional[str]:
        title = title or msgs().MSG_SELECT_DIR_DIALOG_TITLE
        path = filedialog.askdirectory(
            parent=self.parent, title=title, initialdir=initialdir, mustexist=mustexist
        )
        if not path:
            return None
        return path
