"use strict";
var _a, _b, _c, _d;
Object.defineProperty(exports, "__esModule", { value: true });
exports.FieldUtils = exports.Context = exports.Data = exports.JsonPath = void 0;
const jsiiDeprecationWarnings = require("../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const core_1 = require("@aws-cdk/core");
const json_path_1 = require("./private/json-path");
/**
 * Extract a field from the State Machine data or context
 * that gets passed around between states
 *
 * @see https://docs.aws.amazon.com/step-functions/latest/dg/amazon-states-language-paths.html
 */
class JsonPath {
    constructor() { }
    /**
     * Instead of using a literal string, get the value from a JSON path
     */
    static stringAt(path) {
        validateJsonPath(path);
        return new json_path_1.JsonPathToken(path).toString();
    }
    /**
     * Instead of using a literal string list, get the value from a JSON path
     */
    static listAt(path) {
        // does not apply to task context
        validateDataPath(path);
        return core_1.Token.asList(new json_path_1.JsonPathToken(path));
    }
    /**
     * Instead of using a literal number, get the value from a JSON path
     */
    static numberAt(path) {
        validateJsonPath(path);
        return core_1.Token.asNumber(new json_path_1.JsonPathToken(path));
    }
    /**
     * Reference a complete (complex) object in a JSON path location
     */
    static objectAt(path) {
        validateJsonPath(path);
        return new json_path_1.JsonPathToken(path);
    }
    /**
     * Use the entire data structure
     *
     * Will be an object at invocation time, but is represented in the CDK
     * application as a string.
     */
    static get entirePayload() {
        return new json_path_1.JsonPathToken('$').toString();
    }
    /**
     * Determines if the indicated string is an encoded JSON path
     *
     * @param value string to be evaluated
     */
    static isEncodedJsonPath(value) {
        return !!json_path_1.jsonPathString(value);
    }
    /**
     * Return the Task Token field
     *
     * External actions will need this token to report step completion
     * back to StepFunctions using the `SendTaskSuccess` or `SendTaskFailure`
     * calls.
     */
    static get taskToken() {
        return new json_path_1.JsonPathToken('$$.Task.Token').toString();
    }
    /**
     * Use the entire context data structure
     *
     * Will be an object at invocation time, but is represented in the CDK
     * application as a string.
     */
    static get entireContext() {
        return new json_path_1.JsonPathToken('$$').toString();
    }
    /**
     * Make an intrinsic States.Array expression
     *
     * Combine any number of string literals or JsonPath expressions into an array.
     *
     * Use this function if the value of an array element directly has to come
     * from a JSON Path expression (either the State object or the Context object).
     *
     * If the array contains object literals whose values come from a JSON path
     * expression, you do not need to use this function.
     *
     * @see https://docs.aws.amazon.com/step-functions/latest/dg/amazon-states-language-intrinsic-functions.html
     */
    static array(...values) {
        return new json_path_1.JsonPathToken(`States.Array(${values.map(json_path_1.renderInExpression).join(', ')})`).toString();
    }
    /**
     * Make an intrinsic States.Format expression
     *
     * This can be used to embed JSON Path variables inside a format string.
     *
     * For example:
     *
     * ```ts
     * sfn.JsonPath.format('Hello, my name is {}.', sfn.JsonPath.stringAt('$.name'))
     * ```
     *
     * @see https://docs.aws.amazon.com/step-functions/latest/dg/amazon-states-language-intrinsic-functions.html
     */
    static format(formatString, ...values) {
        const allArgs = [formatString, ...values];
        return new json_path_1.JsonPathToken(`States.Format(${allArgs.map(json_path_1.renderInExpression).join(', ')})`).toString();
    }
    /**
     * Make an intrinsic States.StringToJson expression
     *
     * During the execution of the Step Functions state machine, parse the given
     * argument as JSON into its object form.
     *
     * For example:
     *
     * ```ts
     * sfn.JsonPath.stringToJson(sfn.JsonPath.stringAt('$.someJsonBody'))
     * ```
     *
     * @see https://docs.aws.amazon.com/step-functions/latest/dg/amazon-states-language-intrinsic-functions.html
     */
    static stringToJson(jsonString) {
        return new json_path_1.JsonPathToken(`States.StringToJson(${json_path_1.renderInExpression(jsonString)})`);
    }
    /**
     * Make an intrinsic States.JsonToString expression
     *
     * During the execution of the Step Functions state machine, encode the
     * given object into a JSON string.
     *
     * For example:
     *
     * ```ts
     * sfn.JsonPath.jsonToString(sfn.JsonPath.objectAt('$.someObject'))
     * ```
     *
     * @see https://docs.aws.amazon.com/step-functions/latest/dg/amazon-states-language-intrinsic-functions.html
     */
    static jsonToString(value) {
        const path = json_path_1.jsonPathFromAny(value);
        if (!path) {
            throw new Error('Argument to JsonPath.jsonToString() must be a JsonPath object');
        }
        return new json_path_1.JsonPathToken(`States.JsonToString(${path})`).toString();
    }
}
exports.JsonPath = JsonPath;
_a = JSII_RTTI_SYMBOL_1;
JsonPath[_a] = { fqn: "@aws-cdk/aws-stepfunctions.JsonPath", version: "1.202.0" };
/**
 * Special string value to discard state input, output or result
 */
JsonPath.DISCARD = 'DISCARD';
/**
 * Extract a field from the State Machine data that gets passed around between states
 *
 * @deprecated replaced by `JsonPath`
 */
class Data {
    constructor() { }
    /**
     * Instead of using a literal string, get the value from a JSON path
     */
    static stringAt(path) {
        try {
            jsiiDeprecationWarnings.print("@aws-cdk/aws-stepfunctions.Data#stringAt", "replaced by `JsonPath`");
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.stringAt);
            }
            throw error;
        }
        validateDataPath(path);
        return new json_path_1.JsonPathToken(path).toString();
    }
    /**
     * Instead of using a literal string list, get the value from a JSON path
     */
    static listAt(path) {
        try {
            jsiiDeprecationWarnings.print("@aws-cdk/aws-stepfunctions.Data#listAt", "replaced by `JsonPath`");
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.listAt);
            }
            throw error;
        }
        validateDataPath(path);
        return core_1.Token.asList(new json_path_1.JsonPathToken(path));
    }
    /**
     * Instead of using a literal number, get the value from a JSON path
     */
    static numberAt(path) {
        try {
            jsiiDeprecationWarnings.print("@aws-cdk/aws-stepfunctions.Data#numberAt", "replaced by `JsonPath`");
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.numberAt);
            }
            throw error;
        }
        validateDataPath(path);
        return core_1.Token.asNumber(new json_path_1.JsonPathToken(path));
    }
    /**
     * Use the entire data structure
     *
     * Will be an object at invocation time, but is represented in the CDK
     * application as a string.
     */
    static get entirePayload() {
        try {
            jsiiDeprecationWarnings.print("@aws-cdk/aws-stepfunctions.Data#entirePayload", "replaced by `JsonPath`");
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, jsiiDeprecationWarnings.getPropertyDescriptor(this, "entirePayload").get);
            }
            throw error;
        }
        return new json_path_1.JsonPathToken('$').toString();
    }
    /**
     * Determines if the indicated string is an encoded JSON path
     *
     * @param value string to be evaluated
     */
    static isJsonPathString(value) {
        try {
            jsiiDeprecationWarnings.print("@aws-cdk/aws-stepfunctions.Data#isJsonPathString", "replaced by `JsonPath`");
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.isJsonPathString);
            }
            throw error;
        }
        return !!json_path_1.jsonPathString(value);
    }
}
exports.Data = Data;
_b = JSII_RTTI_SYMBOL_1;
Data[_b] = { fqn: "@aws-cdk/aws-stepfunctions.Data", version: "1.202.0" };
/**
 * Extract a field from the State Machine Context data
 *
 * @see https://docs.aws.amazon.com/step-functions/latest/dg/connect-to-resource.html#wait-token-contextobject
 *
 * @deprecated replaced by `JsonPath`
 */
class Context {
    constructor() { }
    /**
     * Instead of using a literal string, get the value from a JSON path
     */
    static stringAt(path) {
        try {
            jsiiDeprecationWarnings.print("@aws-cdk/aws-stepfunctions.Context#stringAt", "replaced by `JsonPath`");
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.stringAt);
            }
            throw error;
        }
        validateContextPath(path);
        return new json_path_1.JsonPathToken(path).toString();
    }
    /**
     * Instead of using a literal number, get the value from a JSON path
     */
    static numberAt(path) {
        try {
            jsiiDeprecationWarnings.print("@aws-cdk/aws-stepfunctions.Context#numberAt", "replaced by `JsonPath`");
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.numberAt);
            }
            throw error;
        }
        validateContextPath(path);
        return core_1.Token.asNumber(new json_path_1.JsonPathToken(path));
    }
    /**
     * Return the Task Token field
     *
     * External actions will need this token to report step completion
     * back to StepFunctions using the `SendTaskSuccess` or `SendTaskFailure`
     * calls.
     */
    static get taskToken() {
        try {
            jsiiDeprecationWarnings.print("@aws-cdk/aws-stepfunctions.Context#taskToken", "replaced by `JsonPath`");
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, jsiiDeprecationWarnings.getPropertyDescriptor(this, "taskToken").get);
            }
            throw error;
        }
        return new json_path_1.JsonPathToken('$$.Task.Token').toString();
    }
    /**
     * Use the entire context data structure
     *
     * Will be an object at invocation time, but is represented in the CDK
     * application as a string.
     */
    static get entireContext() {
        try {
            jsiiDeprecationWarnings.print("@aws-cdk/aws-stepfunctions.Context#entireContext", "replaced by `JsonPath`");
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, jsiiDeprecationWarnings.getPropertyDescriptor(this, "entireContext").get);
            }
            throw error;
        }
        return new json_path_1.JsonPathToken('$$').toString();
    }
}
exports.Context = Context;
_c = JSII_RTTI_SYMBOL_1;
Context[_c] = { fqn: "@aws-cdk/aws-stepfunctions.Context", version: "1.202.0" };
/**
 * Helper functions to work with structures containing fields
 */
class FieldUtils {
    constructor() { }
    /**
     * Render a JSON structure containing fields to the right StepFunctions structure
     */
    static renderObject(obj) {
        return json_path_1.renderObject(obj);
    }
    /**
     * Return all JSON paths used in the given structure
     */
    static findReferencedPaths(obj) {
        return Array.from(json_path_1.findReferencedPaths(obj)).sort();
    }
    /**
     * Returns whether the given task structure contains the TaskToken field anywhere
     *
     * The field is considered included if the field itself or one of its containing
     * fields occurs anywhere in the payload.
     */
    static containsTaskToken(obj) {
        const paths = json_path_1.findReferencedPaths(obj);
        return paths.has('$$.Task.Token') || paths.has('$$.Task') || paths.has('$$');
    }
}
exports.FieldUtils = FieldUtils;
_d = JSII_RTTI_SYMBOL_1;
FieldUtils[_d] = { fqn: "@aws-cdk/aws-stepfunctions.FieldUtils", version: "1.202.0" };
function validateJsonPath(path) {
    if (path !== '$'
        && !path.startsWith('$.')
        && path !== '$$'
        && !path.startsWith('$$.')
        && !path.startsWith('$[')
        && ['Format', 'StringToJson', 'JsonToString', 'Array'].every(fn => !path.startsWith(`States.${fn}`))) {
        throw new Error(`JSON path values must be exactly '$', '$$', start with '$.', start with '$$.', start with '$[', or start with an intrinsic function: States.Format, States.StringToJson, States.JsonToString, or States.Array. Received: ${path}`);
    }
}
function validateDataPath(path) {
    if (path !== '$' && !path.startsWith('$.')) {
        throw new Error("Data JSON path values must either be exactly equal to '$' or start with '$.'");
    }
}
function validateContextPath(path) {
    if (path !== '$$' && !path.startsWith('$$.')) {
        throw new Error("Context JSON path values must either be exactly equal to '$$' or start with '$$.'");
    }
}
//# sourceMappingURL=data:application/json;base64,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