# -*- coding: utf-8 -*-

# PLEASE DO NOT EDIT THIS FILE, IT IS GENERATED AND WILL BE OVERWRITTEN:
# https://github.com/ccxt/ccxt/blob/master/CONTRIBUTING.md#how-to-contribute-code

from ccxt.async_support.base.exchange import Exchange
from ccxt.abstract.arkham import ImplicitAPI
import hashlib
from ccxt.base.types import Account, Any, Balances, Currencies, Currency, DepositAddress, Int, Leverage, LeverageTier, LeverageTiers, Market, Num, Order, OrderBook, OrderSide, OrderType, Position, Str, Strings, Ticker, Tickers, Trade, TradingFees, Transaction
from typing import List
from ccxt.base.errors import ExchangeError
from ccxt.base.errors import AuthenticationError
from ccxt.base.errors import PermissionDenied
from ccxt.base.errors import ArgumentsRequired
from ccxt.base.errors import BadRequest
from ccxt.base.errors import BadSymbol
from ccxt.base.errors import OperationRejected
from ccxt.base.errors import InsufficientFunds
from ccxt.base.errors import InvalidAddress
from ccxt.base.errors import InvalidOrder
from ccxt.base.errors import OrderNotFound
from ccxt.base.errors import OperationFailed
from ccxt.base.errors import RateLimitExceeded
from ccxt.base.decimal_to_precision import TICK_SIZE
from ccxt.base.precise import Precise


class arkham(Exchange, ImplicitAPI):

    def describe(self) -> Any:
        return self.deep_extend(super(arkham, self).describe(), {
            'id': 'arkham',
            'name': 'ARKHAM',
            'countries': ['US'],
            'version': 'v1',
            'rateLimit': 20 / 3,  # 150 req/s
            'certified': False,
            'pro': True,
            'has': {
                'CORS': False,
                'spot': True,
                'margin': False,
                'swap': True,
                'future': False,
                'option': False,
                'cancelAllOrders': True,
                'cancelOrder': True,
                'createDepositAddress': True,
                'createOrder': True,
                'fetchAccounts': True,
                'fetchBalance': True,
                'fetchClosedOrders': True,
                'fetchCurrencies': True,
                'fetchDepositAddress': False,
                'fetchDepositAddressesByNetwork': True,
                'fetchDeposits': True,
                'fetchFundingHistory': True,
                'fetchLeverage': True,
                'fetchLeverageTiers': True,
                'fetchMyTrades': True,
                'fetchOHLCV': True,
                'fetchOpenOrders': True,
                'fetchOrder': True,
                'fetchOrderBook': True,
                'fetchPositions': True,
                'fetchTicker': True,
                'fetchTickers': True,
                'fetchTime': True,
                'fetchTrades': True,
                'fetchTradingFees': True,
                'fetchWithdrawals': True,
                'sandbox': False,
                'setLeverage': True,
                'withdraw': True,
            },
            'timeframes': {
                # enums are wrong in DOCS, these string values need to be in request
                '1m': '1m',
                '5m': '5m',
                '15m': '15m',
                '30m': '30m',
                '1h': '1h',
                '6h': '6h',
                '1d': '24h',
            },
            'urls': {
                'logo': 'https://github.com/user-attachments/assets/5cefdcfb-2c10-445b-835c-fa21317bf5ac',
                'api': {
                    'v1': 'https://arkm.com/api',
                },
                'www': 'https://arkm.com/',
                'referral': {
                    'url': 'https://arkm.com/register?ref=ccxt',
                    'discount': 0,
                },
                'doc': [
                    'https://arkm.com/limits-api',
                    'https://info.arkm.com/api-platform',
                ],
                'fees': 'https://arkm.com/fees',
            },
            'api': {
                'v1': {
                    'public': {
                        'get': {
                            'alerts': 1,
                            'announcements': 1,
                            'assets': 1,
                            'book': 1,
                            'candles': 1,
                            'chains': 1,
                            'contracts': 1,
                            'index-price': 1,
                            'index-prices': 1,
                            'margin-schedules': 1,
                            'marketcapchart': 1,
                            'marketcaps': 1,
                            'pair': 1,
                            'pairs': 1,
                            'server-time': 1,
                            'ticker': 1,
                            'tickers': 1,
                            'trades': 1,
                        },
                    },
                    'private': {
                        # for orders: spot 20/s, todo: perp 40/s
                        'get': {
                            'user': 7.5,
                            'orders': 7.5,
                            'orders/by-client-order-id': 7.5,
                            'orders/history': 7.5,
                            'orders/history/by-client-order-id': 7.5,
                            'orders/history_offset': 7.5,
                            'orders/{id}': 7.5,
                            'trades': 7.5,
                            'trades/history': 7.5,
                            'trades/time': 7.5,
                            'trigger-orders': 7.5,
                            'account/airdrops': 7.5,
                            'account/balance-updates': 7.5,
                            'account/balances': 7.5,
                            'account/balances/ll': 7.5,
                            'account/balances/history': 7.5,
                            'account/balances/commissions': 7.5,
                            'account/deposit/addresses': 7.5,
                            'account/deposits': 7.5,
                            'account/fees': 7.5,
                            'account/funding-rate-payments': 7.5,
                            'account/leverage': 7.5,
                            'account/lsp-assignments': 7.5,
                            'account/margin': 7.5,
                            'account/margin/all': 7.5,
                            'account/notifications': 7.5,
                            'account/position-updates': 7.5,
                            'account/positions': 7.5,
                            'account/realized-pnl': 7.5,
                            'account/rebates': 7.5,
                            'account/referral-links': 7.5,
                            'account/sessions': 7.5,
                            'account/settings': 7.5,
                            'account/settings/price-alert': 7.5,
                            'account/transfers': 7.5,
                            'account/unsubscribe': 7.5,
                            'account/watchlist': 7.5,
                            'account/withdrawal/addresses': 7.5,
                            'account/withdrawal/addresses/{id}': 7.5,
                            'account/withdrawals': 7.5,
                            'subaccounts': 7.5,
                            'airdrop': 7.5,
                            'airdrop/claim': 7.5,
                            'affiliate-dashboard/commission-earned': 7.5,
                            'affiliate-dashboard/min-arkm-last-30d': 7.5,
                            'affiliate-dashboard/points': 7.5,
                            'affiliate-dashboard/points-season-1': 7.5,
                            'affiliate-dashboard/points-season-2': 7.5,
                            'affiliate-dashboard/realized-pnl': 7.5,
                            'affiliate-dashboard/rebate-balance': 7.5,
                            'affiliate-dashboard/referral-count': 7.5,
                            'affiliate-dashboard/referrals-season-1': 7.5,
                            'affiliate-dashboard/referrals-season-2': 7.5,
                            'affiliate-dashboard/trading-volume-stats': 7.5,
                            'affiliate-dashboard/volume-season-1': 7.5,
                            'affiliate-dashboard/volume-season-2': 7.5,
                            'affiliate-dashboard/api-key': 7.5,
                            'competitions/opt-in-status': 7.5,
                            'rewards/info': 7.5,
                            'rewards/vouchers': 7.5,
                        },
                        'post': {
                            'orders/new': 7.5,
                            'trigger-orders/new': 7.5,
                            'orders/cancel': 7.5,
                            'trigger-orders/cancel': 7.5,
                            'orders/cancel/all': 7.5,
                            'trigger-orders/cancel/all': 7.5,
                            'orders/new/simple': 7.5,
                            'account/deposit/addresses/new': 7.5,
                            'account/leverage': 7.5,
                            'account/notifications/read': 7.5,
                            'account/referral-links': 7.5,
                            'account/sessions/delete': 7.5,
                            'account/sessions/terminate-all': 7.5,
                            'account/settings/update': 7.5,
                            'account/watchlist/add': 7.5,
                            'account/watchlist/remove': 7.5,
                            'account/withdraw': 7.5,
                            'account/withdrawal/addresses/confirm': 7.5,
                            'subaccounts': 7.5,
                            'subaccounts/transfer': 7.5,
                            'subaccounts/perp-transfer': 7.5,
                            'subaccounts/update-settings': 7.5,
                            'airdrop': 7.5,
                            'api-key/create': 7.5,
                            'authenticate': 7.5,
                            'competitions/opt-in': 7.5,
                            'rewards/vouchers/claim': 7.5,
                        },
                        'put': {
                            'account/referral-links/{id}/slug': 7.5,
                            'account/settings/price-alert': 7.5,
                            'account/withdrawal/addresses/{id}': 7.5,
                            'subaccounts': 7.5,
                            'api-key/update/{id}': 7.5,
                        },
                        'delete': {
                            'account/settings/price-alert': 7.5,
                            'account/withdrawal/addresses/{id}': 7.5,
                            'subaccounts/{subaccountId}': 7.5,
                            'api-key/{id}': 7.5,
                        },
                    },
                },
            },
            'options': {
                'networks': {
                    'ETH': 'ETH',
                    'ERC20': 'ETH',
                    'BTC': 'BTC',
                    'SOL': 'SOL',
                    'TON': 'TON',
                    'DOGE': 'DOGE',
                    'SUI': 'SUI',
                    'XRP': 'XRP',
                    'OP': 'OP',
                    'AVAXC': 'AVAX',
                    'ARBONE': 'ARB',
                },
                'networksById': {
                    'ETH': 'ERC20',
                    'ERC20': 'ERC20',
                },
                'requestExpiration': 5000,  # 5 seconds
                'timeframeDurations': {
                    '1m': 60000000,
                    '5m': 300000000,
                    '15m': 900000000,
                    '30m': 1800000000,
                    '1h': 3600000000,
                    '6h': 21600000000,
                    '1d': 86400000000,
                },
            },
            'features': {
                'default': {
                    'sandbox': False,
                    'createOrder': {
                        'marginMode': False,
                        'triggerPrice': True,
                        'triggerPriceType': {
                            'mark': True,
                            'index': True,
                            'last': True,
                        },
                        'triggerDirection': True,
                        'stopLossPrice': True,
                        'takeProfitPrice': True,
                        'attachedStopLossTakeProfit': None,
                        'timeInForce': {
                            'IOC': True,
                            'FOK': True,
                            'PO': True,
                            'GTD': False,
                        },
                        'hedged': False,
                        'selfTradePrevention': False,
                        'trailing': False,
                        'iceberg': False,
                        'leverage': False,
                        'marketBuyByCost': False,
                        'marketBuyRequiresPrice': False,
                    },
                    'createOrders': None,
                    'fetchMyTrades': {
                        'marginMode': False,
                        'limit': 100,
                        'daysBack': None,
                        'untilDays': 1,
                        'symbolRequired': False,
                    },
                    'fetchOrder': {
                        'marginMode': False,
                        'trigger': False,
                        'trailing': False,
                        'symbolRequired': False,
                    },
                    'fetchOpenOrders': {
                        'marginMode': True,
                        'limit': None,
                        'trigger': False,
                        'trailing': False,
                        'symbolRequired': False,
                    },
                    'fetchOrders': None,
                    'fetchClosedOrders': {
                        'marginMode': False,
                        'limit': 100,
                        'daysBack': None,
                        'daysBackCanceled': None,
                        'untilDays': None,
                        'trigger': False,
                        'trailing': False,
                        'symbolRequired': False,
                    },
                    'fetchOHLCV': {
                        'limit': 365,
                    },
                },
                'spot': {
                    'extends': 'default',
                },
                'swap': {
                    'linear': {
                        'extends': 'default',
                    },
                    'inverse': None,
                },
                'future': {
                    'linear': {
                        'extends': 'default',
                    },
                    'inverse': None,
                },
            },
            'precisionMode': TICK_SIZE,
            'exceptions': {
                'exact': {
                    # 1XXXX General Errors
                    # These errors can occur for a variety of reasons and may be returned by the API or Websocket on any endpoint.
                    '10000': OperationFailed,
                    '10001': BadRequest,
                    '10002': AuthenticationError,
                    '10003': BadSymbol,
                    '10004': ArgumentsRequired,
                    '10005': RateLimitExceeded,
                    '10006': PermissionDenied,
                    '10007': PermissionDenied,
                    '10008': RateLimitExceeded,
                    '10009': PermissionDenied,
                    '10010': PermissionDenied,
                    '10011': AuthenticationError,
                    '10012': PermissionDenied,
                    '10013': PermissionDenied,
                    '10014': AuthenticationError,
                    '10015': PermissionDenied,
                    '10016': PermissionDenied,
                    '10017': PermissionDenied,
                    '10018': AuthenticationError,
                    '10019': AuthenticationError,
                    '10020': PermissionDenied,
                    '10021': PermissionDenied,
                    '10022': ExchangeError,
                    '10023': BadRequest,
                    '10024': ExchangeError,
                    '10025': BadRequest,
                    #  #2XXXX General Websocket Errors
                    '20001': BadRequest,
                    '20002': ArgumentsRequired,
                    '20003': BadRequest,
                    '20004': ArgumentsRequired,
                    '20005': BadRequest,
                    #  #3XXXX Trading Errors
                    '30001': InvalidOrder,
                    '30002': InvalidOrder,
                    '30003': InvalidOrder,
                    '30004': InvalidOrder,
                    '30005': InvalidOrder,
                    '30006': InvalidOrder,
                    '30007': BadSymbol,
                    '30008': OperationRejected,
                    '30009': OperationRejected,
                    '30010': InsufficientFunds,
                    '30011': BadSymbol,
                    '30012': OperationRejected,
                    '30013': OperationRejected,
                    '30014': InvalidOrder,
                    '30015': OrderNotFound,
                    '30016': InvalidOrder,
                    '30017': InvalidOrder,
                    '30018': InvalidOrder,
                    '30019': OperationRejected,
                    '30020': InvalidOrder,
                    '30021': InvalidOrder,
                    '30022': InvalidOrder,
                    '30023': InvalidOrder,
                    '30024': InvalidOrder,
                    '30025': BadRequest,
                    '30026': PermissionDenied,
                    '30027': PermissionDenied,
                    '30028': OrderNotFound,
                    #  #4XXXX Funding Errors
                    '40001': OperationRejected,
                    '40002': BadRequest,
                    '40003': InvalidAddress,
                    '40004': OperationRejected,
                    '40005': BadRequest,
                    '40006': PermissionDenied,
                    '40007': OperationRejected,
                    '40008': OperationRejected,
                    '40009': OperationRejected,
                    '40010': BadRequest,
                    '40011': OperationRejected,
                    '40012': BadRequest,
                    '40013': BadRequest,
                    #  #9XXXX Other Errors
                    '90001': BadRequest,
                    '90002': BadRequest,
                    '90003': OperationRejected,
                    '90004': BadRequest,
                    '90005': BadRequest,
                    '90006': RateLimitExceeded,
                    '90007': AuthenticationError,
                    '90008': RateLimitExceeded,
                    '90009': PermissionDenied,
                    '90010': BadRequest,
                    '90011': RateLimitExceeded,
                },
                'broad': {
                    'less than min withdrawal ': OperationRejected,  # {"message":"amount 1 less than min withdrawal 5"}
                },
            },
        })

    async def fetch_currencies(self, params={}) -> Currencies:
        """
        fetches all available currencies on an exchange

        https://arkm.com/docs#get/public/assets

        :param dict [params]: extra parameters specific to the exchange API endpoint
        :returns dict: an associative dictionary of currencies
        """
        response = await self.v1PublicGetAssets(params)
        #
        #    [
        #        {
        #            "symbol": "USDT",
        #            "name": "Tether",
        #            "imageUrl": "https://static.arkhamintelligence.com/tokens/tether.png",
        #            "stablecoin": True,
        #            "featuredPair": "BTC_USDT",
        #            "chains": [
        #                {
        #                    "symbol": "ETH",
        #                    "assetSymbol": "ETH",
        #                    "name": "Ethereum",
        #                    "type": "1",
        #                    "confirmations": "6",
        #                    "blockTime": "12000000"
        #                }
        #            ],
        #            "status": "listed",
        #            "minDeposit": "5",
        #            "minWithdrawal": "5",
        #            "withdrawalFee": "2"
        #        },
        #        ...
        #
        result: dict = {}
        for i in range(0, len(response)):
            currency = response[i]
            id = self.safe_string(currency, 'symbol')
            code = self.safe_currency_code(id)
            networks: dict = {}
            chains = self.safe_list(currency, 'chains', [])
            for j in range(0, len(chains)):
                chain = chains[j]
                networkId = self.safe_string(chain, 'symbol')
                network = self.network_id_to_code(networkId)
                networks[network] = {
                    'info': chain,
                    'id': networkId,
                    'network': network,
                    'title': self.safe_string(chain, 'name'),
                    'active': None,
                    'deposit': None,
                    'withdraw': None,
                    'fee': None,
                    'precision': None,
                    'limits': {
                        'withdraw': {
                            'min': None,
                            'max': None,
                        },
                    },
                }
            result[code] = self.safe_currency_structure({
                'info': currency,
                'id': id,
                'code': code,
                'name': self.safe_string(currency, 'name'),
                'active': self.safe_string(currency, 'status') == 'listed',
                'deposit': None,
                'withdraw': None,
                'fee': self.safe_number(currency, 'withdrawalFee'),
                'precision': None,
                'limits': {
                    'amount': {
                        'min': None,
                        'max': None,
                    },
                    'withdraw': {
                        'min': self.safe_number(currency, 'minWithdrawal'),
                        'max': None,
                    },
                    'deposit': {
                        'min': self.safe_number(currency, 'minDeposit'),
                        'max': None,
                    },
                },
                'type': 'crypto',
                'networks': networks,
            })
        return result

    async def fetch_markets(self, params={}) -> List[Market]:
        """

        https://arkm.com/docs#get/public/pairs

        retrieves data on all markets for arkm
        :param dict [params]: extra parameters specific to the exchange API endpoint
        :returns dict[]: an array of objects representing market data
        """
        response = await self.v1PublicGetPairs(params)
        #
        #    [
        #        {
        #            "symbol": "BTC_USDT",
        #            "baseSymbol": "BTC",
        #            "baseImageUrl": "https://static.arkhamintelligence.com/tokens/bitcoin.png",
        #            "baseIsStablecoin": False,
        #            "baseName": "Bitcoin",
        #            "quoteSymbol": "USDT",
        #            "quoteImageUrl": "https://static.arkhamintelligence.com/tokens/tether.png",
        #            "quoteIsStablecoin": True,
        #            "quoteName": "Tether",
        #            "minTickPrice": "0.01",
        #            "minLotSize": "0.00001",
        #            "minSize": "0.00001",
        #            "maxSize": "9000",
        #            "minPrice": "0.01",
        #            "maxPrice": "1000000",
        #            "minNotional": "5",
        #            "maxPriceScalarUp": "1.8",
        #            "maxPriceScalarDown": "0.2",
        #            "pairType": "spot",  # atm, always 'spot' value
        #            "maxLeverage": "0",
        #            "status": "listed"
        #        },
        #        {
        #            "symbol": "BTC_USDT_PERP",
        #            "baseSymbol": "BTC.P",
        #            "baseImageUrl": "https://static.arkhamintelligence.com/tokens/bitcoin.png",
        #            "baseIsStablecoin": False,
        #            "baseName": "Bitcoin Perpetual",
        #            "quoteSymbol": "USDT",
        #            "quoteImageUrl": "https://static.arkhamintelligence.com/tokens/tether.png",
        #            "quoteIsStablecoin": True,
        #            "quoteName": "Tether",
        #            "minTickPrice": "0.01",
        #            "minLotSize": "0.00001",
        #            "minSize": "0.00001",
        #            "maxSize": "9000",
        #            "minPrice": "0.01",
        #            "maxPrice": "1000000",
        #            "minNotional": "5",
        #            "maxPriceScalarUp": "1.5",
        #            "maxPriceScalarDown": "0.5",
        #            "pairType": "perpetual",
        #            "marginSchedule": "C",
        #            "maxLeverage": "25",
        #            "status": "listed"
        #        },
        #        ...
        #
        result = []
        for i in range(0, len(response)):
            market = response[i]
            id = self.safe_string(market, 'symbol')
            baseId = self.safe_string(market, 'baseSymbol')
            quoteId = self.safe_string(market, 'quoteSymbol')
            base = self.safe_currency_code(baseId)
            quote = self.safe_currency_code(quoteId)
            marketType: Str = None
            symbol: Str = None
            pairType = self.safe_string(market, 'pairType')
            isSpot = pairType == 'spot'
            isPerpetual = pairType == 'perpetual'
            settle = None
            settleId = None
            if isSpot:
                marketType = 'spot'
                symbol = base + '/' + quote
            elif isPerpetual:
                marketType = 'swap'
                base = base.replace('.P', '')
                settle = quote
                settleId = quoteId
                symbol = base + '/' + quote + ':' + settle
            result.append({
                'id': id,
                'symbol': symbol,
                'base': base,
                'quote': quote,
                'settle': settle,
                'baseId': baseId,
                'quoteId': quoteId,
                'settleId': settleId,
                'type': marketType,
                'spot': isSpot,
                'margin': None,
                'swap': isPerpetual,
                'future': False,
                'option': False,
                'active': self.safe_string(market, 'status') == 'listed',
                'contract': isPerpetual,
                'linear': True if isPerpetual else None,
                'inverse': False if isPerpetual else None,
                'contractSize': None if isSpot else 1,  # seems 1 per fetchTrades
                'expiry': None,
                'expiryDatetime': None,
                'strike': None,
                'optionType': None,
                'precision': {
                    'price': self.safe_number(market, 'minTickPrice'),
                    'amount': self.safe_number(market, 'minLotSize'),
                },
                'limits': {
                    'leverage': {
                        'min': None,
                        'max': None,
                    },
                    'amount': {
                        'min': self.safe_number(market, 'minSize'),
                        'max': self.safe_number(market, 'maxSize'),
                    },
                    'price': {
                        'min': self.safe_number(market, 'minPrice'),
                        'max': self.safe_number(market, 'maxPrice'),
                    },
                    'cost': {
                        'min': self.safe_number(market, 'minNotional'),
                        'max': None,
                    },
                },
                'created': None,
                'info': market,
            })
        return result

    async def fetch_time(self, params={}):
        """
        fetches the current integer timestamp in milliseconds from the exchange server

        https://arkm.com/docs#get/public/server-time

        :param dict [params]: extra parameters specific to the exchange API endpoint
        :returns int: the current integer timestamp in milliseconds from the exchange server
        """
        response = await self.v1PublicGetServerTime(params)
        #
        #    {
        #        "serverTime": "1753465832770820"
        #    }
        #
        return self.safe_integer_product(response, 'serverTime', 0.001)

    async def fetch_order_book(self, symbol: str, limit: Int = None, params={}) -> OrderBook:
        """
        fetches information on open orders with bid(buy) and ask(sell) prices, volumes and other data

        https://arkm.com/docs#get/public/book

        :param str symbol: unified symbol of the market to fetch the order book for
        :param int [limit]: the number of order book entries to return, max 50
        :param dict [params]: extra parameters specific to the exchange API endpoint
        :returns dict: A dictionary of `order book structures <https://docs.ccxt.com/#/?id=order-book-structure>` indexed by market symbols
        """
        await self.load_markets()
        market = self.market(symbol)
        request: dict = {
            'symbol': market['id'],
        }
        if limit is not None:
            request['limit'] = limit
        response = await self.v1PublicGetBook(self.extend(request, params))
        #
        #    {
        #        "symbol": "BTC_USDT",
        #        "group": "0.01",
        #        "asks": [
        #            {
        #                "price": "122900.43",
        #                "size": "0.0243"
        #            },
        #            {
        #                "price": "121885.53",
        #                "size": "0.00116"
        #            },
        #            ...
        #        ],
        #        "bids": [
        #            {
        #                "price": "20400",
        #                "size": "0.00316"
        #            },
        #            {
        #                "price": "30000",
        #                "size": "0.00116"
        #            },
        #            ...
        #        ],
        #        "lastTime": "1753419275604353"
        #    }
        #
        timestamp = self.safe_integer_product(response, 'lastTime', 0.001)
        marketId = self.safe_string(response, 'symbol')
        return self.parse_order_book(response, self.safe_symbol(marketId, market), timestamp, 'bids', 'asks', 'price', 'size')

    async def fetch_ohlcv(self, symbol: str, timeframe='1m', since: Int = None, limit: Int = None, params={}) -> List[list]:
        """
        fetches historical candlestick data containing the open, high, low, and close price, and the volume of a market

        https://arkm.com/docs#get/public/candles

        :param str symbol: unified symbol of the market to fetch OHLCV data for
        :param str timeframe: the length of time each candle represents
        :param int [since]: timestamp in ms of the earliest candle to fetch
        :param int [limit]: the maximum amount of candles to fetch
        :param dict [params]: extra parameters specific to the exchange API endpoint
        :param int [params.until]: timestamp in ms for the ending date filter, default is the current time
        :param boolean [params.paginate]: default False, when True will automatically paginate by calling self endpoint multiple times. See in the docs all the [availble parameters](https://github.com/ccxt/ccxt/wiki/Manual#pagination-params)
        :returns int[][]: A list of candles ordered, open, high, low, close, volume
        """
        maxLimit = 365
        await self.load_markets()
        paginate = False
        paginate, params = self.handle_option_and_params(params, 'fetchOHLCV', 'paginate', False)
        if paginate:
            return await self.fetch_paginated_call_deterministic('fetchOHLCV', symbol, since, limit, timeframe, params, maxLimit)
        market = self.market(symbol)
        request: dict = {
            'symbol': market['id'],
            'duration': self.safe_string(self.timeframes, timeframe, timeframe),
        }
        durationMs = self.parse_timeframe(timeframe) * 1000
        until = self.safe_integer(params, 'until')
        params = self.omit(params, ['until'])
        selectedLimit = min(limit, maxLimit) if (limit is not None) else maxLimit
        if since is not None:
            request['start'] = since
            request['end'] = self.sum(since, selectedLimit * durationMs)
        else:
            now = self.milliseconds()
            request['end'] = until if (until is not None) else now
            request['start'] = request['end'] - selectedLimit * durationMs
        # exchange needs microseconds
        request['start'] = request['start'] * 1000
        request['end'] = request['end'] * 1000
        response = await self.v1PublicGetCandles(self.extend(request, params))
        #
        #    [
        #        {
        #            "symbol": "BTC_USDT_PERP",
        #            "time": "1753464720000000",
        #            "duration": "60000000",
        #            "open": "116051.35",
        #            "high": "116060.27",
        #            "low": "116051.35",
        #            "close": "116060.27",
        #            "volume": "0.0257",
        #            "quoteVolume": "2982.6724054"
        #        },
        #        ...
        #    ]
        #
        return self.parse_ohlcvs(response, market, timeframe, since, limit)

    def parse_ohlcv(self, ohlcv, market: Market = None) -> list:
        #
        #        {
        #            "symbol": "BTC_USDT_PERP",
        #            "time": "1753464720000000",
        #            "duration": "60000000",
        #            "open": "116051.35",
        #            "high": "116060.27",
        #            "low": "116051.35",
        #            "close": "116060.27",
        #            "volume": "0.0257",
        #            "quoteVolume": "2982.6724054"
        #        }
        #
        return [
            self.safe_integer_product(ohlcv, 'time', 0.001),
            self.safe_number(ohlcv, 'open'),
            self.safe_number(ohlcv, 'high'),
            self.safe_number(ohlcv, 'low'),
            self.safe_number(ohlcv, 'close'),
            self.safe_number(ohlcv, 'volume'),
        ]

    async def fetch_tickers(self, symbols: Strings = None, params={}) -> Tickers:
        response = await self.v1PublicGetTickers(params)
        #
        #    [
        #        {
        #            "symbol": "BTC_USDT_PERP",
        #            "baseSymbol": "BTC.P",
        #            "quoteSymbol": "USDT",
        #            "indexCurrency": "USDT",
        #            "price": "118806.89",
        #            "price24hAgo": "118212.29",
        #            "high24h": "119468.05",
        #            "low24h": "117104.44",
        #            "volume24h": "180.99438",
        #            "quoteVolume24h": "21430157.5928827",
        #            "markPrice": "118814.71",
        #            "indexPrice": "118804.222610343",
        #            "fundingRate": "0.000007",
        #            "nextFundingRate": "0.000006",
        #            "nextFundingTime": "1753390800000000",
        #            "productType": "perpetual",
        #            "openInterest": "2.55847",
        #            "usdVolume24h": "21430157.5928827",
        #            "openInterestUSD": "303963.8638583"
        #        },
        #        ...
        #
        return self.parse_tickers(response, symbols)

    async def fetch_ticker(self, symbol: str, params={}) -> Ticker:
        """
        fetches a price ticker, a statistical calculation with the information calculated over the past 24 hours for a specific market
        :param str symbol: unified symbol of the market to fetch the ticker for
        :param dict [params]: extra parameters specific to the exchange API endpoint
        :returns dict: a `ticker structure <https://docs.ccxt.com/#/?id=ticker-structure>`
        """
        await self.load_markets()
        market = self.market(symbol)
        request: dict = {
            'symbol': market['id'],
        }
        response = await self.v1PublicGetTicker(self.extend(request, params))
        #
        #        {
        #            "symbol": "BTC_USDT_PERP",
        #            "baseSymbol": "BTC.P",
        #            "quoteSymbol": "USDT",
        #            "indexCurrency": "USDT",
        #            "price": "118806.89",
        #            "price24hAgo": "118212.29",
        #            "high24h": "119468.05",
        #            "low24h": "117104.44",
        #            "volume24h": "180.99438",
        #            "quoteVolume24h": "21430157.5928827",
        #            "markPrice": "118814.71",
        #            "indexPrice": "118804.222610343",
        #            "fundingRate": "0.000007",
        #            "nextFundingRate": "0.000006",
        #            "nextFundingTime": "1753390800000000",
        #            "productType": "perpetual",
        #            "openInterest": "2.55847",
        #            "usdVolume24h": "21430157.5928827",
        #            "openInterestUSD": "303963.8638583"
        #        }
        #
        return self.parse_ticker(response, market)

    def parse_ticker(self, ticker: dict, market: Market = None) -> Ticker:
        marketId = self.safe_string(ticker, 'symbol')
        market = self.safe_market(marketId, market)
        return self.safe_ticker({
            'info': ticker,
            'symbol': self.safe_symbol(marketId, market),
            'high': self.safe_number(ticker, 'high24h'),
            'low': self.safe_number(ticker, 'low24h'),
            'bid': self.safe_number(ticker, 'bid'),
            'last': self.safe_number(ticker, 'price'),
            'open': self.safe_number(ticker, 'price24hAgo'),
            'change': self.safe_number(ticker, 'priceChange'),
            'percentage': self.safe_number(ticker, 'priceChangePercent'),
            'baseVolume': self.safe_number(ticker, 'volume24h'),
            'quoteVolume': self.safe_number(ticker, 'usdVolume24h'),
            'markPrice': self.safe_number(ticker, 'markPrice'),
            'indexPrice': self.safe_number(ticker, 'indexPrice'),
            'vwap': None,
            'average': None,
            'previousClose': None,
            'askVolume': None,
            'bidVolume': None,
        })

    async def fetch_trades(self, symbol: str, since: Int = None, limit: Int = None, params={}) -> List[Trade]:
        """
        get the list of most recent trades for a particular symbol

        https://arkm.com/docs#get/public/trades

        :param str symbol: unified symbol of the market to fetch trades for
        :param int [since]: timestamp in ms of the earliest trade to fetch
        :param int [limit]: the maximum amount of trades to fetch
        :param dict [params]: extra parameters specific to the exchange API endpoint
        :param str [params.loc]: crypto location, default: us
        :param str [params.method]: method, default: marketPublicGetV1beta3CryptoLocTrades
        :returns Trade[]: a list of `trade structures <https://docs.ccxt.com/#/?id=public-trades>`
        """
        await self.load_markets()
        market = self.market(symbol)
        marketId = market['id']
        request: dict = {
            'symbol': marketId,
        }
        if limit is not None:
            request['limit'] = limit
        response = await self.v1PublicGetTrades(self.extend(request, params))
        #
        #    [
        #        {
        #            "symbol": "BTC_USDT_PERP",
        #            "revisionId": "1130514101",
        #            "size": "0.01668",
        #            "price": "116309.57",
        #            "takerSide": "sell",
        #            "time": "1753439710374047"
        #        },
        #        ...
        #    ]
        #
        return self.parse_trades(response, market, since, limit)

    def parse_trade(self, trade: dict, market: Market = None) -> Trade:
        #
        # fetchTrades
        #
        #        {
        #            "symbol": "BTC_USDT_PERP",
        #            "revisionId": "1130514101",
        #            "size": "0.01668",
        #            "price": "116309.57",
        #            "takerSide": "sell",
        #            "time": "1753439710374047"
        #        }
        #
        # fetchMyTrades
        #
        #        {
        #            "symbol": "SOL_USDT",
        #            "revisionId": "891839406",
        #            "size": "0.042",
        #            "price": "185.06",
        #            "takerSide": "sell",
        #            "time": "1753773952039342",
        #            "orderId": "3717304929194",
        #            "userSide": "sell",
        #            "quoteFee": "0.00777252",
        #            "arkmFee": "0",
        #            "clientOrderId": ""
        #        }
        #
        marketId = self.safe_string(trade, 'symbol')
        market = self.safe_market(marketId, market)
        timestamp = self.safe_integer_product(trade, 'time', 0.001)
        quoteFee = self.safe_number(trade, 'quoteFee')
        arkmFee = self.safe_number(trade, 'arkmFee')
        fee = None
        if quoteFee is not None:
            fee = {
                'cost': quoteFee,
                'currency': market['quote'],
            }
        elif arkmFee is not None:
            fee = {
                'cost': arkmFee,
                'currency': 'ARKM',
            }
        return self.safe_trade({
            'info': trade,
            'id': self.safe_string(trade, 'revisionId'),
            'timestamp': timestamp,
            'datetime': self.iso8601(timestamp),
            'symbol': market['symbol'],
            'type': None,
            'side': self.safe_string_2(trade, 'userSide', 'takerSide'),  # priority to userSide
            'takerOrMaker': None,
            'price': self.safe_string(trade, 'price'),
            'amount': self.safe_string(trade, 'size'),
            'cost': None,
            'fee': fee,
            'order': self.safe_string(trade, 'orderId'),
        }, market)

    async def fetch_order(self, id: str, symbol: Str = None, params={}):
        """
        fetches information on an order made by the user

        https://arkm.com/docs#get/orders/by-client-order-id

        :param str id: the order id
        :param str symbol: unified symbol of the market the order was made in
        :param dict [params]: extra parameters specific to the exchange API endpoint
        :param str [params.clientOrderId]: a unique id for the order
        :returns dict: An `order structure <https://docs.ccxt.com/#/?id=order-structure>`
        """
        request: dict = {
            'id': int(id),
        }
        response = await self.v1PrivateGetOrdersId(self.extend(request, params))
        #
        #    {
        #        "orderId": "3690478767430",
        #        "userId": "2959123",
        #        "subaccountId": "0",
        #        "symbol": "SOL_USDT",
        #        "time": "1753696843913970",
        #        "side": "sell",
        #        "type": "limitGtc",
        #        "size": "0.066",
        #        "price": "293.2",
        #        "postOnly": False,
        #        "reduceOnly": False,
        #        "executedSize": "0",
        #        "status": "booked",
        #        "avgPrice": "0",
        #        "executedNotional": "0",
        #        "creditFeePaid": "0",
        #        "marginBonusFeePaid": "0",
        #        "quoteFeePaid": "0",
        #        "arkmFeePaid": "0",
        #        "revisionId": "887956326",
        #        "lastTime": "1753696843914830",
        #        "clientOrderId": "",
        #        "lastSize": "0",
        #        "lastPrice": "0",
        #        "lastCreditFee": "0",
        #        "lastMarginBonusFee": "0",
        #        "lastQuoteFee": "0",
        #        "lastArkmFee": "0"
        #    }
        #
        return self.parse_order(response)

    async def fetch_closed_orders(self, symbol: Str = None, since: Int = None, limit: Int = None, params={}) -> List[Order]:
        """
        fetches information on multiple closed orders made by the user

        https://arkm.com/docs#get/orders/history

        :param str symbol: unified market symbol of the market orders were made in
        :param int [since]: the earliest time in ms to fetch orders for
        :param int [limit]: the maximum number of order structures to retrieve
        :param dict [params]: extra parameters specific to the exchange API endpoint
        :param int [params.until]: the latest time in ms to fetch orders for
        :returns Order[]: a list of `order structures <https://docs.ccxt.com/#/?id=order-structure>`
        """
        await self.load_markets()
        request: dict = {}
        market = None
        if symbol is not None:
            market = self.market(symbol)
            request['symbol'] = market['id']
        if limit is not None:
            request['limit'] = limit  # note, API does not work for self param
        response = await self.v1PrivateGetOrdersHistory(self.extend(request, params))
        #
        #     [
        #        {
        #            "orderId": "3690478767430",
        #            "userId": "2959123",
        #            "subaccountId": "0",
        #            "symbol": "SOL_USDT",
        #            "time": "1753696843913970",
        #            "side": "sell",
        #            "type": "limitGtc",
        #            "size": "0.066",
        #            "price": "293.2",
        #            "postOnly": False,
        #            "reduceOnly": False,
        #            "executedSize": "0",
        #            "status": "closed",
        #            "avgPrice": "0",
        #            "executedNotional": "0",
        #            "creditFeePaid": "0",
        #            "marginBonusFeePaid": "0",
        #            "quoteFeePaid": "0",
        #            "arkmFeePaid": "0",
        #            "revisionId": "888084076",
        #            "lastTime": "1753701350088305",
        #            "clientOrderId": "",
        #            "lastSize": "0",
        #            "lastPrice": "0",
        #            "lastCreditFee": "0",
        #            "lastMarginBonusFee": "0",
        #            "lastQuoteFee": "0",
        #            "lastArkmFee": "0"
        #        }
        #    ]
        #
        return self.parse_orders(response, market, since, limit)

    async def fetch_open_orders(self, symbol: Str = None, since: Int = None, limit: Int = None, params={}) -> List[Order]:
        """
        fetch all unfilled currently open orders

        https://arkm.com/docs#get/orders

        :param str symbol: unified market symbol of the market orders were made in
        :param int [since]: the earliest time in ms to fetch orders for
        :param int [limit]: the maximum number of order structures to retrieve
        :param dict [params]: extra parameters specific to the exchange API endpoint
        :param int [params.until]: the latest time in ms to fetch orders for
        :returns Order[]: a list of `order structures <https://docs.ccxt.com/#/?id=order-structure>`
        """
        await self.load_markets()
        market = None
        if symbol is not None:
            market = self.market(symbol)
        isTriggerOrder = self.safe_bool(params, 'trigger')
        params = self.omit(params, 'trigger')
        response = None
        if isTriggerOrder:
            response = await self.v1PrivateGetTriggerOrders(self.extend({}, params))
            #
            #    [
            #        {
            #            "subaccountId": "0",
            #            "symbol": "SOL_USDT",
            #            "side": "sell",
            #            "type": "market",
            #            "size": "0.045",
            #            "price": "99.9",
            #            "postOnly": False,
            #            "reduceOnly": False,
            #            "time": "1753768103780063",
            #            "triggerOrderId": "3715847222127",
            #            "triggerType": "stopLoss",
            #            "triggerPriceType": "last",
            #            "triggerPrice": "111",
            #            "clientOrderId": "",
            #            "status": "staged"
            #        },
            #    ]
            #
        else:
            response = await self.v1PrivateGetOrders(self.extend({}, params))
            #
            # [
            #    {
            #        "orderId": "3690478767430",
            #        "userId": "2959123",
            #        "subaccountId": "0",
            #        "symbol": "SOL_USDT",
            #        "time": "1753696843913970",
            #        "side": "sell",
            #        "type": "limitGtc",
            #        "size": "0.066",
            #        "price": "293.2",
            #        "postOnly": False,
            #        "reduceOnly": False,
            #        "executedSize": "0",
            #        "status": "booked",
            #        "avgPrice": "0",
            #        "executedNotional": "0",
            #        "creditFeePaid": "0",
            #        "marginBonusFeePaid": "0",
            #        "quoteFeePaid": "0",
            #        "arkmFeePaid": "0",
            #        "revisionId": "887956326",
            #        "lastTime": "1753696843914830",
            #        "clientOrderId": "",
            #        "lastSize": "0",
            #        "lastPrice": "0",
            #        "lastCreditFee": "0",
            #        "lastMarginBonusFee": "0",
            #        "lastQuoteFee": "0",
            #        "lastArkmFee": "0"
            #    }
            # ]
            #
        return self.parse_orders(response, market, since, limit)

    async def cancel_order(self, id: str, symbol: Str = None, params={}):
        """
        cancels an open order

        https://arkm.com/docs#post/orders/cancel

        :param str id: order id
        :param str symbol: unified symbol of the market the order was made in
        :param dict [params]: extra parameters specific to the exchange API endpoint
        :returns dict: An `order structure <https://docs.ccxt.com/#/?id=order-structure>`
        """
        isTriggerOrder = self.safe_bool(params, 'trigger')
        params = self.omit(params, 'trigger')
        response = None
        request: dict = {}
        clientOrderId = self.safe_integer(params, 'clientOrderId')
        if clientOrderId is not None:
            params = self.omit(params, 'clientOrderId')
            request['clientOrderId'] = clientOrderId
        else:
            if isTriggerOrder:
                request['triggerOrderId'] = int(id)
            else:
                request['orderId'] = int(id)
        if isTriggerOrder:
            if symbol is None:
                raise ArgumentsRequired(self.id + ' cancelOrder() requires a symbol argument for trigger orders')
            market = self.market(symbol)
            request['symbol'] = market['id']
            response = await self.v1PrivatePostTriggerOrdersCancel(self.extend(request, params))
        else:
            response = await self.v1PrivatePostOrdersCancel(self.extend(request, params))
        #
        # {"orderId":3691703758327}
        #
        return self.parse_order(response)

    async def cancel_all_orders(self, symbol: Str = None, params={}):
        """
        cancel all open orders in a market

        https://arkm.com/docs#post/orders/cancel/all

        :param str symbol: cancel alls open orders
        :param dict [params]: extra parameters specific to the exchange API endpoint
        :returns dict[]: a list of `order structures <https://docs.ccxt.com/#/?id=order-structure>`
        """
        if symbol is not None:
            raise BadRequest(self.id + ' cancelAllOrders() does not support a symbol argument, use cancelOrder() or fetchOpenOrders() instead')
        isTriggerOrder = self.safe_bool(params, 'trigger')
        params = self.omit(params, 'trigger')
        response = None
        if isTriggerOrder:
            response = await self.v1PrivatePostTriggerOrdersCancelAll(params)
        else:
            response = await self.v1PrivatePostOrdersCancelAll(params)
        #
        # []  returns an empty array, even when successfully cancels orders
        #
        return self.parse_orders(response, None)

    async def create_order(self, symbol: str, type: OrderType, side: OrderSide, amount: float, price: Num = None, params={}):
        """
        create a trade order on the exchange

        https://arkm.com/docs#post/orders/new

        :param str symbol: unified CCXT market symbol
        :param str type: "limit" or "market"
        :param str side: "buy" or "sell"
        :param float amount: the amount of currency to trade
        :param float [price]: the price at which the order is to be fulfilled, in units of the quote currency, ignored in market orders
        :param dict [params]: extra parameters specific to the exchange API endpoint
        :param str [params.timeInForce]: "GTC", "IOC", "FOK", or "PO"
        :param float [params.triggerPrice]: price for a trigger(conditional) order
        :param float [params.stopLossPrice]: price for a stoploss order
        :param float [params.takeProfitPrice]: price for a takeprofit order
        :param str [params.triggerDirection]: the direction for trigger orders, 'ascending' or 'descending'
        :param str [params.triggerPriceType]: mark, index or last
        :param bool [params.postOnly]: True or False whether the order is post-only
        :param bool [params.reduceOnly]: True or False whether the order is reduce-only
        :returns: `An order structure <https://docs.ccxt.com/#/?id=order-structure>`
        """
        await self.load_markets()
        market = self.market(symbol)
        isTriggerOrder = self.safe_number_n(params, ['triggerPrice', 'stopLossPrice', 'takeProfitPrice']) is not None
        request = self.create_order_request(symbol, type, side, amount, price, params)
        response = None
        if isTriggerOrder:
            response = await self.v1PrivatePostTriggerOrdersNew(request)
            #
            #    {
            #        "triggerOrderId": "3716436645573",
            #        "symbol": "SOL_USDT_PERP",
            #        "side": "buy",
            #        "type": "limitGtc",
            #        "size": "0.05",
            #        "price": "150"
            #    }
            #
        else:
            response = await self.v1PrivatePostOrdersNew(request)
            #
            #    {
            #        "orderId": "3694872060678",
            #        "clientOrderId": "test123",
            #        "symbol": "SOL_USDT",
            #        "subaccountId": "0",
            #        "side": "buy",
            #        "type": "limitGtc",
            #        "size": "0.05",
            #        "price": "170",
            #        "time": "1753710501474043"
            #    }
            #
        return self.parse_order(response, market)

    def create_order_request(self, symbol: str, type: OrderType, side: OrderSide, amount: float, price: Num = None, params={}):
        market = self.market(symbol)
        symbol = market['symbol']
        request: dict = {
            'symbol': market['id'],
            'side': side,
            'size': self.amount_to_precision(symbol, amount),
        }
        isBuy = (side == 'buy')
        stopLossPrice = self.safe_number(params, 'stopLossPrice')
        takeProfitPrice = self.safe_number(params, 'takeProfitPrice')
        triggerPriceAny = self.safe_string_n(params, ['triggerPrice', 'stopLossPrice', 'takeProfitPrice'])
        if triggerPriceAny is not None:
            request['triggerPrice'] = self.price_to_precision(symbol, triggerPriceAny)
            if stopLossPrice is not None:
                request['triggerType'] = 'stopLoss' if isBuy else 'takeProfit'
            elif takeProfitPrice is not None:
                request['triggerType'] = 'takeProfit' if isBuy else 'stopLoss'
            else:
                triggerDirection = self.safe_string(params, 'triggerDirection')
                if triggerDirection is None:
                    raise ArgumentsRequired(self.id + ' createOrder() requires a triggerDirection parameter when triggerPrice is specified, must be "ascending" or "descending"')
                if triggerDirection is not None:
                    if triggerDirection == 'ascending':
                        request['triggerType'] = 'stopLoss' if isBuy else 'takeProfit'
                    elif triggerDirection == 'descending':
                        request['triggerType'] = 'takeProfit' if isBuy else 'stopLoss'
            # mandatory triggerPriceType
            if self.safe_string(params, 'triggerPriceType') is None:
                request['triggerPriceType'] = 'last'  # default
        isMarketOrder = (type == 'market')
        isLimitOrder = (type == 'limit')
        isLimitExchangeSpecific = self.in_array(type, ['limitGtc', 'limitIoc', 'limitFok'])
        postOnly = self.is_post_only(isMarketOrder, False, params)
        timeInForce = self.safe_string(params, 'timeInForce')
        params = self.omit(params, ['postOnly', 'timeInForce', 'triggerPrice', 'stopLossPrice', 'takeProfitPrice', 'triggerDirection'])
        if postOnly:
            request['postOnly'] = True
        if isLimitOrder or isLimitExchangeSpecific:
            request['price'] = self.price_to_precision(symbol, price)
            #
            if timeInForce == 'IOC':
                request['type'] = 'limitIoc'
            elif timeInForce == 'FOK':
                request['type'] = 'limitFok'
            else:
                request['type'] = 'limitGtc'
        elif isMarketOrder:
            request['type'] = 'market'
        # we don't need to manually handle `reduceOnly`, `clientOrderId`, `triggerPriceType` here-specific keyname & values matches
        return self.extend(request, params)

    def parse_order(self, order: dict, market: Market = None) -> Order:
        #
        # createOrder
        #
        #    {
        #        "orderId": "3694872060678",
        #        "clientOrderId": "test123",
        #        "symbol": "SOL_USDT",
        #        "subaccountId": "0",
        #        "side": "buy",
        #        "type": "limitGtc",
        #        "size": "0.05",
        #        "price": "170",
        #        "time": "1753710501474043"
        #    }
        #
        # fetchOrder, fetchOpenOrders, fetchClosedOrders
        #
        #    {
        #        "orderId": "3690478767430",
        #        "userId": "2959123",
        #        "subaccountId": "0",
        #        "symbol": "SOL_USDT",
        #        "time": "1753696843913970",
        #        "side": "sell",
        #        "type": "limitGtc",
        #        "size": "0.066",
        #        "price": "293.2",
        #        "postOnly": False,
        #        "reduceOnly": False,
        #        "executedSize": "0",
        #        "status": "booked",
        #        "avgPrice": "0",
        #        "executedNotional": "0",
        #        "creditFeePaid": "0",
        #        "marginBonusFeePaid": "0",
        #        "quoteFeePaid": "0",
        #        "arkmFeePaid": "0",
        #        "revisionId": "887956326",
        #        "lastTime": "1753696843914830",
        #        "clientOrderId": "",
        #        "lastSize": "0",
        #        "lastPrice": "0",
        #        "lastCreditFee": "0",
        #        "lastMarginBonusFee": "0",
        #        "lastQuoteFee": "0",
        #        "lastArkmFee": "0"
        #    }
        #
        # trigger-orders: createOrder
        #
        #    {
        #        "triggerOrderId": "3716436645573",
        #        "symbol": "SOL_USDT_PERP",
        #        "side": "buy",
        #        "type": "limitGtc",
        #        "size": "0.05",
        #        "price": "150"
        #    }
        #
        # trigger-orders: fetchOpenOrders
        #
        #    {
        #            "subaccountId": "0",
        #            "symbol": "SOL_USDT",
        #            "side": "sell",
        #            "type": "market",
        #            "size": "0.045",
        #            "price": "99.9",
        #            "postOnly": False,
        #            "reduceOnly": False,
        #            "time": "1753768103780063",
        #            "triggerOrderId": "3715847222127",
        #            "triggerType": "stopLoss",
        #            "triggerPriceType": "last",
        #            "triggerPrice": "111",
        #            "clientOrderId": "",
        #            "status": "staged"
        #    }
        #
        marketId = self.safe_string(order, 'symbol')
        market = self.safe_market(marketId, market)
        isPostOnly = self.safe_bool(order, 'postOnly')
        typeRaw = self.safe_string(order, 'type')
        orderType = 'limit' if isPostOnly else self.parse_order_type(typeRaw)
        timeInForce = 'PO' if isPostOnly else self.parse_time_in_force(typeRaw)
        quoteFeePaid = self.safe_string(order, 'quoteFeePaid')
        arkmFeePaid = self.safe_string(order, 'arkmFeePaid')
        fees = []
        if quoteFeePaid is not None:
            fees.append({
                'cost': quoteFeePaid,
                'currency': self.safe_string(market, 'quote'),
            })
        if arkmFeePaid is not None:
            fees.append({
                'cost': arkmFeePaid,
                'currency': 'ARKM',
            })
        timestamp = self.safe_integer_product(order, 'time', 0.001)
        return self.safe_order({
            'id': self.safe_string_2(order, 'orderId', 'triggerOrderId'),
            'clientOrderId': self.safe_string(order, 'clientOrderId'),
            'timestamp': timestamp,
            'datetime': self.iso8601(timestamp),
            'lastTradeTimeStamp': None,
            'lastUpdateTimestamp': self.safe_integer_product(order, 'lastTime', 0.001),
            'status': self.parse_order_status(self.safe_string(order, 'status')),
            'symbol': market['symbol'],
            'type': orderType,
            'timeInForce': timeInForce,
            'postOnly': None,
            'side': self.safe_string(order, 'side'),
            'price': self.safe_number(order, 'price'),
            'triggerPrice': None,
            'cost': self.safe_number(order, 'executedNotional'),
            'average': self.safe_number_omit_zero(order, 'avgPrice'),
            'amount': self.safe_number(order, 'size'),
            'filled': self.safe_number(order, ''),
            'remaining': None,
            'trades': None,
            'fees': fees,
            'reduceOnly': self.safe_bool(order, 'reduceOnly'),
            'info': order,
        }, market)

    def parse_order_type(self, type: Str) -> Str:
        types: dict = {
            'limitGtc': 'limit',
            'limitIoc': 'limit',
            'limitFok': 'limit',
            'market': 'market',
        }
        return self.safe_string_upper(types, type, type)

    def parse_time_in_force(self, type: Str) -> Str:
        types: dict = {
            'limitGtc': 'GTC',
            'limitIoc': 'IOC',
            'limitFok': 'FOK',
            'market': 'IOC',
        }
        return self.safe_string_upper(types, type, type)

    def parse_order_status(self, status: Str):
        statuses: dict = {
            'new': 'pending',
            'staged': 'open',
            'booked': 'open',
            'taker': 'closed',
            'maker': 'closed',
            'cancelled': 'canceled',
            'closed': 'closed',
        }
        return self.safe_string(statuses, status, status)

    async def fetch_my_trades(self, symbol: Str = None, since: Int = None, limit: Int = None, params={}):
        """
        fetch all trades made by the user

        https://arkm.com/docs#get/trades/time

        :param str [symbol]: unified market symbol
        :param int [since]: the earliest time in ms to fetch trades for
        :param int [limit]: the maximum number of trade structures to retrieve
        :param dict [params]: extra parameters specific to the exchange API endpoint
        :param int [params.until]: the latest time in ms to fetch trades for
        :param str [params.page_token]: page_token - used for paging
        :returns Trade[]: a list of `trade structures <https://docs.ccxt.com/#/?id=trade-structure>`
        """
        await self.load_markets()
        request: dict = {}
        if limit is not None:
            request['limit'] = limit
        # exchange needs to obtain some `from & to` values, otherwise it does not return any result
        defaultRange = 24 * 60 * 60 * 1000  # default to last 24 hours
        if since is not None:
            request['from'] = since * 1000  # convert ms to microseconds
        else:
            request['from'] = (self.milliseconds() - defaultRange) * 1000  # default to last 24 hours
        until = self.safe_integer(params, 'until')
        if until is not None:
            params = self.omit(params, 'until')
            request['to'] = until * 1000  # convert ms to microseconds
        else:
            request['to'] = self.sum(request['from'], defaultRange * 1000)
        request, params = self.handle_until_option('until', request, params)
        response = await self.v1PrivateGetTradesTime(self.extend(request, params))
        #
        #    [
        #        {
        #            "symbol": "SOL_USDT",
        #            "revisionId": "891839406",
        #            "size": "0.042",
        #            "price": "185.06",
        #            "takerSide": "sell",
        #            "time": "1753773952039342",
        #            "orderId": "3717304929194",
        #            "userSide": "sell",
        #            "quoteFee": "0.00777252",
        #            "arkmFee": "0",
        #            "clientOrderId": ""
        #        },
        #        ...
        #
        return self.parse_trades(response, None, since, limit)

    async def fetch_accounts(self, params={}) -> List[Account]:
        """
        fetch all the accounts associated with a profile

        https://arkm.com/docs#get/user

        :param dict [params]: extra parameters specific to the exchange API endpoint
        :returns dict: a dictionary of `account structures <https://docs.ccxt.com/#/?id=account-structure>` indexed by the account type
        """
        await self.load_markets()
        request: dict = {}
        accountId = None
        accountId, params = self.handle_option_and_params(params, 'fetchAccounts', 'accountId')
        if accountId is not None:
            request['subAccountId'] = accountId
        response = await self.v1PrivateGetUser(self.extend(request, params))
        #
        #    {
        #        "id": "2959123",
        #        "email": "xyz@gmail.com",
        #        "username": "t.123",
        #        "requireMFA": True,
        #        "kycVerifiedAt": "1753434515850673",
        #        "pmm": False,
        #        "dmm": False,
        #        "becameVipAt": "0",
        #        "subaccounts": [
        #            {
        #                "id": "0",
        #                "name": "Primary",
        #                "pinned": True,
        #                "isLsp": False,
        #                "futuresEnabled": True,
        #                "payFeesInArkm": False,
        #                "lspSettings": []
        #            }
        #        ],
        #        "settings": {
        #            "autogenDepositAddresses": False,
        #            "hideBalances": False,
        #            "confirmBeforePlaceOrder": False,
        #            "tickerTapeScroll": True,
        #            "updatesFlash": True,
        #            "notifyOrderFills": False,
        #            "notifyAnnouncements": False,
        #            "notifyMarginUsage": False,
        #            "marginUsageThreshold": "0.5",
        #            "notifyWithdrawals": True,
        #            "notifyDeposits": True,
        #            "notifySendEmail": True,
        #            "notifyRebates": True,
        #            "notifyCommissions": True,
        #            "allowSequenceEmails": True,
        #            "language": "en"
        #        },
        #        "airdropKycAt": null
        #    }
        #
        subAccounts = self.safe_list(response, 'subaccounts', [])
        return self.parse_accounts(subAccounts, params)

    def parse_account(self, account):
        #
        #            {
        #                "id": "0",
        #                "name": "Primary",
        #                "pinned": True,
        #                "isLsp": False,
        #                "futuresEnabled": True,
        #                "payFeesInArkm": False,
        #                "lspSettings": []
        #            }
        #
        return {
            'id': self.safe_string(account, 'id'),
            'type': None,
            'code': None,
            'info': account,
        }

    async def fetch_balance(self, params={}) -> Balances:
        """
        query for account info

        https://arkm.com/docs#get/account/balances

        :param dict [params]: extra parameters specific to the exchange API endpoint
        :returns dict: a `balance structure <https://docs.ccxt.com/#/?id=balance-structure>`
        """
        await self.load_markets()
        response = await self.v1PrivateGetAccountBalances(params)
        #
        #    [
        #        {
        #            "subaccountId": "0",
        #            "symbol": "USDT",
        #            "balance": "19.66494694",
        #            "free": "19.66494694",
        #            "priceUSDT": "1",
        #            "balanceUSDT": "19.66494694",
        #            "freeUSDT": "19.66494694",
        #            "lastUpdateReason": "orderFill",
        #            "lastUpdateTime": "1753773952039342",
        #            "lastUpdateId": "248507437",
        #            "lastUpdateAmount": "7.77252"
        #        },
        #        {
        #            "subaccountId": "0",
        #            "symbol": "SOL",
        #            "balance": "0",
        #            "free": "0",
        #            "priceUSDT": "186.025584673",
        #            "balanceUSDT": "0",
        #            "freeUSDT": "0",
        #            "lastUpdateReason": "orderFill",
        #            "lastUpdateTime": "1753773952039342",
        #            "lastUpdateId": "248507435",
        #            "lastUpdateAmount": "-0.042"
        #        }
        #    ]
        #
        return self.parse_balance(response)

    def parse_balance(self, response) -> Balances:
        timestamp = self.safe_integer_product(response, 'lastUpdateTime', 0.001)
        result: dict = {
            'info': response,
            'timestamp': timestamp,
            'datetime': self.iso8601(timestamp),
        }
        for i in range(0, len(response)):
            balance = response[i]
            symbol = self.safe_string(balance, 'symbol')
            code = self.safe_currency_code(symbol)
            account = self.account()
            account['total'] = self.safe_string(balance, 'balance')
            account['free'] = self.safe_string(balance, 'free')
            result[code] = account
        return self.safe_balance(result)

    async def create_deposit_address(self, code: str, params={}) -> DepositAddress:
        """
        create a currency deposit address

        https://arkm.com/docs#post/account/deposit/addresses/new

        :param str code: unified currency code of the currency for the deposit address
        :param dict [params]: extra parameters specific to the exchange API endpoint
        :returns dict: an `address structure <https://docs.ccxt.com/#/?id=address-structure>`
        """
        await self.load_markets()
        networkCode = None
        networkCode, params = self.handle_network_code_and_params(params)
        if networkCode is None:
            raise ArgumentsRequired(self.id + ' createDepositAddress() requires a "network" param')
        request: dict = {
            'chain': networkCode,
        }
        response = await self.v1PrivatePostAccountDepositAddressesNew(self.extend(request, params))
        #
        #    {
        #        "addresses": "12NauJ26TUT9aYkpId7YdePJJDRMGbAsEMVoTVUvBErV"
        #    }
        #
        address = self.safe_string(response, 'addresses')
        return self.parse_deposit_address(address, self.currency(code))

    async def fetch_deposit_addresses_by_network(self, code: str, params={}) -> List[DepositAddress]:
        """
        fetch the deposit addresses for a currency associated with self account

        https://arkm.com/docs#get/account/deposit/addresses

        :param str code: unified currency code
        :param dict [params]: extra parameters specific to the exchange API endpoint
        :returns dict: a dictionary `address structures <https://docs.ccxt.com/#/?id=address-structure>`, indexed by the network
        """
        await self.load_markets()
        networkCode = None
        networkCode, params = self.handle_network_code_and_params(params)
        if networkCode is None:
            raise ArgumentsRequired(self.id + ' fetchDepositAddressesByNetwork() requires a "network" param')
        request: dict = {
            'chain': self.network_code_to_id(networkCode),
        }
        response = await self.v1PrivateGetAccountDepositAddresses(self.extend(request, params))
        #
        #    {
        #        "addresses": [
        #            "12NauJ26TUT9aYkpId7YdePJJDRMGbAsEMVoTVUvBErV"
        #        ]
        #    }
        #
        data = self.safe_list(response, 'addresses')
        parsed = self.parse_deposit_addresses(data, None, False, {'network': networkCode})
        return self.index_by(parsed, 'network')

    def parse_deposit_address(self, entry, currency: Currency = None) -> DepositAddress:
        #
        #     "12NauJ26TUT9aYkpId7YdePJJDRMGbAsEMVoTVUvBErV"
        #
        return {
            'info': entry,
            'currency': self.safe_string(currency, 'code'),
            'network': None,
            'address': entry,
            'tag': None,
        }

    async def fetch_deposit_address(self, code: str, params={}) -> DepositAddress:
        """
        fetch the deposit address for a currency associated with self account

        https://arkm.com/docs#get/account/deposit/addresses

        :param str code: unified currency code
        :param dict [params]: extra parameters specific to the exchange API endpoint
        :returns dict: an `address structure <https://docs.ccxt.com/#/?id=address-structure>`
        """
        await self.load_markets()
        currency = self.currency(code)
        networkCodeAndParams = self.handle_network_code_and_params(params)
        networkCode = networkCodeAndParams[0]
        indexedAddresses = await self.fetch_deposit_addresses_by_network(code, params)
        selectedNetworkCode = self.select_network_code_from_unified_networks(currency['code'], networkCode, indexedAddresses)
        address = self.safe_dict(indexedAddresses, selectedNetworkCode)
        if address is None:
            raise InvalidAddress(self.id + ' fetchDepositAddress() could not find a deposit address for ' + code)
        return address

    async def fetch_deposits(self, code: Str = None, since: Int = None, limit: Int = None, params={}) -> List[Transaction]:
        """
        fetch all deposits made to an account

        https://arkm.com/docs#get/account/deposits

        :param str code: unified currency code
        :param int [since]: the earliest time in ms to fetch deposits for
        :param int [limit]: the maximum number of deposits structures to retrieve
        :param dict [params]: extra parameters specific to the exchange API endpoint
        :returns dict[]: a list of `transaction structures <https://docs.ccxt.com/#/?id=transaction-structure>`
        """
        await self.load_markets()
        request: dict = {}
        if limit is not None:
            request['limit'] = limit
        response = await self.v1PrivateGetAccountDeposits(self.extend(request, params))
        #
        #    [
        #        {
        #            "id": "238644",
        #            "symbol": "SOL",
        #            "amount": "0.104",
        #            "time": "1753436404000000",
        #            "confirmed": True,
        #            "transactionHash": "1DRxbbyePTsMuB82SDf2fG5gLXH5iYnY8TQDstDPLULpLtjMJtF1ug1T4Mf8B6DSb8fp2sb5YtdbyqieZ2tkE1Ve",
        #            "chain": "Solana",
        #            "depositAddress": "12NauJ26TUT9aYkpId7YdePJJDRMGbAsEMVoTVUvBErV",
        #            "price": "180.322010164"
        #        }
        #    ]
        #
        currency = None
        if code is not None:
            currency = self.currency(code)
        return self.parse_transactions(response, currency, since, limit)

    def parse_transaction(self, transaction: dict, currency: Currency = None) -> Transaction:
        #
        #        {
        #            "id": "238644",
        #            "symbol": "SOL",
        #            "amount": "0.104",
        #            "time": "1753436404000000",
        #            "confirmed": True,
        #            "transactionHash": "1DRxbbyePTsMuB82SDf2fG5gLXH5iYnY8TQDstDPLULpLtjMJtF1ug1T4Mf8B6DSb8fp2sb5YtdbyqieZ2tkE1Ve",
        #            "chain": "Solana",
        #            "depositAddress": "12NauJ26TUT9aYkpId7YdePJJDRMGbAsEMVoTVUvBErV",
        #            "price": "180.322010164"
        #        }
        #
        address = self.safe_string(transaction, 'depositAddress')
        timestamp = self.safe_integer_product(transaction, 'time', 0.001)
        confirmd = self.safe_bool(transaction, 'confirmed')
        status = None
        if confirmd:
            status = 'ok'
        currencyId = self.safe_string(transaction, 'symbol')
        code = self.safe_currency_code(currencyId, currency)
        return {
            'info': transaction,
            'id': self.safe_string(transaction, 'id'),
            'txid': self.safe_string(transaction, 'transactionHash'),
            'type': None,
            'currency': code,
            'network': self.network_id_to_code(self.safe_string(transaction, 'chain')),
            'amount': self.safe_number(transaction, 'amount'),
            'status': status,
            'timestamp': timestamp,
            'datetime': self.iso8601(timestamp),
            'address': address,
            'addressFrom': None,
            'addressTo': address,
            'tag': None,
            'tagFrom': None,
            'tagTo': None,
            'updated': None,
            'comment': None,
            'fee': None,
            'internal': False,
        }

    async def fetch_trading_fees(self, params={}) -> TradingFees:
        """
        fetch the trading fees for multiple markets

        https://arkm.com/docs#get/account/fees

        :param dict [params]: extra parameters specific to the exchange API endpoint
        :param str [params.subType]: "linear" or "inverse"
        :returns dict: a dictionary of `fee structures <https://docs.ccxt.com/#/?id=fee-structure>` indexed by market symbols
        """
        await self.load_markets()
        response = await self.v1PrivateGetAccountFees(params)
        #
        # {
        #   "perpMakerFee": "1.23",
        #   "perpTakerFee": "1.23",
        #   "spotMakerFee": "1.23",
        #   "spotTakerFee": "1.23"
        # }
        #
        symbols = list(self.markets.keys())
        result: dict = {}
        spotMaker = self.safe_number(response, 'spotMakerFee')
        spotTaker = self.safe_number(response, 'spotTakerFee')
        perpMaker = self.safe_number(response, 'perpMakerFee')
        perpTaker = self.safe_number(response, 'perpTakerFee')
        for i in range(0, len(symbols)):
            symbol = symbols[i]
            market = self.markets[symbol]
            result[symbol] = {
                'info': response,
                'symbol': symbol,
            }
            if market['spot']:
                result[symbol]['maker'] = spotMaker
                result[symbol]['taker'] = spotTaker
            elif market['swap'] or market['future']:
                result[symbol]['maker'] = perpMaker
                result[symbol]['taker'] = perpTaker
        return result

    async def fetch_funding_history(self, symbol: Str = None, since: Int = None, limit: Int = None, params={}):
        """
        fetch the history of funding payments paid and received on self account

        https://arkm.com/docs#get/account/funding-rate-payments

        :param str [symbol]: unified market symbol
        :param int [since]: the earliest time in ms to fetch funding history for
        :param int [limit]: the maximum number of funding history structures to retrieve
        :param dict [params]: extra parameters specific to the exchange API endpoint
        :param boolean [params.paginate]: default False, when True will automatically paginate by calling self endpoint multiple times. See in the docs all the [available parameters](https://github.com/ccxt/ccxt/wiki/Manual#pagination-params)
        :returns dict: a `funding history structure <https://docs.ccxt.com/#/?id=funding-history-structure>`
        """
        await self.load_markets()
        market = None
        if symbol is not None:
            market = self.market(symbol)
        request: dict = {}
        if limit is not None:
            request['limit'] = limit
        response = await self.v1PrivateGetAccountFundingRatePayments(self.extend(request, params))
        #
        #     [
        #         {
        #             "amount": "20.1",
        #             "assetSymbol": "BTC",
        #             "indexPrice": "1.23",
        #             "pairSymbol": "BTC_USDT",
        #             "time": 1704067200000000,
        #             "id": 1,
        #             "subaccountId": 1,
        #             "userId": 1
        #         },
        #         ...
        #     ]
        #
        return self.parse_incomes(response, market, since, limit)

    def parse_income(self, income, market: Market = None):
        #
        #         {
        #             "amount": "20.1",
        #             "assetSymbol": "BTC",
        #             "indexPrice": "1.23",
        #             "pairSymbol": "BTC_USDT",
        #             "time": 1704067200000000,
        #             "id": 1,
        #             "subaccountId": 1,
        #             "userId": 1
        #         }
        #
        marketId = self.safe_string(income, 'pairSymbol')
        currencyId = self.safe_string(income, 'assetSymbol')
        timestamp = self.safe_integer_product(income, 'time', 0.001)
        return {
            'info': income,
            'symbol': self.safe_symbol(marketId, market),
            'code': self.safe_currency_code(currencyId),
            'timestamp': timestamp,
            'datetime': self.iso8601(timestamp),
            'id': self.safe_string(income, 'id'),
            'amount': self.safe_number(income, 'amount'),
        }

    async def fetch_leverage(self, symbol: str, params={}) -> Leverage:
        """
        fetch the set leverage for a market

        https://arkm.com/docs#get/account/leverage

        :param str symbol: unified market symbol
        :param dict [params]: extra parameters specific to the exchange API endpoint
        :returns dict: a `leverage structure <https://docs.ccxt.com/#/?id=leverage-structure>`
        """
        await self.load_markets()
        market = self.market(symbol)
        marketId = self.safe_string(market, 'id')
        request: dict = {
            'symbol': marketId,
        }
        response = await self.v1PrivateGetAccountLeverage(self.extend(request, params))
        #
        # might be empty if not changed from default value(which is 1x)
        #
        #    [
        #        {
        #            "symbol": "BTC_USDT_PERP",
        #            "leverage": "7"
        #        },
        #        {
        #            "symbol": "ETH_USDT_PERP",
        #            "leverage": "5"
        #        }
        #    ]
        #
        indexed = self.index_by(response, 'symbol')
        data = self.safe_dict(indexed, marketId, {})
        return self.parse_leverage(data, market)

    def parse_leverage(self, leverage: dict, market: Market = None) -> Leverage:
        #
        #        {
        #            "symbol": "ETH_USDT_PERP",
        #            "leverage": "5"
        #        }
        #
        marketId = self.safe_string(leverage, 'symbol')
        leverageNum = self.safe_number(leverage, 'leverage')  # default leverage is 1 typically
        return {
            'info': leverage,
            'symbol': self.safe_symbol(marketId, market),
            'marginMode': None,
            'longLeverage': leverageNum,
            'shortLeverage': leverageNum,
        }

    async def set_leverage(self, leverage: int, symbol: Str = None, params={}):
        """
        set the level of leverage for a market

        https://arkm.com/docs#post/account/leverage

        :param float leverage: the rate of leverage
        :param str symbol: unified market symbol
        :param dict [params]: extra parameters specific to the exchange API endpoint
        :returns dict: response from the exchange
        """
        if symbol is None:
            raise ArgumentsRequired(self.id + ' setLeverage() requires a symbol argument')
        await self.load_markets()
        market = self.market(symbol)
        leverageString = self.number_to_string(leverage)
        marketId = self.safe_string(market, 'id')
        request: dict = {
            'symbol': marketId,
            'leverage': leverageString,
        }
        response = await self.v1PrivatePostAccountLeverage(self.extend(request, params))
        #
        # response is just empty string
        #
        return self.parse_leverage(response, market)

    async def fetch_positions(self, symbols: Strings = None, params={}) -> List[Position]:
        """
        fetch all open positions

        https://arkm.com/docs#get/account/positions

        :param str[]|None symbols: list of unified market symbols
        :param dict [params]: extra parameters specific to the exchange API endpoint
        :param boolean [params.standard]: whether to fetch standard contract positions
        :returns dict[]: a list of `position structures <https://docs.ccxt.com/#/?id=position-structure>`
        """
        await self.load_markets()
        symbols = self.market_symbols(symbols)
        response = await self.v1PrivateGetAccountPositions(params)
        #
        #    [
        #        {
        #            "subaccountId": "0",
        #            "symbol": "SOL_USDT_PERP",
        #            "base": "0.037",
        #            "quote": "-6.44614",
        #            "openBuySize": "0",
        #            "openSellSize": "0",
        #            "openBuyNotional": "0",
        #            "openSellNotional": "0",
        #            "lastUpdateReason": "orderFill",
        #            "lastUpdateTime": "1753903829389966",
        #            "lastUpdateId": "250434684",
        #            "lastUpdateBaseDelta": "0.037",
        #            "lastUpdateQuoteDelta": "-6.44614",
        #            "breakEvenPrice": "174.22",
        #            "markPrice": "174.33",
        #            "value": "6.45021",
        #            "pnl": "0.00407",
        #            "initialMargin": "0.645021",
        #            "maintenanceMargin": "0.3870126",
        #            "averageEntryPrice": "174.22"
        #        }
        #    ]
        #
        return self.parse_positions(response, symbols)

    def parse_position(self, position: dict, market: Market = None):
        #
        #        {
        #            "subaccountId": "0",
        #            "symbol": "SOL_USDT_PERP",
        #            "base": "0.037",                             # negative for short position
        #            "quote": "-6.44614",                         # negative for long position
        #            "openBuySize": "0",
        #            "openSellSize": "0",
        #            "openBuyNotional": "0",
        #            "openSellNotional": "0",
        #            "lastUpdateReason": "orderFill",
        #            "lastUpdateTime": "1753903829389966",
        #            "lastUpdateId": "250434684",
        #            "lastUpdateBaseDelta": "0.037",
        #            "lastUpdateQuoteDelta": "-6.44614",
        #            "breakEvenPrice": "174.22",
        #            "markPrice": "174.33",
        #            "value": "6.45021",
        #            "pnl": "0.00407",
        #            "initialMargin": "0.645021",
        #            "maintenanceMargin": "0.3870126",
        #            "averageEntryPrice": "174.22"
        #        }
        #
        base = self.safe_string(position, 'base')
        baseAbs = Precise.string_abs(base)
        isLong = Precise.string_ge(base, '0')
        side = 'long' if isLong else 'short'
        marketId = self.safe_string(position, 'symbol')
        notional = self.safe_string(position, 'value')
        return self.safe_position({
            'info': position,
            'id': None,
            'symbol': self.safe_symbol(marketId, market),
            'notional': self.parse_number(Precise.string_abs(notional)),
            'marginMode': None,
            'liquidationPrice': None,
            'entryPrice': self.safe_number(position, 'averageEntryPrice'),
            'unrealizedPnl': self.safe_number(position, 'pnl'),
            'realizedPnl': None,
            'percentage': None,
            'contracts': self.parse_number(baseAbs),
            'contractSize': None,
            'markPrice': self.safe_number(position, 'markPrice'),
            'lastPrice': None,
            'side': side,
            'hedged': None,
            'timestamp': None,
            'datetime': None,
            'lastUpdateTimestamp': self.safe_integer(position, 'lastUpdateTime'),
            'maintenanceMargin': self.safe_number(position, 'maintenanceMargin'),
            'maintenanceMarginPercentage': None,
            'collateral': None,
            'initialMargin': self.safe_number(position, 'initialMargin'),
            'initialMarginPercentage': None,
            'leverage': None,
            'marginRatio': None,
            'stopLossPrice': None,
            'takeProfitPrice': None,
        })

    async def withdraw(self, code: str, amount: float, address: str, tag: Str = None, params={}) -> Transaction:
        """
        make a withdrawal

        https://arkm.com/docs#post/account/withdraw
        https://arkm.com/docs#get/account/withdrawal/addresses

        :param str code: unified currency code
        :param float amount: the amount to withdraw
        :param str address: the address to withdraw to
        :param str tag:
        :param dict [params]: extra parameters specific to the exchange API endpoint
        :returns dict: a `transaction structure <https://docs.ccxt.com/#/?id=transaction-structure>`
        """
        tag, params = self.handle_withdraw_tag_and_params(tag, params)
        await self.load_markets()
        withdrawalAddresses = await self.v1PrivateGetAccountWithdrawalAddresses()
        #
        #    [
        #        {
        #            "id": "12345",
        #            "chain": "ETH",
        #            "address": "0x743f79D65EA07AA222F4a83c10dee4210A920a6e",
        #            "label": "my_binance",
        #            "createdAt": "1753905200074355",
        #            "updatedAt": "1753905213464278",
        #            "confirmed": True
        #        }
        #    ]
        #
        currency = self.currency(code)
        request: dict = {
            'symbol': currency['id'],
            'amount': self.currency_to_precision(code, amount),
            'subaccountId': self.safe_integer(params, 'subAccountId', 0),
        }
        networkCode = None
        networkCode, params = self.handle_network_code_and_params(params)
        if networkCode is None:
            raise ArgumentsRequired(self.id + ' withdraw() requires a "network" param')
        indexedList = self.group_by(withdrawalAddresses, 'address')
        if not (address in indexedList):
            raise InvalidAddress(self.id + ' withdraw() requires an address that has been previously added to the whitelisted addresses')
        withdrawalObjects = indexedList[address]
        foundWithdrawalObject = None
        for i in range(0, len(withdrawalObjects)):
            withdrawalObject = withdrawalObjects[i]
            if withdrawalObject['chain'] == networkCode:
                foundWithdrawalObject = withdrawalObject
                break
        if foundWithdrawalObject is None:
            raise InvalidAddress(self.id + ' withdraw() can not find whitelisted withdrawal address for ' + address + ' with network ' + networkCode)
        request['addressId'] = self.safe_integer(foundWithdrawalObject, 'id')
        response = await self.v1PrivatePostAccountWithdraw(self.extend(request, params))
        #
        # response is a weird string like:
        #
        #    "1234709779980\\n"
        #
        responseString = response.replace('\n', '')
        data = {'id': responseString}
        return self.parse_transaction(data, currency)

    async def fetch_leverage_tiers(self, symbols: Strings = None, params={}) -> LeverageTiers:
        """
        retrieve information on the maximum leverage, and maintenance margin for trades of varying trade sizes

        https://arkm.com/docs#get/public/margin-schedules

        :param str[]|None symbols: list of unified market symbols
        :param dict [params]: extra parameters specific to the exchange API endpoint
        :returns dict: a dictionary of `leverage tiers structures <https://docs.ccxt.com/#/?id=leverage-tiers-structure>`, indexed by market symbols
        """
        await self.load_markets()
        if symbols is None:
            raise ArgumentsRequired(self.id + ' fetchLeverageTiers() requires a symbols argument')
        symbols = self.market_symbols(symbols)
        response = await self.v1PublicGetMarginSchedules(params)
        #
        #    [
        #        {
        #            "name": "A",
        #            "bands": [
        #                {
        #                    "positionLimit": "1000000",
        #                    "leverageRate": "50",
        #                    "marginRate": "0.02",
        #                    "rebate": "0"
        #                },
        #                {
        #                    "positionLimit": "2000000",
        #                    "leverageRate": "25",
        #                    "marginRate": "0.04",
        #                    "rebate": "20000"
        #                },
        #                {
        #                    "positionLimit": "5000000",
        #                    "leverageRate": "20",
        #                    "marginRate": "0.05",
        #                    "rebate": "40000"
        #                }
        #            ]
        #        },
        #        {
        #            "name": "B",
        #            ...
        #
        return self.parse_leverage_tiers(response, symbols)

    def parse_leverage_tiers(self, response: Any, symbols: List[str] = None, marketIdKey=None) -> LeverageTiers:
        # overloaded method
        indexed = self.index_by(response, 'name')
        symbols = self.market_symbols(symbols)
        tiers = {}
        for i in range(0, len(symbols)):
            symbol = symbols[i]
            market = self.market(symbol)
            marginSchedule = self.safe_string(market['info'], 'marginSchedule')
            if marginSchedule is None:
                raise BadSymbol(self.id + ' fetchLeverageTiers() could not find marginSchedule for ' + symbol)
            selectedDict = self.safe_dict(indexed, marginSchedule, {})
            bands = self.safe_list(selectedDict, 'bands', [])
            tiers[symbol] = self.parse_market_leverage_tiers(bands, market)
        return tiers

    def parse_market_leverage_tiers(self, info, market: Market = None) -> List[LeverageTier]:
        tiers = []
        brackets = info
        minNotional = 0
        for i in range(0, len(brackets)):
            tier = brackets[i]
            marketId = self.safe_string(info, 'market')
            market = self.safe_market(marketId, market, None, 'swap')
            maxNotional = self.safe_number(tier, 'positionLimit')
            tiers.append({
                'tier': self.sum(i, 1),
                'symbol': self.safe_symbol(marketId, market, None, 'swap'),
                'currency': market['base'] if market['linear'] else market['quote'],
                'minNotional': minNotional,
                'maxNotional': maxNotional,
                'maintenanceMarginRate': self.safe_number(tier, 'marginRate'),
                'maxLeverage': self.safe_integer(tier, 'leverageRate'),
                'info': tier,
            })
            minNotional = maxNotional
        return tiers

    def find_timeframe_by_duration(self, duration: Int) -> str:
        # self method is used to find the timeframe by duration in seconds
        timeframes = self.safe_dict(self.options, 'timeframeDurations', {})
        keys = list(timeframes.keys())
        for i in range(0, len(keys)):
            timeframe = keys[i]
            durationInMicroseconds = self.safe_integer(timeframes, timeframe)
            if durationInMicroseconds == duration:
                return timeframe
        return None

    def sign(self, path, api='public', method='GET', params={}, headers=None, body=None):
        type = self.safe_string(api, 0)
        access = self.safe_string(api, 1)
        accessPart = access + '/' if (access == 'public') else ''
        query = self.omit(params, self.extract_params(path))
        path = self.implode_params(path, params)
        url = self.urls['api'][type] + '/' + accessPart + path
        queryString = ''
        if method == 'GET':
            if query:
                queryString = self.urlencode(query)
                url += '?' + queryString
        if access == 'private':
            self.check_required_credentials()
            expires = (self.milliseconds() + self.safe_integer(self.options, 'requestExpiration', 5000)) * 1000  # need macroseconds
            if method == 'POST':
                body = self.json(params)
            if queryString != '':
                path = path + '?' + queryString
            bodyStr = body if (body is not None) else ''
            payload = self.apiKey + str(expires) + method.upper() + '/' + path + bodyStr
            decodedSecret = self.base64_to_binary(self.secret)
            signature = self.hmac(self.encode(payload), decodedSecret, hashlib.sha256, 'base64')
            headers = {
                'Content-Type': 'application/json',
                'Accept': 'application/json',
                'Arkham-Api-Key': self.apiKey,
                'Arkham-Expires': str(expires),
                'Arkham-Signature': signature,
                'Arkham-Broker-Id': '1001',
            }
        return {'url': url, 'method': method, 'body': body, 'headers': headers}

    def handle_errors(self, code: int, reason: str, url: str, method: str, headers: dict, body: str, response, requestHeaders, requestBody):
        #
        # error example:
        #
        #    {
        #        "id": "30005",
        #        "name": "InvalidNotional",
        #        "message": "order validation failed: invalid notional: notional 0.25 is less than min notional 1"
        #    }
        #
        message = self.safe_string(response, 'message')
        if message is not None:
            errorCode = self.safe_string(response, 'id')
            feedback = self.id + ' ' + body
            self.throw_exactly_matched_exception(self.exceptions['exact'], errorCode, feedback)
            self.throw_exactly_matched_exception(self.exceptions['exact'], message, feedback)
            self.throw_broadly_matched_exception(self.exceptions['broad'], message, feedback)
            raise ExchangeError(self.id + ' ' + body)
        return None
