""" mplsoccer formations. """

from dataclasses import dataclass, field

from typing import Optional, List

STATSBOMB_POSITIONS = {1: 'GK',
                       2: 'RB',
                       3: 'RCB',
                       4: 'CB',
                       5: 'LCB',
                       6: 'LB',
                       7: 'RWB',
                       8: 'LWB',
                       9: 'RDM',
                       10: 'CDM',
                       11: 'LDM',
                       12: 'RM',
                       13: 'RCM',
                       14: 'CM',
                       15: 'LCM',
                       16: 'LM',
                       17: 'RW',
                       18: 'RAM',
                       19: 'CAM',
                       20: 'LAM',
                       21: 'LW',
                       22: 'RCF',
                       23: 'ST',
                       24: 'LCF',
                       25: 'SS',
                       }


@dataclass
class Coordinate:
    """ Dataclass for holding coordinates (x, y)"""
    x: float
    y: float
    x_flip: float
    y_flip: float
    x_half: float
    y_half: float
    x_half_flip: float
    y_half_flip: float


@dataclass
class PositionLine4:
    """ Dataclass for positions (x, y) coordinates in lines of four."""
    GK: Coordinate
    RB: Coordinate
    RCB: Coordinate
    LCB: Coordinate
    LB: Coordinate
    RWB: Coordinate
    RDM: Coordinate
    LDM: Coordinate
    LWB: Coordinate
    RM: Coordinate
    RCM: Coordinate
    LCM: Coordinate
    LM: Coordinate
    RW: Coordinate
    RAM: Coordinate
    LAM: Coordinate
    LW: Coordinate
    RCF: Coordinate
    LCF: Coordinate


@dataclass
class PositionLine5:
    """ Dataclass for positions (x, y) coordinates in lines of five."""
    GK: Coordinate
    RB: Coordinate
    RCB: Coordinate
    CB: Coordinate
    LCB: Coordinate
    LB: Coordinate
    RWB: Coordinate
    RDM: Coordinate
    CDM: Coordinate
    LDM: Coordinate
    LWB: Coordinate
    RM: Coordinate
    RCM: Coordinate
    CM: Coordinate
    LCM: Coordinate
    LM: Coordinate
    RW: Coordinate
    RAM: Coordinate
    CAM: Coordinate
    LAM: Coordinate
    LW: Coordinate
    RCF: Coordinate
    ST: Coordinate
    LCF: Coordinate


@dataclass
class PositionLine5WithSecondStriker:
    """ Dataclass for positions (x, y) coordinates in lines of five.
    Including a line for a second striker between the attacking midfielders
    and the forwards. This extra line pushes the attacking midfield back."""
    GK: Coordinate
    RB: Coordinate
    RCB: Coordinate
    CB: Coordinate
    LCB: Coordinate
    LB: Coordinate
    RWB: Coordinate
    RDM: Coordinate
    CDM: Coordinate
    LDM: Coordinate
    LWB: Coordinate
    RM: Coordinate
    RCM: Coordinate
    CM: Coordinate
    LCM: Coordinate
    LM: Coordinate
    RW: Coordinate
    RAM: Coordinate
    CAM: Coordinate
    LAM: Coordinate
    LW: Coordinate
    RCF: Coordinate
    ST: Coordinate
    LCF: Coordinate
    SS: Coordinate


@dataclass
class Position:
    """ Dataclass for position."""
    name: str
    location: Coordinate = field(repr=False)
    statsbomb: Optional[List[int]] = None
    opta: Optional[int] = None
    wyscout: Optional[str] = None
    x: float = None
    y: float = None
    x_flip: float = None
    y_flip: float = None
    x_half: float = None
    y_half: float = None
    x_half_flip: float = None
    y_half_flip: float = None

    def __post_init__(self):
        self.x = self.location.x
        self.y = self.location.y
        self.x_flip = self.location.x_flip
        self.y_flip = self.location.y_flip
        self.x_half = self.location.x_half
        self.y_half = self.location.y_half
        self.x_half_flip = self.location.x_half_flip
        self.y_half_flip = self.location.y_half_flip


@dataclass
class Formation:
    """ Dataclass for formations."""
    position_line4: PositionLine4
    position_line5: PositionLine5
    position_line4_with_ss: PositionLine4
    position_line5_with_ss: PositionLine5WithSecondStriker

    def __post_init__(self):
        """ Create formations from the player positions."""
        self.formations = {
            '442': [Position('GK', self.position_line5.GK, statsbomb=[1], opta=1, wyscout='gk'),
                    Position('RB', self.position_line4.RB, statsbomb=[2], opta=2, wyscout='rb'),
                    Position('RCB', self.position_line4.RCB, statsbomb=[3], opta=5, wyscout='rcb'),
                    Position('LCB', self.position_line4.LCB, statsbomb=[5], opta=6, wyscout='lcb'),
                    Position('LB', self.position_line4.LB, statsbomb=[6], opta=3, wyscout='lb'),
                    Position('RM', self.position_line4.RM, statsbomb=[12, 17], opta=7,
                             wyscout='rw'),
                    Position('RCM', self.position_line4.RCM, statsbomb=[13, 9], opta=4,
                             wyscout='rcmf'),
                    Position('LCM', self.position_line4.LCM, statsbomb=[15, 11], opta=8,
                             wyscout='lcmf'),
                    Position('LM', self.position_line4.LM, statsbomb=[16, 21], opta=11,
                             wyscout='lw'),
                    Position('RCF', self.position_line4.RCF, statsbomb=[22], opta=10, wyscout='ss'),
                    Position('LCF', self.position_line4.LCF, statsbomb=[24], opta=9, wyscout='cf'),
                    ],
            '41212': [Position('GK', self.position_line5.GK, statsbomb=[1], opta=1),
                      Position('RB', self.position_line4.RB, statsbomb=[2], opta=2),
                      Position('RCB', self.position_line4.RCB, statsbomb=[3], opta=5),
                      Position('LCB', self.position_line4.LCB, statsbomb=[5], opta=6),
                      Position('LB', self.position_line4.LB, statsbomb=[6], opta=3),
                      Position('CDM', self.position_line5.CDM, statsbomb=[10], opta=4),
                      Position('RCM', self.position_line5.RCM, statsbomb=[13, 12], opta=7),
                      Position('LCM', self.position_line5.LCM, statsbomb=[15, 16], opta=11),
                      Position('CAM', self.position_line5.CAM, statsbomb=[19, 25], opta=8),
                      Position('RCF', self.position_line4.RCF, statsbomb=[22], opta=10),
                      Position('LCF', self.position_line4.LCF, statsbomb=[24], opta=9),
                      ],
            '433': [Position('GK', self.position_line5.GK, statsbomb=[1], opta=1, wyscout='gk'),
                    Position('RB', self.position_line4.RB, statsbomb=[2], opta=2, wyscout='rb'),
                    Position('RCB', self.position_line4.RCB, statsbomb=[3], opta=5, wyscout='rcb'),
                    Position('LCB', self.position_line4.LCB, statsbomb=[5], opta=6, wyscout='lcb'),
                    Position('LB', self.position_line4.LB, statsbomb=[6], opta=3, wyscout='lb'),
                    Position('CDM', self.position_line5.CDM, statsbomb=[10, 14], opta=4,
                             wyscout='dmf'),
                    Position('RCM', self.position_line5.RCM, statsbomb=[13, 12, 9], opta=7,
                             wyscout='rcmf3'),
                    Position('LCM', self.position_line5.LCM, statsbomb=[15, 16, 11], opta=8,
                             wyscout='lcmf3'),
                    Position('RW', self.position_line5.RW, statsbomb=[17, 22], opta=10,
                             wyscout='rwf'),
                    Position('LW', self.position_line5.LW, statsbomb=[21, 24], opta=11,
                             wyscout='lwf'),
                    Position('ST', self.position_line5.ST, statsbomb=[23, 25, 19], opta=9,
                             wyscout='cf'),
                    ],
            '451': [Position('GK', self.position_line5.GK, statsbomb=[1], opta=1, wyscout='gk'),
                    Position('RB', self.position_line4.RB, statsbomb=[2], opta=2, wyscout='rb'),
                    Position('RCB', self.position_line4.RCB, statsbomb=[3], opta=5, wyscout='rcb'),
                    Position('LCB', self.position_line4.LCB, statsbomb=[5], opta=6, wyscout='lcb'),
                    Position('LB', self.position_line4.LB, statsbomb=[6], opta=3, wyscout='lb'),
                    Position('RM', self.position_line5.RM, statsbomb=[12, 17], opta=7,
                             wyscout='rw'),
                    Position('RCM', self.position_line5.RCM, statsbomb=[13, 9], opta=4,
                             wyscout='rcmf3'),
                    Position('CM', self.position_line5.CM, statsbomb=[10, 14, 19], opta=10,
                             wyscout='dmf'),
                    Position('LCM', self.position_line5.LCM, statsbomb=[15, 11], opta=8,
                             wyscout='lcmf3'),
                    Position('LM', self.position_line5.LM, statsbomb=[16, 21], opta=11,
                             wyscout='lw'),
                    Position('ST', self.position_line5.ST, statsbomb=[23, 25], opta=9,
                             wyscout='cf'),
                    ],
            '4411': [Position('GK', self.position_line5.GK, statsbomb=[1], opta=1, wyscout='gk'),
                     Position('RB', self.position_line4.RB, statsbomb=[2], opta=2, wyscout='rb'),
                     Position('RCB', self.position_line4.RCB, statsbomb=[3], opta=5, wyscout='rcb'),
                     Position('LCB', self.position_line4.LCB, statsbomb=[5], opta=6, wyscout='lcb'),
                     Position('LB', self.position_line4.LB, statsbomb=[6], opta=3, wyscout='lb'),
                     Position('RM', self.position_line4.RM, statsbomb=[12, 17], opta=7,
                              wyscout='rw'),
                     Position('RCM', self.position_line4.RCM, statsbomb=[9, 13], opta=4,
                              wyscout='rcmf'),
                     Position('LCM', self.position_line4.LCM, statsbomb=[11, 15], opta=8,
                              wyscout='lcmf'),
                     Position('LM', self.position_line4.LM, statsbomb=[16, 21], opta=11,
                              wyscout='lw'),
                     Position('CAM', self.position_line5.CAM, statsbomb=[19, 25], opta=10,
                              wyscout='amf'),
                     Position('ST', self.position_line5.ST, statsbomb=[23], opta=9, wyscout='cf'),
                     ],
            '4141': [Position('GK', self.position_line5.GK, statsbomb=[1], opta=1, wyscout='gk'),
                     Position('RB', self.position_line4.RB, statsbomb=[2], opta=2, wyscout='rb'),
                     Position('RCB', self.position_line4.RCB, statsbomb=[3], opta=5, wyscout='rcb'),
                     Position('LCB', self.position_line4.LCB, statsbomb=[5], opta=6, wyscout='lcb'),
                     Position('LB', self.position_line4.LB, statsbomb=[6], opta=3, wyscout='lb'),
                     Position('CDM', self.position_line5.CDM, statsbomb=[10, 14], opta=4,
                              wyscout='dmf'),
                     Position('RM', self.position_line4.RM, statsbomb=[12, 17], opta=7,
                              wyscout='rw'),
                     Position('RCM', self.position_line4.RCM, statsbomb=[13, 18], opta=8,
                              wyscout='rcmf'),
                     Position('LCM', self.position_line4.LCM, statsbomb=[15, 20], opta=10,
                              wyscout='lcmf'),
                     Position('LM', self.position_line4.LM, statsbomb=[16, 21], opta=11,
                              wyscout='lw'),
                     Position('ST', self.position_line5.ST, statsbomb=[23], opta=9, wyscout='cf'),
                     ],
            '4231': [Position('GK', self.position_line5.GK, statsbomb=[1], opta=1, wyscout='gk'),
                     Position('RB', self.position_line4.RB, statsbomb=[2], opta=2, wyscout='rb'),
                     Position('RCB', self.position_line4.RCB, statsbomb=[3], opta=5, wyscout='rcb'),
                     Position('LCB', self.position_line4.LCB, statsbomb=[5], opta=6, wyscout='lcb'),
                     Position('LB', self.position_line4.LB, statsbomb=[6], opta=3, wyscout='lb'),
                     Position('RDM', self.position_line5.RDM, statsbomb=[9, 13], opta=8,
                              wyscout='rdmf'),
                     Position('LDM', self.position_line5.LDM, statsbomb=[11, 15], opta=4,
                              wyscout='ldmf'),
                     Position('RW', self.position_line5.RW, statsbomb=[17, 12, 18], opta=7,
                              wyscout='ramf'),
                     Position('CAM', self.position_line5.CAM, statsbomb=[19], opta=10,
                              wyscout='amf'),
                     Position('LW', self.position_line5.LW, statsbomb=[21, 16, 20], opta=11,
                              wyscout='lamf'),
                     Position('ST', self.position_line5.ST, statsbomb=[23], opta=9, wyscout='cf'),
                     ],
            '4321': [Position('GK', self.position_line5.GK, statsbomb=[1], opta=1, wyscout='gk'),
                     Position('RB', self.position_line4.RB, statsbomb=[2], opta=2, wyscout='rb'),
                     Position('RCB', self.position_line4.RCB, statsbomb=[3], opta=5, wyscout='rcb'),
                     Position('LCB', self.position_line4.LCB, statsbomb=[5], opta=6, wyscout='lcb'),
                     Position('LB', self.position_line4.LB, statsbomb=[6], opta=3, wyscout='lb'),
                     Position('RCM', self.position_line5.RCM, statsbomb=[13, 12], opta=8,
                              wyscout='rcmf3'),
                     Position('CM', self.position_line5.CM, statsbomb=[14, 10], opta=4,
                              wyscout='dmf'),
                     Position('LCM', self.position_line5.LCM, statsbomb=[15, 16], opta=7,
                              wyscout='lcmf3'),
                     Position('RW', self.position_line5.RW, statsbomb=[17, 18], opta=10,
                              wyscout='ramf'),
                     Position('LW', self.position_line5.LW, statsbomb=[21, 20], opta=11,
                              wyscout='lamf'),
                     Position('ST', self.position_line5.ST, statsbomb=[23, 25], opta=9,
                              wyscout='cf'),
                     ],
            '532': [Position('GK', self.position_line5.GK, statsbomb=[1], opta=1, wyscout='gk'),
                    Position('RB', self.position_line5.RB, statsbomb=[2, 7], opta=2, wyscout='rb5'),
                    Position('RCB', self.position_line5.RCB, statsbomb=[3], opta=6, wyscout='rcb3'),
                    Position('CB', self.position_line5.CB, statsbomb=[4], opta=5, wyscout='cb'),
                    Position('LCB', self.position_line5.LCB, statsbomb=[5], opta=4, wyscout='lcb3'),
                    Position('LB', self.position_line5.LB, statsbomb=[6, 8], opta=3, wyscout='lb5'),
                    Position('CDM', self.position_line5.CDM, statsbomb=[10, 14], opta=8,
                             wyscout='dmf'),
                    Position('RCM', self.position_line5.RCM, statsbomb=[13, 12, 18], opta=7,
                             wyscout='rcmf3'),
                    Position('LCM', self.position_line5.LCM, statsbomb=[15, 16, 20], opta=11,
                             wyscout='lcmf3'),
                    Position('RCF', self.position_line4.RCF, statsbomb=[22], opta=10, wyscout='ss'),
                    Position('LCF', self.position_line4.LCF, statsbomb=[24], opta=9, wyscout='cf'),
                    ],
            '541': [Position('GK', self.position_line5.GK, statsbomb=[1], opta=1, wyscout='gk'),
                    Position('RCB', self.position_line5.RCB, statsbomb=[3], opta=6, wyscout='rcb3'),
                    Position('CB', self.position_line5.CB, statsbomb=[4], opta=5, wyscout='cb'),
                    Position('LCB', self.position_line5.LCB, statsbomb=[5], opta=4, wyscout='lcb3'),
                    Position('RB', self.position_line5.RB, statsbomb=[2, 7], opta=2, wyscout='rb5'),
                    Position('LB', self.position_line5.LB, statsbomb=[6, 8], opta=3, wyscout='lb5'),
                    Position('RM', self.position_line4.RM, statsbomb=[12, 18, 17], opta=7,
                             wyscout='rw'),
                    Position('RCM', self.position_line4.RCM, statsbomb=[13, 9], opta=8,
                             wyscout='rcmf'),
                    Position('LCM', self.position_line4.LCM, statsbomb=[15, 11], opta=10,
                             wyscout='lcmf'),
                    Position('LM', self.position_line4.LM, statsbomb=[16, 20, 21], opta=11,
                             wyscout='lw'),
                    Position('ST', self.position_line5.ST, statsbomb=[23], opta=9, wyscout='cf'),
                    ],
            '352': [Position('GK', self.position_line5.GK, statsbomb=[1], opta=1, wyscout='gk'),
                    Position('RCB', self.position_line5.RCB, statsbomb=[3], opta=6, wyscout='rcb3'),
                    Position('CB', self.position_line5.CB, statsbomb=[4], opta=5, wyscout='cb'),
                    Position('LCB', self.position_line5.LCB, statsbomb=[5], opta=4, wyscout='lcb3'),
                    Position('RWB', self.position_line5.RWB, statsbomb=[7, 12, 17], opta=2,
                             wyscout='rwb'),
                    Position('LWB', self.position_line5.LWB, statsbomb=[8, 16, 21], opta=3,
                             wyscout='lwb'),
                    Position('RCM', self.position_line5.RCM, statsbomb=[13, 18], opta=7,
                             wyscout='rcmf3'),
                    Position('CM', self.position_line5.CM, statsbomb=[10, 14], opta=11,
                             wyscout='dmf'),
                    Position('LCM', self.position_line5.LCM, statsbomb=[15, 20], opta=8,
                             wyscout='lcmf3'),
                    Position('RCF', self.position_line4.RCF, statsbomb=[22], opta=10, wyscout='ss'),
                    Position('LCF', self.position_line4.LCF, statsbomb=[24], opta=9, wyscout='cf'),
                    ],
            '532flat': [Position('GK', self.position_line5.GK, statsbomb=[1], opta=1, wyscout='gk'),
                    Position('RCB', self.position_line5.RCB, statsbomb=[3], opta=6, wyscout='rcb3'),
                    Position('CB', self.position_line5.CB, statsbomb=[4], opta=5, wyscout='cb'),
                    Position('LCB', self.position_line5.LCB, statsbomb=[5], opta=4, wyscout='lcb3'),
                    Position('RB', self.position_line5.RB, statsbomb=[7, 12, 17], opta=2,
                             wyscout='rwb'),
                    Position('LB', self.position_line5.LB, statsbomb=[8, 16, 21], opta=3,
                             wyscout='lwb'),
                    Position('RCM', self.position_line5.RCM, statsbomb=[13, 18], opta=7,
                             wyscout='rcmf3'),
                    Position('CM', self.position_line5.CM, statsbomb=[10, 14], opta=11,
                             wyscout='dmf'),
                    Position('LCM', self.position_line5.LCM, statsbomb=[15, 20], opta=8,
                             wyscout='lcmf3'),
                    Position('RCF', self.position_line4.RCF, statsbomb=[22], opta=10, wyscout='ss'),
                    Position('LCF', self.position_line4.LCF, statsbomb=[24], opta=9, wyscout='cf'),
                    ],
            '343': [Position('GK', self.position_line5.GK, statsbomb=[1], opta=1, wyscout='gk'),
                    Position('RCB', self.position_line5.RCB, statsbomb=[3], opta=6, wyscout='rcb3'),
                    Position('CB', self.position_line5.CB, statsbomb=[4], opta=5, wyscout='cb'),
                    Position('LCB', self.position_line5.LCB, statsbomb=[5], opta=4, wyscout='lcb3'),
                    Position('RWB', self.position_line5.RWB, statsbomb=[7, 12], opta=2,
                             wyscout='rwb'),
                    Position('LWB', self.position_line5.LWB, statsbomb=[8, 16], opta=3,
                             wyscout='lwb'),
                    Position('RCM', self.position_line5.RCM, statsbomb=[9, 13], opta=7,
                             wyscout='rcmf'),
                    Position('LCM', self.position_line5.LCM, statsbomb=[11, 15], opta=8,
                             wyscout='lcmf'),
                    Position('RW', self.position_line5.RW, statsbomb=[17, 22, 18], opta=10,
                             wyscout='rwf'),
                    Position('LW', self.position_line5.LW, statsbomb=[21, 24, 20], opta=11,
                             wyscout='lwf'),
                    Position('ST', self.position_line5.ST, statsbomb=[23, 25], opta=9,
                             wyscout='cf'),
                    ],
            '343flat': [Position('GK', self.position_line5.GK, statsbomb=[1], opta=1, wyscout='gk'),
                    Position('RCB', self.position_line5.RCB, statsbomb=[3], opta=6, wyscout='rcb3'),
                    Position('CB', self.position_line5.CB, statsbomb=[4], opta=5, wyscout='cb'),
                    Position('LCB', self.position_line5.LCB, statsbomb=[5], opta=4, wyscout='lcb3'),
                    Position('RM', self.position_line5.RM, statsbomb=[7, 12], opta=2,
                             wyscout='rwb'),
                    Position('LM', self.position_line5.LM, statsbomb=[8, 16], opta=3,
                             wyscout='lwb'),
                    Position('RCM', self.position_line5.RCM, statsbomb=[9, 13], opta=7,
                             wyscout='rcmf'),
                    Position('LCM', self.position_line5.LCM, statsbomb=[11, 15], opta=8,
                             wyscout='lcmf'),
                    Position('RW', self.position_line5.RW, statsbomb=[17, 22, 18], opta=10,
                             wyscout='rwf'),
                    Position('LW', self.position_line5.LW, statsbomb=[21, 24, 20], opta=11,
                             wyscout='lwf'),
                    Position('ST', self.position_line5.ST, statsbomb=[23, 25], opta=9,
                             wyscout='cf'),
                    ],
            '31312': [Position('GK', self.position_line5_with_ss.GK, opta=1),
                      Position('RCB', self.position_line5_with_ss.RCB, opta=6),
                      Position('CB', self.position_line5_with_ss.CB, opta=5),
                      Position('LCB', self.position_line5_with_ss.LCB, opta=4),
                      Position('CDM', self.position_line5_with_ss.CDM, opta=7),
                      Position('RCM', self.position_line5_with_ss.RCM, opta=2),
                      Position('CM', self.position_line5_with_ss.CM, opta=8),
                      Position('LCM', self.position_line5_with_ss.LCM, opta=3),
                      Position('RCF', self.position_line5_with_ss.RCF, opta=10),
                      Position('LCF', self.position_line5_with_ss.LCF, opta=11),
                      Position('SS', self.position_line5_with_ss.SS, opta=9),
                      ],
            '4222': [Position('GK', self.position_line5.GK, statsbomb=[1], opta=1, wyscout='gk'),
                     Position('RB', self.position_line4.RB, statsbomb=[2], opta=2, wyscout='rb'),
                     Position('RCB', self.position_line4.RCB, statsbomb=[3], opta=5, wyscout='rcb'),
                     Position('LCB', self.position_line4.LCB, statsbomb=[5], opta=6, wyscout='lcb'),
                     Position('LB', self.position_line4.LB, statsbomb=[6], opta=3, wyscout='lb'),
                     Position('RCM', self.position_line5.RCM, statsbomb=[13, 9], opta=4,
                              wyscout='rdmf'),
                     Position('LCM', self.position_line5.LCM, statsbomb=[15, 11], opta=8,
                              wyscout='ldmf'),
                     Position('RW', self.position_line5.RW, statsbomb=[17, 12, 18], opta=7,
                              wyscout='ramf'),
                     Position('LW', self.position_line5.LW, statsbomb=[21, 16, 20], opta=11,
                              wyscout='lamf'),
                     Position('RCF', self.position_line5.RCF, statsbomb=[22], opta=10,
                              wyscout='ss'),
                     Position('LCF', self.position_line5.LCF, statsbomb=[24], opta=9, wyscout='cf'),
                     ],
            '3511': [Position('GK', self.position_line5.GK, statsbomb=[1], opta=1, wyscout='gk'),
                     Position('RCB', self.position_line5.RCB, statsbomb=[3], opta=5,
                              wyscout='rcb3'),
                     Position('CB', self.position_line5.CB, statsbomb=[4], opta=6, wyscout='cb'),
                     Position('LCB', self.position_line5.LCB, statsbomb=[5], opta=4,
                              wyscout='lcb3'),
                     Position('RWB', self.position_line5.RWB, statsbomb=[7, 12], opta=2,
                              wyscout='rcmf3'),
                     Position('CDM', self.position_line5.CDM, statsbomb=[10, 14], opta=11,
                              wyscout='dmf'),
                     Position('LWB', self.position_line5.LWB, statsbomb=[8, 16], opta=3,
                              wyscout='lcmf3'),
                     Position('RCM', self.position_line5.RCM, statsbomb=[13, 9], opta=7,
                              wyscout='rw'),
                     Position('LCM', self.position_line5.LCM, statsbomb=[15, 11], opta=8,
                              wyscout='lw'),
                     Position('CAM', self.position_line5.CAM, statsbomb=[19, 25], opta=10,
                              wyscout='amf'),
                     Position('ST', self.position_line5.ST, statsbomb=[23], opta=9, wyscout='cf'),
                     ],
            '3511flat': [Position('GK', self.position_line5.GK, statsbomb=[1], opta=1, wyscout='gk'),
                     Position('RCB', self.position_line5.RCB, statsbomb=[3], opta=5,
                              wyscout='rcb3'),
                     Position('CB', self.position_line5.CB, statsbomb=[4], opta=6, wyscout='cb'),
                     Position('LCB', self.position_line5.LCB, statsbomb=[5], opta=4,
                              wyscout='lcb3'),
                     Position('RM', self.position_line5.RM, statsbomb=[7, 12], opta=2,
                              wyscout='rcmf3'),
                     Position('CDM', self.position_line5.CDM, statsbomb=[10, 14], opta=11,
                              wyscout='dmf'),
                     Position('LM', self.position_line5.LM, statsbomb=[8, 16], opta=3,
                              wyscout='lcmf3'),
                     Position('RCM', self.position_line5.RCM, statsbomb=[13, 9], opta=7,
                              wyscout='rw'),
                     Position('LCM', self.position_line5.LCM, statsbomb=[15, 11], opta=8,
                              wyscout='lw'),
                     Position('CAM', self.position_line5.CAM, statsbomb=[19, 25], opta=10,
                              wyscout='amf'),
                     Position('ST', self.position_line5.ST, statsbomb=[23], opta=9, wyscout='cf'),
                     ],
            '3421': [Position('GK', self.position_line5.GK, statsbomb=[1], opta=1, wyscout='gk'),
                     Position('RCB', self.position_line5.RCB, statsbomb=[3], opta=6,
                              wyscout='rcb3'),
                     Position('CB', self.position_line5.CB, statsbomb=[4], opta=5, wyscout='cb'),
                     Position('LCB', self.position_line5.LCB, statsbomb=[5], opta=4,
                              wyscout='lcb3'),
                     Position('RWB', self.position_line4.RWB, statsbomb=[7, 12], opta=2,
                              wyscout='rwb'),
                     Position('RDM', self.position_line4.RDM, statsbomb=[9, 13], opta=7,
                              wyscout='rcmf'),
                     Position('LDM', self.position_line4.LDM, statsbomb=[11, 15], opta=8,
                              wyscout='lcmf'),
                     Position('LWB', self.position_line4.LWB, statsbomb=[8, 16], opta=3,
                              wyscout='lwb'),
                     Position('RAM', self.position_line4.RAM, statsbomb=[18, 17], opta=10,
                              wyscout='ramf'),
                     Position('LAM', self.position_line4.LAM, statsbomb=[20, 21], opta=11,
                              wyscout='lamf'),
                     Position('ST', self.position_line5.ST, statsbomb=[23], opta=9, wyscout='cf'),
                     ],
            '3421flat': [
                     Position('GK', self.position_line5.GK, statsbomb=[1], opta=1, wyscout='gk'),
                     Position('RCB', self.position_line5.RCB, statsbomb=[3], opta=6,
                              wyscout='rcb3'),
                     Position('CB', self.position_line5.CB, statsbomb=[4], opta=5, wyscout='cb'),
                     Position('LCB', self.position_line5.LCB, statsbomb=[5], opta=4,
                              wyscout='lcb3'),
                     Position('RM', self.position_line4.RM, statsbomb=[7, 12], opta=2,
                              wyscout='rwb'),
                     Position('RDM', self.position_line4.RDM, statsbomb=[9, 13], opta=7,
                              wyscout='rcmf'),
                     Position('LDM', self.position_line4.LDM, statsbomb=[11, 15], opta=8,
                              wyscout='lcmf'),
                     Position('LM', self.position_line4.LM, statsbomb=[8, 16], opta=3,
                              wyscout='lwb'),
                     Position('RAM', self.position_line4.RAM, statsbomb=[18, 17], opta=10,
                              wyscout='ramf'),
                     Position('LAM', self.position_line4.LAM, statsbomb=[20, 21], opta=11,
                              wyscout='lamf'),
                     Position('ST', self.position_line5.ST, statsbomb=[23], opta=9, wyscout='cf'),
                     ],
            '3412': [Position('GK', self.position_line5.GK, statsbomb=[1], opta=1, wyscout='gk'),
                     Position('RCB', self.position_line5.RCB, statsbomb=[3], opta=6,
                              wyscout='rcb3'),
                     Position('CB', self.position_line5.CB, statsbomb=[4], opta=5, wyscout='cb'),
                     Position('LCB', self.position_line5.LCB, statsbomb=[5], opta=4,
                              wyscout='lcb3'),
                     Position('RWB', self.position_line5.RWB, statsbomb=[7, 12], opta=2,
                              wyscout='rwb'),
                     Position('LWB', self.position_line5.LWB, statsbomb=[8, 16], opta=3,
                              wyscout='lwb'),
                     Position('RCM', self.position_line5.RCM, statsbomb=[9, 13], opta=7,
                              wyscout='rcmf'),
                     Position('LCM', self.position_line5.LCM, statsbomb=[11, 15], opta=8,
                              wyscout='lcmf'),
                     Position('CAM', self.position_line5.CAM, statsbomb=[19], opta=9,
                              wyscout='amf'),
                     Position('RCF', self.position_line5.RCF, statsbomb=[22], opta=10,
                              wyscout='ss'),
                     Position('LCF', self.position_line5.LCF, statsbomb=[24], opta=11,
                              wyscout='cf'),
                     ],
            '3412flat': [Position('GK', self.position_line5.GK, statsbomb=[1], opta=1, wyscout='gk'),
                     Position('RCB', self.position_line5.RCB, statsbomb=[3], opta=6,
                              wyscout='rcb3'),
                     Position('CB', self.position_line5.CB, statsbomb=[4], opta=5, wyscout='cb'),
                     Position('LCB', self.position_line5.LCB, statsbomb=[5], opta=4,
                              wyscout='lcb3'),
                     Position('RM', self.position_line5.RM, statsbomb=[7, 12], opta=2,
                              wyscout='rwb'),
                     Position('LM', self.position_line5.LM, statsbomb=[8, 16], opta=3,
                              wyscout='lwb'),
                     Position('RCM', self.position_line5.RCM, statsbomb=[9, 13], opta=7,
                              wyscout='rcmf'),
                     Position('LCM', self.position_line5.LCM, statsbomb=[11, 15], opta=8,
                              wyscout='lcmf'),
                     Position('CAM', self.position_line5.CAM, statsbomb=[19], opta=9,
                              wyscout='amf'),
                     Position('RCF', self.position_line5.RCF, statsbomb=[22], opta=10,
                              wyscout='ss'),
                     Position('LCF', self.position_line5.LCF, statsbomb=[24], opta=11,
                              wyscout='cf'),
                     ],
            '3142': [Position('GK', self.position_line5.GK, statsbomb=[1], opta=1),
                     Position('RCB', self.position_line5.RCB, statsbomb=[3], opta=5),
                     Position('CB', self.position_line5.CB, statsbomb=[4], opta=4),
                     Position('LCB', self.position_line5.LCB, statsbomb=[5], opta=6),
                     Position('CDM', self.position_line5.CDM, statsbomb=[10], opta=8),
                     Position('RM', self.position_line4.RM, statsbomb=[12], opta=2),
                     Position('RCM', self.position_line4.RCM, statsbomb=[13], opta=7),
                     Position('LCM', self.position_line4.LCM, statsbomb=[15], opta=11),
                     Position('LM', self.position_line4.LM, statsbomb=[16], opta=3),
                     Position('RCF', self.position_line5.RCF, statsbomb=[22], opta=9),
                     Position('LCF', self.position_line5.LCF, statsbomb=[24], opta=10),
                     ],
            '31213': [Position('GK', self.position_line5.GK, opta=1),
                      Position('RCB', self.position_line5.RCB, opta=6),
                      Position('CB', self.position_line5.CB, opta=5),
                      Position('LCB', self.position_line5.LCB, opta=4),
                      Position('CDM', self.position_line5.CDM, opta=8),
                      Position('RCM', self.position_line5.RCM, opta=2),
                      Position('LCM', self.position_line5.LCM, opta=3),
                      Position('RW', self.position_line5.RW, opta=10),
                      Position('CAM', self.position_line5.CAM, opta=7),
                      Position('LW', self.position_line5.LW, opta=11),
                      Position('ST', self.position_line5.ST, opta=9),
                      ],
            '4132': [Position('GK', self.position_line5.GK, statsbomb=[1], opta=1, wyscout='gk'),
                     Position('RB', self.position_line4.RB, statsbomb=[2], opta=2, wyscout='rb'),
                     Position('RCB', self.position_line4.RCB, statsbomb=[3], opta=5, wyscout='rcb'),
                     Position('LCB', self.position_line4.LCB, statsbomb=[5], opta=6, wyscout='lcb'),
                     Position('LB', self.position_line4.LB, statsbomb=[6], opta=3, wyscout='lb'),
                     Position('CDM', self.position_line5.CDM, statsbomb=[10], opta=4,
                              wyscout='dmf'),
                     Position('RW', self.position_line4.RW, statsbomb=[17, 12], opta=7,
                              wyscout='ramf'),
                     Position('CAM', self.position_line5.CAM, statsbomb=[19], opta=8,
                              wyscout='amf'),
                     Position('LW', self.position_line4.LW, statsbomb=[21, 16], opta=11,
                              wyscout='lamf'),
                     Position('RCF', self.position_line5.RCF, statsbomb=[22], opta=9, wyscout='ss'),
                     Position('LCF', self.position_line5.LCF, statsbomb=[24], opta=10,
                              wyscout='cf'),
                     ],
            '424': [Position('GK', self.position_line5.GK, opta=1),
                    Position('RB', self.position_line4.RB, opta=2),
                    Position('RCB', self.position_line4.RCB, opta=5),
                    Position('LCB', self.position_line4.LCB, opta=6),
                    Position('LB', self.position_line4.LB, opta=3),
                    Position('RDM', self.position_line5.RDM, opta=4),
                    Position('LDM', self.position_line5.LDM, opta=8),
                    Position('RW', self.position_line4.RW, opta=7),
                    Position('LW', self.position_line4.LW, opta=11),
                    Position('RCF', self.position_line4.RCF, opta=9),
                    Position('LCF', self.position_line4.LCF, opta=10),
                    ],
            '4312': [Position('GK', self.position_line5.GK, statsbomb=[1], opta=1, wyscout='gk'),
                     Position('RB', self.position_line4.RB, statsbomb=[2], opta=2, wyscout='rb'),
                     Position('RCB', self.position_line4.RCB, statsbomb=[3], opta=5, wyscout='rcb'),
                     Position('LCB', self.position_line4.LCB, statsbomb=[5], opta=6, wyscout='lcb'),
                     Position('LB', self.position_line4.LB, statsbomb=[6], opta=3, wyscout='lb'),
                     Position('RCM', self.position_line5.RCM, statsbomb=[13, 9], opta=7,
                              wyscout='rcmf3'),
                     Position('CM', self.position_line5.CM, statsbomb=[14, 10], opta=4,
                              wyscout='dmf'),
                     Position('LCM', self.position_line5.LCM, statsbomb=[15, 11], opta=11,
                              wyscout='lcmf3'),
                     Position('CAM', self.position_line5.CAM, statsbomb=[19], opta=8,
                              wyscout='amf'),
                     Position('RCF', self.position_line4.RCF, statsbomb=[22], opta=9, wyscout='ss'),
                     Position('LCF', self.position_line4.LCF, statsbomb=[24], opta=10,
                              wyscout='cf'),
                     ],
            '3241': [Position('GK', self.position_line5.GK, opta=1),
                     Position('RCB', self.position_line5.RCB, opta=6),
                     Position('CB', self.position_line5.CB, opta=5),
                     Position('LCB', self.position_line5.LCB, opta=4),
                     Position('RDM', self.position_line4.RDM, opta=2),
                     Position('LDM', self.position_line4.LDM, opta=3),
                     Position('RW', self.position_line4.RW, opta=10),
                     Position('RAM', self.position_line4.RAM, opta=7),
                     Position('LAM', self.position_line4.LAM, opta=8),
                     Position('LW', self.position_line4.LW, opta=11),
                     Position('ST', self.position_line5.ST, opta=9),
                     ],
            '3331': [Position('GK', self.position_line5.GK, opta=1, wyscout='gk'),
                     Position('RCB', self.position_line5.RCB, opta=6, wyscout='rcb3'),
                     Position('CB', self.position_line5.CB, opta=5, wyscout='cb'),
                     Position('LCB', self.position_line5.LCB, opta=4, wyscout='lcb3'),
                     Position('RDM', self.position_line5.RDM, opta=2, wyscout='rcmf3'),
                     Position('CDM', self.position_line5.CDM, opta=8, wyscout='dmf'),
                     Position('LDM', self.position_line5.LDM, opta=3, wyscout='lcmf3'),
                     Position('RM', self.position_line5.RM, opta=10, wyscout='ramf'),
                     Position('LM', self.position_line5.LM, opta=11, wyscout='lamf'),
                     Position('CAM', self.position_line5.CAM, opta=7, wyscout='amf'),
                     Position('ST', self.position_line5.ST, opta=9, wyscout='cf'),
                     ],
            'pyramid': [Position('GK', self.position_line5.GK),
                        Position('RCB', self.position_line5.RCB),
                        Position('LCB', self.position_line5.LCB),
                        Position('RCM', self.position_line5.RCM),
                        Position('CM', self.position_line5.CM),
                        Position('LCM', self.position_line5.LCM),
                        Position('RW', self.position_line5.RW),
                        Position('LW', self.position_line5.LW),
                        Position('RCF', self.position_line5.RCF),
                        Position('ST', self.position_line5.ST),
                        Position('LCF', self.position_line5.LCF),
                        ],
            'metodo': [Position('GK', self.position_line5.GK),
                       Position('RCB', self.position_line5.RCB),
                       Position('LCB', self.position_line5.LCB),
                       Position('RWB', self.position_line5.RWB),
                       Position('CDM', self.position_line5.CDM),
                       Position('LWB', self.position_line5.LWB),
                       Position('RCM', self.position_line5.RCM),
                       Position('LCM', self.position_line5.LCM),
                       Position('RW', self.position_line5.RW),
                       Position('LW', self.position_line5.LW),
                       Position('ST', self.position_line5.ST),
                       ],
            'wm': [Position('GK', self.position_line5.GK),
                   Position('RCB', self.position_line5.RCB),
                   Position('CB', self.position_line5.CB),
                   Position('LCB', self.position_line5.LCB),
                   Position('RDM', self.position_line4.RDM),
                   Position('LDM', self.position_line4.LDM),
                   Position('RCM', self.position_line5.RCM),
                   Position('LCM', self.position_line5.LCM),
                   Position('RW', self.position_line5.RW),
                   Position('LW', self.position_line5.LW),
                   Position('ST', self.position_line5.ST),
                   ],
            '41221': [Position('GK', self.position_line5.GK, statsbomb=[1]),
                      Position('RB', self.position_line4.RB, statsbomb=[2]),
                      Position('RCB', self.position_line4.RCB, statsbomb=[3]),
                      Position('LCB', self.position_line4.LCB, statsbomb=[5]),
                      Position('LB', self.position_line4.LB, statsbomb=[6]),
                      Position('CDM', self.position_line5.CDM, statsbomb=[10]),
                      Position('RCM', self.position_line5.RCM, statsbomb=[13]),
                      Position('LCM', self.position_line5.LCM, statsbomb=[15]),
                      Position('RW', self.position_line5.RW, statsbomb=[17, 18]),
                      Position('LW', self.position_line5.LW, statsbomb=[21, 20]),
                      Position('ST', self.position_line5.ST, statsbomb=[23, 25]),
                      ],
            '42211': [Position('GK', self.position_line5.GK, statsbomb=[1]),
                      Position('RB', self.position_line4.RB, statsbomb=[2]),
                      Position('RCB', self.position_line4.RCB, statsbomb=[3]),
                      Position('LCB', self.position_line4.LCB, statsbomb=[5]),
                      Position('LB', self.position_line4.LB, statsbomb=[6]),
                      Position('RDM', self.position_line5.RDM, statsbomb=[9, 13]),
                      Position('LDM', self.position_line5.LDM, statsbomb=[11, 15]),
                      Position('RM', self.position_line5.RM, statsbomb=[12, 17]),
                      Position('LM', self.position_line5.LM, statsbomb=[16, 21]),
                      Position('CAM', self.position_line5.CAM, statsbomb=[19]),
                      Position('ST', self.position_line5.ST, statsbomb=[23, 25]),
                      ],
            '32221': [Position('GK', self.position_line5.GK, statsbomb=[1]),
                      Position('RCB', self.position_line5.RCB, statsbomb=[3]),
                      Position('CB', self.position_line5.CB, statsbomb=[4]),
                      Position('LCB', self.position_line5.LCB, statsbomb=[5]),
                      Position('RDM', self.position_line4.RDM, statsbomb=[9, 7]),
                      Position('LDM', self.position_line4.LDM, statsbomb=[11, 8]),
                      Position('RM', self.position_line5.RM, statsbomb=[12, 13]),
                      Position('LM', self.position_line5.LM, statsbomb=[16, 15]),
                      Position('RAM', self.position_line4.RAM, statsbomb=[18, 17]),
                      Position('LAM', self.position_line4.LAM, statsbomb=[20, 21]),
                      Position('ST', self.position_line5.ST, statsbomb=[23]),
                      ],
            '5221': [Position('GK', self.position_line5.GK, statsbomb=[1]),
                     Position('RB', self.position_line5.RB, statsbomb=[2]),
                     Position('RCB', self.position_line5.RCB, statsbomb=[3]),
                     Position('CB', self.position_line5.CB, statsbomb=[4]),
                     Position('LCB', self.position_line5.LCB, statsbomb=[5]),
                     Position('LB', self.position_line5.LB, statsbomb=[6]),
                     Position('RCM', self.position_line5.RCM, statsbomb=[13, 9]),
                     Position('LCM', self.position_line5.LCM, statsbomb=[15, 11]),
                     Position('RW', self.position_line5.RW, statsbomb=[17]),
                     Position('LW', self.position_line5.LW, statsbomb=[21]),
                     Position('ST', self.position_line5.ST, statsbomb=[23]),
                     ],
            '3232': [Position('GK', self.position_line5.GK, statsbomb=[1], wyscout='gk'),
                     Position('RCB', self.position_line5.RCB, statsbomb=[3], wyscout='rcb3'),
                     Position('CB', self.position_line5.CB, statsbomb=[4], wyscout='cb'),
                     Position('LCB', self.position_line5.LCB, statsbomb=[5], wyscout='lcb3'),
                     Position('RWB', self.position_line5.RWB, statsbomb=[7], wyscout='rdmf'),
                     Position('LWB', self.position_line5.LWB, statsbomb=[8], wyscout='ldmf'),
                     Position('RCM', self.position_line5.RCM, statsbomb=[13], wyscout='ramf'),
                     Position('CM', self.position_line5.CM, statsbomb=[14], wyscout='amf'),
                     Position('LCM', self.position_line5.LCM, statsbomb=[15], wyscout='lamf'),
                     Position('RCF', self.position_line5.RCF, statsbomb=[22], wyscout='ss'),
                     Position('LCF', self.position_line5.LCF, statsbomb=[24], wyscout='cf'),
                     ],
            '312112': [Position('GK', self.position_line5_with_ss.GK, statsbomb=[1]),
                       Position('RCB', self.position_line5_with_ss.RCB, statsbomb=[3]),
                       Position('CB', self.position_line5_with_ss.CB, statsbomb=[4]),
                       Position('LCB', self.position_line5_with_ss.LCB, statsbomb=[5]),
                       Position('CDM', self.position_line5_with_ss.CDM, statsbomb=[10]),
                       Position('RCM', self.position_line5_with_ss.RCM, statsbomb=[13]),
                       Position('LCM', self.position_line5_with_ss.LCM, statsbomb=[15]),
                       Position('CAM', self.position_line5_with_ss.CAM, statsbomb=[19]),
                       Position('RCF', self.position_line5_with_ss.RCF, statsbomb=[22]),
                       Position('LCF', self.position_line5_with_ss.LCF, statsbomb=[24]),
                       Position('SS', self.position_line5_with_ss.SS, statsbomb=[25]),
                       ],
            '42121': [Position('GK', self.position_line5.GK, statsbomb=[1]),
                      Position('RB', self.position_line4.RB, statsbomb=[2]),
                      Position('RCB', self.position_line4.RCB, statsbomb=[3]),
                      Position('LCB', self.position_line4.LCB, statsbomb=[5]),
                      Position('LB', self.position_line4.LB, statsbomb=[6]),
                      Position('RDM', self.position_line5.RDM, statsbomb=[9]),
                      Position('LDM', self.position_line5.LDM, statsbomb=[11]),
                      Position('CM', self.position_line5.CM, statsbomb=[14]),
                      Position('RW', self.position_line5.RW, statsbomb=[17]),
                      Position('LW', self.position_line5.LW, statsbomb=[21]),
                      Position('ST', self.position_line5.ST, statsbomb=[23]),
                      ],
            '31222': [Position('GK', self.position_line5.GK, statsbomb=[1]),
                      Position('RCB', self.position_line5.RCB, statsbomb=[3]),
                      Position('CB', self.position_line5.CB, statsbomb=[4]),
                      Position('LCB', self.position_line5.LCB, statsbomb=[5]),
                      Position('CDM', self.position_line5.CDM, statsbomb=[10]),
                      Position('RCM', self.position_line5.RCM, statsbomb=[13]),
                      Position('LCM', self.position_line5.LCM, statsbomb=[15]),
                      Position('RW', self.position_line5.RW, statsbomb=[17]),
                      Position('LW', self.position_line5.LW, statsbomb=[21]),
                      Position('RCF', self.position_line5.RCF, statsbomb=[22]),
                      Position('LCF', self.position_line5.LCF, statsbomb=[24]),
                      ],
            '4213': [Position('GK', self.position_line5.GK, statsbomb=[1], wyscout='gk'),
                     Position('RB', self.position_line4.RB, statsbomb=[2], wyscout='rb'),
                     Position('RCB', self.position_line4.RCB, statsbomb=[3], wyscout='rcb'),
                     Position('LCB', self.position_line4.LCB, statsbomb=[5], wyscout='lcb'),
                     Position('LB', self.position_line4.LB, statsbomb=[6], wyscout='lb'),
                     Position('RDM', self.position_line5.RDM, statsbomb=[9], wyscout='rdmf'),
                     Position('LDM', self.position_line5.LDM, statsbomb=[11], wyscout='ldmf'),
                     Position('CAM', self.position_line5.CAM, statsbomb=[19], wyscout='amf'),
                     Position('RCF', self.position_line5.RCF, statsbomb=[22], wyscout='rwf'),
                     Position('ST', self.position_line5.ST, statsbomb=[23], wyscout='cf'),
                     Position('LCF', self.position_line5.LCF, statsbomb=[24], wyscout='lwf'),
                     ],
            '32122': [Position('GK', self.position_line5.GK, statsbomb=[1]),
                      Position('RCB', self.position_line5.RCB, statsbomb=[3]),
                      Position('CB', self.position_line5.CB, statsbomb=[4]),
                      Position('LCB', self.position_line5.LCB, statsbomb=[5]),
                      Position('RWB', self.position_line5.RWB, statsbomb=[7]),
                      Position('LWB', self.position_line5.LWB, statsbomb=[8]),
                      Position('CM', self.position_line5.CM, statsbomb=[14]),
                      Position('RAM', self.position_line5.RAM, statsbomb=[18]),
                      Position('LAM', self.position_line5.LAM, statsbomb=[20]),
                      Position('RCF', self.position_line5.RCF, statsbomb=[22]),
                      Position('LCF', self.position_line5.LCF, statsbomb=[24]),
                      ],
            '3322': [Position('GK', self.position_line5.GK, statsbomb=[1]),
                      Position('RCB', self.position_line5.RCB, statsbomb=[3]),
                      Position('CB', self.position_line5.CB, statsbomb=[4]),
                      Position('LCB', self.position_line5.LCB, statsbomb=[5]),
                      Position('RCM', self.position_line5.RCM, statsbomb=[7]),
                      Position('LCM', self.position_line5.LCM, statsbomb=[8]),
                      Position('CM', self.position_line5.CM, statsbomb=[14]),
                      Position('RAM', self.position_line5.RAM, statsbomb=[18]),
                      Position('LAM', self.position_line5.LAM, statsbomb=[20]),
                      Position('RCF', self.position_line5.RCF, statsbomb=[22]),
                      Position('LCF', self.position_line5.LCF, statsbomb=[24]),
                      ],
            '41131': [Position('GK', self.position_line5.GK, statsbomb=[1]),
                      Position('RB', self.position_line4.RB, statsbomb=[2]),
                      Position('RCB', self.position_line4.RCB, statsbomb=[3]),
                      Position('LCB', self.position_line4.LCB, statsbomb=[5]),
                      Position('LB', self.position_line4.LB, statsbomb=[6]),
                      Position('CDM', self.position_line5.CDM, statsbomb=[10]),
                      Position('CM', self.position_line5.CM, statsbomb=[14]),
                      Position('RW', self.position_line5.RW, statsbomb=[17]),
                      Position('CAM', self.position_line5.CAM, statsbomb=[19]),
                      Position('LW', self.position_line5.LW, statsbomb=[21]),
                      Position('ST', self.position_line5.ST, statsbomb=[23]),
                      ],
            '432': [Position('GK', self.position_line5.GK, wyscout='gk'),
                    Position('RB', self.position_line4.RB, wyscout='rb'),
                    Position('RCB', self.position_line4.RCB, wyscout='rcb'),
                    Position('LCB', self.position_line4.LCB, wyscout='lcb'),
                    Position('LB', self.position_line4.LB, wyscout='lb'),
                    Position('CDM', self.position_line5.CDM, wyscout='dmf'),
                    Position('RCM', self.position_line5.RCM, wyscout='rcmf3'),
                    Position('LCM', self.position_line5.LCM, wyscout='lcmf3'),
                    Position('RCF', self.position_line5.RCF, wyscout='ss'),
                    Position('LCF', self.position_line5.LCF, wyscout='cf'),
                    ],
            '441': [Position('GK', self.position_line5.GK, wyscout='gk'),
                    Position('RB', self.position_line4.RB, wyscout='rb'),
                    Position('RCB', self.position_line4.RCB, wyscout='rcb'),
                    Position('LCB', self.position_line4.LCB, wyscout='lcb'),
                    Position('LB', self.position_line4.LB, wyscout='lb'),
                    Position('RM', self.position_line4.RM, wyscout='rw'),
                    Position('RCM', self.position_line4.RCM, wyscout='rcmf'),
                    Position('LCM', self.position_line4.LCM, wyscout='lcmf'),
                    Position('LM', self.position_line4.LM, wyscout='lw'),
                    Position('ST', self.position_line5.ST, wyscout='cf'),
                    ],
            '4311': [Position('GK', self.position_line5.GK, wyscout='gk'),
                     Position('RB', self.position_line4.RB, wyscout='rb'),
                     Position('RCB', self.position_line4.RCB, wyscout='rcb'),
                     Position('LCB', self.position_line4.LCB, wyscout='lcb'),
                     Position('LB', self.position_line4.LB, wyscout='lb'),
                     Position('RCM', self.position_line5.RCM, wyscout='rcmf3'),
                     Position('CM', self.position_line5.CM, wyscout='dmf'),
                     Position('LCM', self.position_line5.LCM, wyscout='lcmf3'),
                     Position('CAM', self.position_line5.CAM, wyscout='amf'),
                     Position('ST', self.position_line5.ST, wyscout='cf'),
                     ],
            '4221': [Position('GK', self.position_line5.GK, wyscout='gk'),
                     Position('RB', self.position_line4.RB, wyscout='rb'),
                     Position('RCB', self.position_line4.RCB, wyscout='rcb'),
                     Position('LCB', self.position_line4.LCB, wyscout='lcb'),
                     Position('LB', self.position_line4.LB, wyscout='lb'),
                     Position('RCM', self.position_line5.RCM, wyscout='rdmf'),
                     Position('LCM', self.position_line5.LCM, wyscout='ldmf'),
                     Position('RW', self.position_line5.RW, wyscout='ramf'),
                     Position('LW', self.position_line5.LW, wyscout='lamf'),
                     Position('ST', self.position_line5.ST, wyscout='cf'),
                     ],
            '4131': [Position('GK', self.position_line5.GK, wyscout='gk'),
                     Position('RB', self.position_line4.RB, wyscout='rb'),
                     Position('RCB', self.position_line4.RCB, wyscout='rcb'),
                     Position('LCB', self.position_line4.LCB, wyscout='lcb'),
                     Position('LB', self.position_line4.LB, wyscout='lb'),
                     Position('CDM', self.position_line5.CDM, wyscout='dmf'),
                     Position('RW', self.position_line4.RW, wyscout='ramf'),
                     Position('CAM', self.position_line5.CAM, wyscout='amf'),
                     Position('LW', self.position_line4.LW, wyscout='lamf'),
                     Position('ST', self.position_line5.ST, wyscout='cf'),
                     ],
            '4212': [Position('GK', self.position_line5.GK, wyscout='gk'),
                     Position('RB', self.position_line4.RB, wyscout='rb'),
                     Position('RCB', self.position_line4.RCB, wyscout='rcb'),
                     Position('LCB', self.position_line4.LCB, wyscout='lcb'),
                     Position('LB', self.position_line4.LB, wyscout='lb'),
                     Position('RDM', self.position_line5.RDM, wyscout='rdmf'),
                     Position('LDM', self.position_line5.LDM, wyscout='ldmf'),
                     Position('CAM', self.position_line5.CAM, wyscout='amf'),
                     Position('RCF', self.position_line5.RCF, wyscout='ss'),
                     Position('LCF', self.position_line5.LCF, wyscout='cf'),
                     ],
            '342': [Position('GK', self.position_line5.GK, wyscout='gk'),
                    Position('RCB', self.position_line5.RCB, wyscout='rcb3'),
                    Position('CB', self.position_line5.CB, wyscout='cb'),
                    Position('LCB', self.position_line5.LCB, wyscout='lcb3'),
                    Position('RWB', self.position_line5.RWB, wyscout='rwb'),
                    Position('LWB', self.position_line5.LWB, wyscout='lwb'),
                    Position('RCM', self.position_line5.RCM, wyscout='rcmf'),
                    Position('LCM', self.position_line5.LCM, wyscout='lcmf'),
                    Position('RCF', self.position_line5.RCF, wyscout='ss'),
                    Position('LCF', self.position_line5.LCF, wyscout='cf'),
                    ],
            '3411': [Position('GK', self.position_line5.GK, wyscout='gk'),
                     Position('RCB', self.position_line5.RCB, wyscout='rcb3'),
                     Position('CB', self.position_line5.CB, wyscout='cb'),
                     Position('LCB', self.position_line5.LCB, wyscout='lcb3'),
                     Position('RWB', self.position_line5.RWB, wyscout='rwb'),
                     Position('LWB', self.position_line5.LWB, wyscout='lwb'),
                     Position('RCM', self.position_line5.RCM, wyscout='rcmf'),
                     Position('LCM', self.position_line5.LCM, wyscout='lcmf'),
                     Position('CAM', self.position_line5.CAM, wyscout='amf'),
                     Position('ST', self.position_line5.ST, wyscout='cf'),
                     ],
            '351': [Position('GK', self.position_line5.GK, wyscout='gk'),
                    Position('RCB', self.position_line5.RCB, wyscout='rcb3'),
                    Position('CB', self.position_line5.CB, wyscout='cb'),
                    Position('LCB', self.position_line5.LCB, wyscout='lcb3'),
                    Position('RWB', self.position_line5.RWB, wyscout='rwb'),
                    Position('LWB', self.position_line5.LWB, wyscout='lwb'),
                    Position('RCM', self.position_line5.RCM, wyscout='rcmf3'),
                    Position('CM', self.position_line5.CM, wyscout='dmf'),
                    Position('LCM', self.position_line5.LCM, wyscout='lcmf3'),
                    Position('ST', self.position_line5.ST, wyscout='cf'),
                    ],
            '531': [Position('GK', self.position_line5.GK, wyscout='gk'),
                    Position('RB', self.position_line5.RB, wyscout='rb5'),
                    Position('RCB', self.position_line5.RCB, wyscout='rcb3'),
                    Position('CB', self.position_line5.CB, wyscout='cb'),
                    Position('LCB', self.position_line5.LCB, wyscout='lcb3'),
                    Position('LB', self.position_line5.LB, wyscout='lb5'),
                    Position('CDM', self.position_line5.CDM, wyscout='dmf'),
                    Position('RCM', self.position_line5.RCM, wyscout='rcmf3'),
                    Position('LCM', self.position_line5.LCM, wyscout='lcmf3'),
                    Position('ST', self.position_line5.ST, wyscout='cf'),
                    ],
            '431': [Position('GK', self.position_line5.GK, wyscout='gk'),
                    Position('RB', self.position_line4.RB, wyscout='rb'),
                    Position('RCB', self.position_line4.RCB, wyscout='rcb'),
                    Position('LCB', self.position_line4.LCB, wyscout='lcb'),
                    Position('LB', self.position_line4.LB, wyscout='lb'),
                    Position('RCM', self.position_line5.RCM, wyscout='rcmf3'),
                    Position('CM', self.position_line5.CM, wyscout='dmf'),
                    Position('LCM', self.position_line5.LCM, wyscout='lcmf3'),
                    Position('ST', self.position_line5.ST, wyscout='cf'),
                    ],
            '44': [Position('GK', self.position_line5.GK, wyscout='gk'),
                   Position('RB', self.position_line4.RB, wyscout='rb'),
                   Position('RCB', self.position_line4.RCB, wyscout='rcb'),
                   Position('LCB', self.position_line4.LCB, wyscout='lcb'),
                   Position('LB', self.position_line4.LB, wyscout='lb'),
                   Position('RM', self.position_line4.RM, wyscout='rw'),
                   Position('RCM', self.position_line4.RCM, wyscout='rcmf'),
                   Position('LCM', self.position_line4.LCM, wyscout='lcmf'),
                   Position('LM', self.position_line4.LM, wyscout='lw'),
                   ],
            '422': [Position('GK', self.position_line5.GK, wyscout='gk'),
                    Position('RB', self.position_line4.RB, wyscout='rb'),
                    Position('RCB', self.position_line4.RCB, wyscout='rcb'),
                    Position('LCB', self.position_line4.LCB, wyscout='lcb'),
                    Position('LB', self.position_line4.LB, wyscout='lb'),
                    Position('RCM', self.position_line5.RCM, wyscout='rdmf'),
                    Position('LCM', self.position_line5.LCM, wyscout='ldmf'),
                    Position('RW', self.position_line5.RW, wyscout='ramf'),
                    Position('LW', self.position_line5.LW, wyscout='lamf'),
                    ],
            '341': [Position('GK', self.position_line5.GK, wyscout='gk'),
                    Position('RCB', self.position_line5.RCB, wyscout='rcb3'),
                    Position('CB', self.position_line5.CB, wyscout='cb'),
                    Position('LCB', self.position_line5.LCB, wyscout='lcb3'),
                    Position('RWB', self.position_line5.RWB, wyscout='rwb'),
                    Position('LWB', self.position_line5.LWB, wyscout='lwb'),
                    Position('RCM', self.position_line5.RCM, wyscout='rcmf'),
                    Position('LCM', self.position_line5.LCM, wyscout='lcmf'),
                    Position('ST', self.position_line5.ST, wyscout='cf'),
                    ],
            '53': [Position('GK', self.position_line5.GK, wyscout='gk'),
                   Position('RB', self.position_line5.RB, wyscout='rb5'),
                   Position('RCB', self.position_line5.RCB, wyscout='rcb3'),
                   Position('CB', self.position_line5.CB, wyscout='cb'),
                   Position('LCB', self.position_line5.LCB, wyscout='lcb3'),
                   Position('LB', self.position_line5.LB, wyscout='lb5'),
                   Position('CDM', self.position_line5.CDM, wyscout='dmf'),
                   Position('RCM', self.position_line5.RCM, wyscout='rcmf3'),
                   Position('LCM', self.position_line5.LCM, wyscout='lcmf3'),
                   ],
            # Attacking Formations
            '1234': [Position('GK', self.position_line5.GK),
                    Position('CB', self.position_line5.CB),
                    Position('RDM', self.position_line5.RDM),
                    Position('LDM', self.position_line5.LDM),
                    Position('CM', self.position_line5.CM),
                    Position('LCM', self.position_line5.LCM),
                    Position('RCM', self.position_line5.RCM),
                    Position('RW', self.position_line5.RW),
                    Position('LW', self.position_line5.LW),
                    Position('RCF', self.position_line4.RCF),
                    Position('LCF', self.position_line4.LCF),
                    ],
            '1324': [Position('GK', self.position_line5.GK),
                    Position('CB', self.position_line5.CB),
                    Position('RDM', self.position_line5.RDM),
                    Position('LDM', self.position_line5.LDM),
                    Position('CDM', self.position_line5.CDM),
                    Position('LCM', self.position_line5.LCM),
                    Position('RCM', self.position_line5.RCM),
                    Position('RW', self.position_line5.RW),
                    Position('LW', self.position_line5.LW),
                    Position('RCF', self.position_line4.RCF),
                    Position('LCF', self.position_line4.LCF),
                    ],
            '1432': [Position('GK', self.position_line5.GK),
                    Position('CB', self.position_line5.CB),
                    Position('RM', self.position_line5.RM),
                    Position('LM', self.position_line5.LM),
                    Position('LCM', self.position_line5.LCM),
                    Position('RCM', self.position_line5.RCM),
                    Position('RAM', self.position_line5.RAM),
                    Position('LAM', self.position_line5.LAM),
                    Position('CAM', self.position_line5.CAM),
                    Position('RCF', self.position_line4.RCF),
                    Position('LCF', self.position_line4.LCF),
                    ],
            '1423': [Position('GK', self.position_line5.GK),
                    Position('CB', self.position_line5.CB),
                    Position('RM', self.position_line5.RM),
                    Position('LM', self.position_line5.LM),
                    Position('LCM', self.position_line5.LCM),
                    Position('RCM', self.position_line5.RCM),
                    Position('RAM', self.position_line5.RAM),
                    Position('LAM', self.position_line5.LAM,),
                    Position('ST', self.position_line5.ST),
                    Position('RCF', self.position_line4.RCF),
                    Position('LCF', self.position_line4.LCF),
                    ],
            '2422': [Position('GK', self.position_line5.GK),
                    Position('RCB', self.position_line5.RCB),
                    Position('LCB', self.position_line5.LCB),
                    Position('RWB', self.position_line5.RWB),
                    Position('LWB', self.position_line5.LWB),
                    Position('LCM', self.position_line5.LCM),
                    Position('RCM', self.position_line5.RCM),
                    Position('RAM', self.position_line5.RAM),
                    Position('LAM', self.position_line5.LAM),
                    
                    Position('RCF', self.position_line4.RCF),
                    Position('LCF', self.position_line4.LCF),
                    ],
            '2413': [Position('GK', self.position_line5.GK),
                    Position('RCB', self.position_line5.RCB),
                    Position('LCB', self.position_line5.LCB),
                    Position('RWB', self.position_line5.RWB),
                    Position('LWB', self.position_line5.LWB),
                    Position('LCM', self.position_line5.LCM),
                    Position('RCM', self.position_line5.RCM),
                    Position('CAM', self.position_line5.CAM),
                    Position('ST', self.position_line5.ST),
                    
                    Position('RCF', self.position_line4.RCF),
                    Position('LCF', self.position_line4.LCF),
                    ],
            '2431': [Position('GK', self.position_line5.GK),
                    Position('RCB', self.position_line5.RCB),
                    Position('LCB', self.position_line5.LCB),
                    Position('RWB', self.position_line5.RWB),
                    Position('LWB', self.position_line5.LWB),
                    Position('LCM', self.position_line5.LCM),
                    Position('RCM', self.position_line5.RCM),
                    Position('CAM', self.position_line5.CAM),
                    Position('LAM', self.position_line5.LAM),
                    
                    Position('RAM', self.position_line5.RAM),
                    Position('ST', self.position_line5.ST)
                    ],
            '2332': [Position('GK', self.position_line5.GK),
                    Position('RCB', self.position_line5.RCB),
                    Position('LCB', self.position_line5.LCB),
                    Position('CDM', self.position_line5.CDM),
                    Position('LCM', self.position_line5.LCM),
                    Position('RCM', self.position_line5.RCM),
                    Position('CAM', self.position_line5.CAM),
                    Position('LAM', self.position_line5.LAM),
                    Position('RAM', self.position_line5.RAM),
                    Position('RCF', self.position_line5.RCF),
                    Position('LCF', self.position_line5.LCF)
                    ],
            '2233': [Position('GK', self.position_line5.GK),
                    Position('RCB', self.position_line5.RCB),
                    Position('LCB', self.position_line5.LCB),
                    Position('LCM', self.position_line5.LCM),
                    Position('RCM', self.position_line5.RCM),
                    Position('CAM', self.position_line5.CAM),
                    Position('LAM', self.position_line5.LAM),
                    Position('RAM', self.position_line5.RAM),
                    Position('RCF', self.position_line5.RCF),
                    Position('LCF', self.position_line5.LCF),
                    Position('ST', self.position_line5.ST),
                    ]
        }

