from django.contrib.auth import get_user_model
from django.contrib.contenttypes.fields import GenericForeignKey
from django.contrib.contenttypes.models import ContentType
from django.db import models
from django.utils import timezone
from django.utils.translation import gettext_lazy as _

USER = get_user_model()


class BiddingRound(models.Model):
    name = models.CharField(
        max_length=200, verbose_name=_("Name"), null=True, blank=True
    )
    season = models.CharField(
        max_length=200, verbose_name=_("Season"), null=True, blank=True
    )
    start_time = models.DateTimeField(verbose_name=_("Start"))
    end_time = models.DateTimeField(verbose_name=_("End time"))
    guiding_value = models.DecimalField(
        max_digits=6,
        decimal_places=2,
        verbose_name=_("Guiding value"),
        null=True,
        blank=True,
    )
    comment = models.TextField(verbose_name=_("Comment"), null=True, blank=True)

    class Meta:
        ordering = ["-start_time"]

    @property
    def is_active(self):
        return self.start_time <= timezone.now() <= self.end_time

    @property
    def is_in_future(self):
        return self.start_time > timezone.now()

    def get_average_bid_amount(self, color):
        valid_colors = {
            "green": "green_contribution_amount",
            "yellow": "yellow_contribution_amount",
            "red": "red_contribution_amount",
        }

        if color not in valid_colors:
            raise ValueError("Invalid color. Choose 'green', 'yellow', or 'red'.")

        bids = Bid.objects.filter(biddinground_target__biddinground=self)
        if not bids.exists():
            return 0

        total_amount = sum(getattr(bid, valid_colors[color]) for bid in bids)
        return round(total_amount / bids.count(), 2)

    def __str__(self):
        return f"BiddingRound({self.pk}): {self.name} {self.start_time.date()} -- {self.end_time.date()} ({'not ' if not self.is_active else ''}active)"


class BiddingRoundTarget(models.Model):
    """
    Represents a target object for a BiddingRound.
    Can be a Subscription, Farm, or any other model using GenericForeignKey.
    """

    biddinground = models.ForeignKey(
        BiddingRound,
        on_delete=models.CASCADE,
        related_name="targets",
        verbose_name=_("Bidding round"),
    )
    content_type = models.ForeignKey(
        ContentType, on_delete=models.CASCADE, verbose_name=_("Content type")
    )
    object_id = models.PositiveBigIntegerField(verbose_name=_("Object ID"))
    target = GenericForeignKey("content_type", "object_id")

    class Meta:
        unique_together = ("biddinground", "content_type", "object_id")

    def __str__(self):
        return f"Target({self.pk}): {self.target} for {self.biddinground}"


class Bid(models.Model):
    biddinground_target = models.ForeignKey(
        BiddingRoundTarget,
        on_delete=models.CASCADE,
        related_name="bids",
        verbose_name=_("Bidding target"),
    )
    bidder = models.ForeignKey(USER, on_delete=models.CASCADE, verbose_name=_("Bidder"))

    green_contribution_amount = models.DecimalField(
        max_digits=6, decimal_places=2, verbose_name=_("Green bid (comfort zone)")
    )
    yellow_contribution_amount = models.DecimalField(
        max_digits=6, decimal_places=2, verbose_name=_("Yellow bid (feasible)")
    )
    red_contribution_amount = models.DecimalField(
        max_digits=6, decimal_places=2, verbose_name=_("Red bid (pain threshold)")
    )

    comment = models.TextField(verbose_name=_("Kommentar"), null=True, blank=True)

    class Meta:
        constraints = [
            models.UniqueConstraint(
                fields=["biddinground_target", "bidder"],
                name="unique_bid_per_target_per_bidder",
            )
        ]

    def __str__(self):
        return f"Bid({self.pk}) by {self.bidder} for target {self.biddinground_target}"

    def get_biddinground(self):
        return self.biddinground_target.biddinground
