from django.db import IntegrityError
from django.test import TestCase
from django.utils import timezone
from django.contrib.contenttypes.models import ContentType
from dummy_testapp.models import DummySubscription
from biddingrounds.models import Bid, BiddingRoundTarget
from tests.factories import (
    BidFactory,
    BiddingRoundFactory,
    BiddingRoundTargetFactory,
    UserFactory,
    DummySubscriptionFactory,
)


class BiddingRoundTestCase(TestCase):
    def setUp(self):
        self.biddinground = BiddingRoundFactory()
        self.subscription = DummySubscriptionFactory()
        self.target = BiddingRoundTarget.objects.create(
            biddinground=self.biddinground,
            content_type=ContentType.objects.get_for_model(DummySubscription),
            object_id=self.subscription.id,
        )

    def test_biddinground_is_active_within_timespan(self):
        now = timezone.now()
        biddinground = BiddingRoundFactory(
            start_time=now - timezone.timedelta(hours=1),
            end_time=now + timezone.timedelta(hours=1),
        )
        self.assertTrue(biddinground.is_active)
        self.assertFalse(biddinground.is_in_future)

        future_round = BiddingRoundFactory(
            start_time=now + timezone.timedelta(days=1),
            end_time=now + timezone.timedelta(days=2),
        )
        self.assertFalse(future_round.is_active)
        self.assertTrue(future_round.is_in_future)

    def test_get_average_bid_amount(self):
        user1 = UserFactory()
        user2 = UserFactory()
        assert user1 != user2
        Bid.objects.create(
            biddinground_target=self.target,
            bidder=user1,
            green_contribution_amount=10,
            yellow_contribution_amount=20,
            red_contribution_amount=30,
        )

        Bid.objects.create(
            biddinground_target=self.target,
            bidder=user2,
            green_contribution_amount=20,
            yellow_contribution_amount=30,
            red_contribution_amount=40,
        )
        self.assertEqual(self.biddinground.get_average_bid_amount("green"), 15)
        self.assertEqual(self.biddinground.get_average_bid_amount("yellow"), 25)
        self.assertEqual(self.biddinground.get_average_bid_amount("red"), 35)

    def test_get_average_bid_amount_invalid_color(self):
        with self.assertRaises(ValueError):
            self.biddinground.get_average_bid_amount("blue")

    def test_get_average_bid_amount_no_bids_returns_zero(self):
        self.assertEqual(self.biddinground.get_average_bid_amount("green"), 0)

    def test_deleting_round_deletes_bids(self):
        bid = BidFactory(biddinground_target=self.target)
        self.biddinground.delete()
        self.assertFalse(Bid.objects.filter(id=bid.id).exists())

    def test_unique_bid_per_bidder_and_round(self):
        user = UserFactory()
        BidFactory(biddinground_target=self.target, bidder=user)

        with self.assertRaises(IntegrityError):
            Bid.objects.create(
                biddinground_target=self.target,
                bidder=user,
                green_contribution_amount=50,
                yellow_contribution_amount=60,
                red_contribution_amount=70,
            )

    def test_round_can_have_multiple_targets(self):
        biddinground = BiddingRoundFactory()
        subscription_1 = DummySubscriptionFactory()
        subscription_2 = DummySubscriptionFactory()
        assert subscription_1 != subscription_2
        target1 = BiddingRoundTarget.objects.create(
            biddinground=biddinground,
            content_type=ContentType.objects.get_for_model(DummySubscription),
            object_id=subscription_1.id,
        )
        target2 = BiddingRoundTarget.objects.create(
            biddinground=biddinground,
            content_type=ContentType.objects.get_for_model(DummySubscription),
            object_id=subscription_2.id,
        )

        self.assertEqual(biddinground.targets.count(), 2)
        self.assertIn(target1, biddinground.targets.all())
        self.assertIn(target2, biddinground.targets.all())

    def test_user_can_bid_once_per_target(self):
        bidder = UserFactory()
        BidFactory(biddinground_target=self.target, bidder=bidder)
        with self.assertRaises(IntegrityError):
            BidFactory(biddinground_target=self.target, bidder=bidder)

    def test_user_can_bid_for_different_targets(self):
        bidder = UserFactory()
        subscription_1 = DummySubscriptionFactory()
        subscription_2 = DummySubscriptionFactory()
        target_1 = BiddingRoundTargetFactory(
            biddinground=self.biddinground,
            content_type=ContentType.objects.get_for_model(DummySubscription),
            object_id=subscription_1.id,
        )
        target_2 = BiddingRoundTargetFactory(
            biddinground=self.biddinground,
            content_type=ContentType.objects.get_for_model(DummySubscription),
            object_id=subscription_2.id,
        )
        BidFactory(biddinground_target=target_1, bidder=bidder)
        BidFactory(biddinground_target=target_2, bidder=bidder)
        assert Bid.objects.count() == 2
