"""
AgentOps Documentation Export System

This module implements comprehensive documentation export capabilities:
- Multiple export formats (Markdown, PDF, HTML, DOCX)
- Professional templates and styling
- Customizable content and structure
- Modular feature system with switch on/off capabilities
- Tier-based access control
"""

from enum import Enum
from dataclasses import dataclass, field
from typing import Dict, List, Optional, Set, Any, Union
from pathlib import Path
import json
import yaml
from datetime import datetime
import logging
from jinja2 import Template, Environment, FileSystemLoader
import markdown
from weasyprint import HTML
from docx import Document
from docx.shared import Inches

from .features import get_feature_manager, is_feature_enabled
from .pricing import pricing_manager, PricingTier


class ExportFormat(Enum):
    """Supported documentation export formats."""
    
    MARKDOWN = "markdown"
    PDF = "pdf"
    HTML = "html"
    DOCX = "docx"
    JSON = "json"
    YAML = "yaml"


class TemplateType(Enum):
    """Documentation template types."""
    
    PROFESSIONAL = "professional"
    MINIMAL = "minimal"
    DETAILED = "detailed"
    TECHNICAL = "technical"
    EXECUTIVE = "executive"


@dataclass
class DocumentationConfig:
    """Configuration for documentation export."""
    
    # Export settings
    formats: Set[ExportFormat] = field(default_factory=lambda: {ExportFormat.MARKDOWN})
    template: TemplateType = TemplateType.PROFESSIONAL
    include_diagrams: bool = True
    include_code_samples: bool = True
    include_metadata: bool = True
    
    # Content settings
    sections: List[str] = field(default_factory=lambda: [
        "overview", "requirements", "tests", "coverage", "analysis"
    ])
    custom_sections: Dict[str, str] = field(default_factory=dict)
    
    # Styling settings
    theme: str = "default"
    custom_css: Optional[str] = None
    custom_js: Optional[str] = None
    
    # Output settings
    output_directory: str = ".agentops/docs"
    filename_template: str = "agentops_documentation_{timestamp}"
    
    # Metadata
    project_name: str = "AgentOps Project"
    version: str = "1.0.0"
    author: str = "AgentOps"
    company: str = "AgentOps"


@dataclass
class DocumentationSection:
    """Individual section of documentation."""
    
    id: str
    title: str
    content: str
    level: int = 1
    order: int = 0
    
    # Metadata
    metadata: Dict[str, Any] = field(default_factory=dict)
    
    # Dependencies
    dependencies: List[str] = field(default_factory=list)
    
    # Status
    status: str = "draft"
    last_updated: datetime = field(default_factory=datetime.now)


@dataclass
class DocumentationProject:
    """Complete documentation project."""
    
    project_id: str
    config: DocumentationConfig
    sections: Dict[str, DocumentationSection] = field(default_factory=dict)
    
    # Metadata
    created_at: datetime = field(default_factory=datetime.now)
    updated_at: datetime = field(default_factory=datetime.now)
    
    # Statistics
    total_sections: int = 0
    total_pages: int = 0


class DocumentationExporter:
    """Manages documentation export operations."""
    
    def __init__(self):
        """Initialize the documentation exporter."""
        self.feature_manager = get_feature_manager()
        self.templates = self._load_templates()
        
        # Check if documentation export feature is enabled
        if not is_feature_enabled("documentation_export"):
            logging.warning("Documentation export feature is not enabled")
    
    def _load_templates(self) -> Dict[str, str]:
        """Load documentation templates.
        
        Returns:
            Dictionary of template content
        """
        templates = {
            "professional_markdown": """
# {{ project_name }} Documentation

**Version:** {{ version }}  
**Generated:** {{ generated_at }}  
**Author:** {{ author }}

## Table of Contents
{% for section in sections %}
{{ "  " * (section.level - 1) }}- [{{ section.title }}](#{{ section.id }})
{% endfor %}

{% for section in sections %}
## {{ section.title }}

{{ section.content }}

{% if section.metadata %}
**Metadata:**
- Status: {{ section.metadata.get('status', 'unknown') }}
- Last Updated: {{ section.metadata.get('last_updated', 'unknown') }}
{% endif %}

---
{% endfor %}

## Summary

This documentation was generated by AgentOps on {{ generated_at }}.

**Total Sections:** {{ sections|length }}  
**Total Pages:** {{ total_pages }}
            """,
            
            "minimal_markdown": """
# {{ project_name }}

Generated: {{ generated_at }}

{% for section in sections %}
## {{ section.title }}

{{ section.content }}

{% endfor %}
            """,
            
            "detailed_markdown": """
# {{ project_name }} - Comprehensive Documentation

**Project Details:**
- **Version:** {{ version }}
- **Generated:** {{ generated_at }}
- **Author:** {{ author }}
- **Company:** {{ company }}

## Executive Summary

This document provides comprehensive documentation for the {{ project_name }} project, including requirements analysis, test coverage, and implementation details.

## Table of Contents

{% for section in sections %}
{{ "  " * (section.level - 1) }}1. [{{ section.title }}](#{{ section.id }})
{% endfor %}

{% for section in sections %}
## {{ section.title }}

{{ section.content }}

{% if section.metadata %}
### Section Details
- **Status:** {{ section.metadata.get('status', 'unknown') }}
- **Last Updated:** {{ section.metadata.get('last_updated', 'unknown') }}
- **Dependencies:** {{ section.metadata.get('dependencies', [])|join(', ') }}
{% endif %}

{% if section.metadata.get('code_samples') %}
### Code Examples
{% for sample in section.metadata.get('code_samples', []) %}
```{{ sample.language }}
{{ sample.code }}
```
{% endfor %}
{% endif %}

---
{% endfor %}

## Appendices

### A. Project Statistics
- **Total Sections:** {{ sections|length }}
- **Total Pages:** {{ total_pages }}
- **Coverage:** {{ coverage_percentage }}%

### B. Change History
{% for change in change_history %}
- **{{ change.date }}:** {{ change.description }}
{% endfor %}
            """,
            
            "html_template": """
<!DOCTYPE html>
<html>
<head>
    <title>{{ project_name }} Documentation</title>
    <meta charset="utf-8">
    <style>
        body {{ font-family: Arial, sans-serif; margin: 40px; line-height: 1.6; }}
        .header {{ background-color: #f8f9fa; padding: 20px; border-radius: 5px; margin-bottom: 30px; }}
        .toc {{ background-color: #e9ecef; padding: 15px; border-radius: 5px; margin-bottom: 30px; }}
        .section {{ margin-bottom: 30px; }}
        .metadata {{ background-color: #f8f9fa; padding: 10px; border-radius: 3px; margin: 10px 0; }}
        code {{ background-color: #f1f3f4; padding: 2px 4px; border-radius: 3px; }}
        pre {{ background-color: #f8f9fa; padding: 15px; border-radius: 5px; overflow-x: auto; }}
        .footer {{ margin-top: 50px; padding-top: 20px; border-top: 1px solid #dee2e6; color: #6c757d; }}
    </style>
</head>
<body>
    <div class="header">
        <h1>{{ project_name }}</h1>
        <p><strong>Version:</strong> {{ version }} | <strong>Generated:</strong> {{ generated_at }} | <strong>Author:</strong> {{ author }}</p>
    </div>
    
    <div class="toc">
        <h2>Table of Contents</h2>
        <ul>
        {% for section in sections %}
            <li><a href="#{{ section.id }}">{{ section.title }}</a></li>
        {% endfor %}
        </ul>
    </div>
    
    {% for section in sections %}
    <div class="section" id="{{ section.id }}">
        <h2>{{ section.title }}</h2>
        {{ section.content|safe }}
        
        {% if section.metadata %}
        <div class="metadata">
            <strong>Status:</strong> {{ section.metadata.get('status', 'unknown') }} | 
            <strong>Last Updated:</strong> {{ section.metadata.get('last_updated', 'unknown') }}
        </div>
        {% endif %}
    </div>
    {% endfor %}
    
    <div class="footer">
        <p>Generated by AgentOps on {{ generated_at }}. Total sections: {{ sections|length }}.</p>
    </div>
</body>
</html>
            """
        }
        
        return templates
    
    def create_documentation_project(self, project_id: str, config: DocumentationConfig) -> DocumentationProject:
        """Create a new documentation project.
        
        Args:
            project_id: Project identifier
            config: Documentation configuration
            
        Returns:
            New documentation project
        """
        if not is_feature_enabled("documentation_export"):
            raise RuntimeError("Documentation export feature is not enabled")
        
        project = DocumentationProject(project_id=project_id, config=config)
        return project
    
    def add_section(self, project: DocumentationProject, section: DocumentationSection) -> bool:
        """Add a section to the documentation project.
        
        Args:
            project: Documentation project
            section: Section to add
            
        Returns:
            True if section was added, False otherwise
        """
        if not is_feature_enabled("documentation_export"):
            return False
        
        project.sections[section.id] = section
        project.total_sections = len(project.sections)
        project.updated_at = datetime.now()
        
        return True
    
    def export_documentation(self, project: DocumentationProject, 
                           format_type: ExportFormat = ExportFormat.MARKDOWN,
                           output_path: Optional[str] = None) -> str:
        """Export documentation to specified format.
        
        Args:
            project: Documentation project
            format_type: Export format
            output_path: Output file path
            
        Returns:
            Path to exported file
        """
        if not is_feature_enabled("documentation_export"):
            raise RuntimeError("Documentation export feature is not enabled")
        
        # Check tier access for export formats
        current_tier = pricing_manager.current_tier
        
        if format_type == ExportFormat.PDF and current_tier == PricingTier.DEVELOPER:
            raise RuntimeError("PDF export not available in developer tier")
        
        if format_type == ExportFormat.DOCX and current_tier in [PricingTier.DEVELOPER, PricingTier.PROFESSIONAL]:
            raise RuntimeError("DOCX export not available in current tier")
        
        # Generate output path
        if not output_path:
            timestamp = datetime.now().strftime("%Y%m%d_%H%M%S")
            output_path = f"{project.config.filename_template.format(timestamp=timestamp)}.{format_type.value}"
            output_path = Path(project.config.output_directory) / output_path
        
        # Create output directory
        Path(output_path).parent.mkdir(parents=True, exist_ok=True)
        
        # Export based on format
        if format_type == ExportFormat.MARKDOWN:
            return self._export_markdown(project, output_path)
        elif format_type == ExportFormat.HTML:
            return self._export_html(project, output_path)
        elif format_type == ExportFormat.PDF:
            return self._export_pdf(project, output_path)
        elif format_type == ExportFormat.DOCX:
            return self._export_docx(project, output_path)
        elif format_type == ExportFormat.JSON:
            return self._export_json(project, output_path)
        elif format_type == ExportFormat.YAML:
            return self._export_yaml(project, output_path)
        else:
            raise ValueError(f"Unsupported format: {format_type}")
    
    def _export_markdown(self, project: DocumentationProject, output_path: str) -> str:
        """Export documentation to Markdown format.
        
        Args:
            project: Documentation project
            output_path: Output file path
            
        Returns:
            Path to exported file
        """
        # Prepare template data
        template_data = {
            "project_name": project.config.project_name,
            "version": project.config.version,
            "author": project.config.author,
            "company": project.config.company,
            "generated_at": datetime.now().strftime("%Y-%m-%d %H:%M:%S"),
            "sections": sorted(project.sections.values(), key=lambda x: x.order),
            "total_pages": project.total_pages,
            "coverage_percentage": 85.0,  # Placeholder
            "change_history": []  # Placeholder
        }
        
        # Select template based on config
        template_name = f"{project.config.template.value}_markdown"
        template_content = self.templates.get(template_name, self.templates["professional_markdown"])
        
        # Render template
        template = Template(template_content)
        content = template.render(**template_data)
        
        # Write to file
        with open(output_path, 'w', encoding='utf-8') as f:
            f.write(content)
        
        return str(output_path)
    
    def _export_html(self, project: DocumentationProject, output_path: str) -> str:
        """Export documentation to HTML format.
        
        Args:
            project: Documentation project
            output_path: Output file path
            
        Returns:
            Path to exported file
        """
        # Prepare template data
        template_data = {
            "project_name": project.config.project_name,
            "version": project.config.version,
            "author": project.config.author,
            "generated_at": datetime.now().strftime("%Y-%m-%d %H:%M:%S"),
            "sections": sorted(project.sections.values(), key=lambda x: x.order)
        }
        
        # Render HTML template
        template_content = self.templates["html_template"]
        template = Template(template_content)
        html_content = template.render(**template_data)
        
        # Write to file
        with open(output_path, 'w', encoding='utf-8') as f:
            f.write(html_content)
        
        return str(output_path)
    
    def _export_pdf(self, project: DocumentationProject, output_path: str) -> str:
        """Export documentation to PDF format.
        
        Args:
            project: Documentation project
            output_path: Output file path
            
        Returns:
            Path to exported file
        """
        # First generate HTML
        html_path = output_path.replace('.pdf', '.html')
        html_path = self._export_html(project, html_path)
        
        # Convert HTML to PDF
        try:
            HTML(filename=html_path).write_pdf(output_path)
        except Exception as e:
            logging.error(f"Failed to convert HTML to PDF: {e}")
            raise RuntimeError(f"PDF generation failed: {e}")
        
        # Clean up temporary HTML file
        Path(html_path).unlink(missing_ok=True)
        
        return str(output_path)
    
    def _export_docx(self, project: DocumentationProject, output_path: str) -> str:
        """Export documentation to DOCX format.
        
        Args:
            project: Documentation project
            output_path: Output file path
            
        Returns:
            Path to exported file
        """
        # Create Word document
        doc = Document()
        
        # Add title
        doc.add_heading(project.config.project_name, 0)
        
        # Add metadata
        doc.add_paragraph(f"Version: {project.config.version}")
        doc.add_paragraph(f"Generated: {datetime.now().strftime('%Y-%m-%d %H:%M:%S')}")
        doc.add_paragraph(f"Author: {project.config.author}")
        
        # Add sections
        for section in sorted(project.sections.values(), key=lambda x: x.order):
            doc.add_heading(section.title, level=1)
            doc.add_paragraph(section.content)
            
            if section.metadata:
                doc.add_paragraph(f"Status: {section.metadata.get('status', 'unknown')}")
        
        # Save document
        doc.save(output_path)
        
        return str(output_path)
    
    def _export_json(self, project: DocumentationProject, output_path: str) -> str:
        """Export documentation to JSON format.
        
        Args:
            project: Documentation project
            output_path: Output file path
            
        Returns:
            Path to exported file
        """
        # Prepare data for JSON export
        export_data = {
            "project": {
                "id": project.project_id,
                "name": project.config.project_name,
                "version": project.config.version,
                "author": project.config.author,
                "created_at": project.created_at.isoformat(),
                "updated_at": project.updated_at.isoformat()
            },
            "config": {
                "template": project.config.template.value,
                "formats": [fmt.value for fmt in project.config.formats],
                "sections": project.config.sections
            },
            "sections": {
                section_id: {
                    "title": section.title,
                    "content": section.content,
                    "level": section.level,
                    "order": section.order,
                    "status": section.status,
                    "metadata": section.metadata,
                    "last_updated": section.last_updated.isoformat()
                }
                for section_id, section in project.sections.items()
            },
            "statistics": {
                "total_sections": project.total_sections,
                "total_pages": project.total_pages
            }
        }
        
        # Write JSON file
        with open(output_path, 'w', encoding='utf-8') as f:
            json.dump(export_data, f, indent=2, ensure_ascii=False)
        
        return str(output_path)
    
    def _export_yaml(self, project: DocumentationProject, output_path: str) -> str:
        """Export documentation to YAML format.
        
        Args:
            project: Documentation project
            output_path: Output file path
            
        Returns:
            Path to exported file
        """
        # Prepare data for YAML export (same as JSON)
        export_data = {
            "project": {
                "id": project.project_id,
                "name": project.config.project_name,
                "version": project.config.version,
                "author": project.config.author,
                "created_at": project.created_at.isoformat(),
                "updated_at": project.updated_at.isoformat()
            },
            "config": {
                "template": project.config.template.value,
                "formats": [fmt.value for fmt in project.config.formats],
                "sections": project.config.sections
            },
            "sections": {
                section_id: {
                    "title": section.title,
                    "content": section.content,
                    "level": section.level,
                    "order": section.order,
                    "status": section.status,
                    "metadata": section.metadata,
                    "last_updated": section.last_updated.isoformat()
                }
                for section_id, section in project.sections.items()
            },
            "statistics": {
                "total_sections": project.total_sections,
                "total_pages": project.total_pages
            }
        }
        
        # Write YAML file
        with open(output_path, 'w', encoding='utf-8') as f:
            yaml.dump(export_data, f, default_flow_style=False, allow_unicode=True)
        
        return str(output_path)
    
    def get_available_formats(self) -> List[ExportFormat]:
        """Get available export formats for current tier.
        
        Returns:
            List of available export formats
        """
        if not is_feature_enabled("documentation_export"):
            return []
        
        current_tier = pricing_manager.current_tier
        available_formats = [ExportFormat.MARKDOWN, ExportFormat.HTML, ExportFormat.JSON]
        
        if current_tier in [PricingTier.PROFESSIONAL, PricingTier.TEAM, PricingTier.ENTERPRISE]:
            available_formats.append(ExportFormat.PDF)
        
        if current_tier in [PricingTier.TEAM, PricingTier.ENTERPRISE]:
            available_formats.append(ExportFormat.DOCX)
        
        if current_tier == PricingTier.ENTERPRISE:
            available_formats.append(ExportFormat.YAML)
        
        return available_formats


# Global documentation exporter instance
_documentation_exporter: Optional[DocumentationExporter] = None


def get_documentation_exporter() -> DocumentationExporter:
    """Get the global documentation exporter instance.
    
    Returns:
        DocumentationExporter instance
    """
    global _documentation_exporter
    if _documentation_exporter is None:
        _documentation_exporter = DocumentationExporter()
    return _documentation_exporter


def create_documentation_project(project_id: str, config: DocumentationConfig) -> DocumentationProject:
    """Create a new documentation project.
    
    Args:
        project_id: Project identifier
        config: Documentation configuration
        
    Returns:
        New documentation project
    """
    return get_documentation_exporter().create_documentation_project(project_id, config)


def add_documentation_section(project: DocumentationProject, section: DocumentationSection) -> bool:
    """Add a section to the documentation project.
    
    Args:
        project: Documentation project
        section: Section to add
        
    Returns:
        True if section was added, False otherwise
    """
    return get_documentation_exporter().add_section(project, section)


def export_documentation(project: DocumentationProject, 
                        format_type: ExportFormat = ExportFormat.MARKDOWN,
                        output_path: Optional[str] = None) -> str:
    """Export documentation to specified format.
    
    Args:
        project: Documentation project
        format_type: Export format
        output_path: Output file path
        
    Returns:
        Path to exported file
    """
    return get_documentation_exporter().export_documentation(project, format_type, output_path)


def get_available_documentation_formats() -> List[ExportFormat]:
    """Get available documentation export formats.
    
    Returns:
        List of available export formats
    """
    return get_documentation_exporter().get_available_formats() 