import numpy as np

from snuffled._core.compatibility import numba
from snuffled._core.utils.numba import geomean

from ..shared import fitting_cost
from ._fit_exact import fit_curve_exact_three_points
from ._helpers import initialize_params, param_step


# =================================================================================================
#  Find solution WITHOUT uncertainty - OPTIMAL FIT
# =================================================================================================
@numba.njit
def fit_curve(
    x: np.ndarray,
    fx: np.ndarray,
    range_a: tuple[float, float],
    range_b: tuple[float, float],
    range_c: tuple[float, float],
    reg: float,
    n_iters: int = 20,
    debug_flag: bool = False,
) -> tuple[float, float, float, float]:
    """
    Tailored approach to find approximately optimal initial parameters (according to 'fitting_cost' method)
    with provided bounds of b and c parameters.

    Prerequisites:
     - (x, fx)-values have been generated by...
        - generating x_delta-values using compute_x_deltas for a certain value of dx
        - generating fx-values by computing s1*f(root + s2*x_delta), with s1 = ±1 and s2 = ±2
        - scaling:
            - x-values are rescaled such that median(x)==1.0, i.e., by dividing by 2*dx
            - fx-values are rescaled such that their order of magnitude is ~1
     - fx-values can have any sign, however we assume...
        - not all fx-values are 0
        - the dominant sign of fx is positive, i.e. we expect positive a-values, hence range_a should be positive

    :return: (a,b,c,cost)-tuple of parameters that are considered optimal, with a, b, c in requested ranges.
    """

    # --- initial approximate solution --------------------
    a_opt, b_opt, c_opt, optimal_cost, _ = fit_curve_approx(x, fx, range_a, range_b, range_c, reg, debug_flag)

    # --- refine - step 2 - grid --------------------------
    for step_methods in (["a", "b", "c"], ["ac", "ba", "bc"], ["a", "b", "c"], ["ac", "ba", "bc"]):
        # First optimize over primary directions, then over the combined directions.
        # This should improve the odds of being able to take large steps in the combined directions in the 2nd phase.
        # In order to get robust results, we perform this routine twice.
        step_size = 1.0
        for i in range(n_iters):
            # start with step_size=1 and reduce with factor 2x each iteration
            if i > 0:
                step_size *= 0.5

            # take discrete steps
            for step_method in step_methods:
                for step_dir in [-1.0, 1.0]:
                    # compute candidate (a,b,c) values by taking a step from (a_opt, b_opt, c_opt)
                    a_cand, b_cand, c_cand = param_step(
                        a=a_opt,
                        b=b_opt,
                        c=c_opt,
                        method=step_method,
                        step_size=step_dir * step_size,
                        range_a=range_a,
                        range_b=range_b,
                        range_c=range_c,
                    )

                    # evaluate
                    if (a_cand != a_opt) or (b_cand != b_opt) or (c_cand != c_opt):
                        # only evaluate if this is an actually new set of parameters
                        # (can happen that there's no change if we're already at the boundary of the search space)
                        current_cost = fitting_cost(x, fx, a_cand, b_cand, c_cand, reg)
                        if current_cost < optimal_cost:
                            if debug_flag:
                                print(
                                    "Improved along ",
                                    step_size * step_dir,
                                    f"x {step_method}: ",
                                    optimal_cost,
                                    " --> ",
                                    current_cost,
                                )
                            a_opt = a_cand
                            b_opt = b_cand
                            c_opt = c_cand
                            optimal_cost = current_cost
                            break  # no need to explore the other step direction (back to where we came from)

    # --- return ------------------------------------------
    return a_opt, b_opt, c_opt, optimal_cost


# =================================================================================================
#  Find solution WITHOUT uncertainty - APPROXIMATE FIT
# =================================================================================================
@numba.njit
def fit_curve_approx(
    x: np.ndarray,
    fx: np.ndarray,
    range_a: tuple[float, float],
    range_b: tuple[float, float],
    range_c: tuple[float, float],
    reg: float,
    debug_flag: bool = False,
) -> tuple[float, float, float, float, str]:
    """
    Find approximately optimal solution (a,b,c) to curve fitting problem.

    A few different heuristical approaches are used, with the one with the lowest cost being selected.

    Tailored approach to find approximately optimal initial parameters (according to 'fitting_cost' method)
    with provided bounds of b and c parameters.

    Prerequisites:
     - (x, fx)-values have been generated by...
        - generating x_delta-values using compute_x_deltas for a certain value of dx
        - generating fx-values by computing s1*f(root + s2*x_delta), with s1 = ±1 and s2 = ±2
        - scaling:
            - x-values are rescaled such that median(x)==1.0, i.e., by dividing by 2*dx
            - fx-values are rescaled such that their order of magnitude is ~1
     - fx-values can have any sign, however we assume...
        - the dominant sign of fx is positive, i.e. we expect positive a-values, hence range_a should be positive

    :return: (a,b,c, cost, method)-tuple of parameters that are approximately optimal, with a, b, c in requested ranges.
               a,b,c:   optimal parameter values
               cost:    optimal cost of this solution
               method:  one of 'default', 'median_local', 'median_global', 'geomean_local', 'geomean_global'
    """

    # --- init --------------------------------------------
    n = len(fx)
    fx_group_1 = fx[: n // 3]
    fx_group_2 = fx[n // 3 : 2 * n // 3]
    fx_group_3 = fx[2 * n // 3 :]

    # --- default values ----------------------------------
    a_opt, b_opt, c_opt = initialize_params(range_a, range_b, range_c)
    optimal_cost = fitting_cost(x, fx, a_opt, b_opt, c_opt, reg)
    optimal_method = "default"

    if debug_flag:
        print(f"Method 'default'  --> cost=", optimal_cost)

    # --- heuristics --------------------------------------

    # prepare a few reference values
    fx_median = np.median(fx)
    fx_05_median = np.median(fx_group_1)
    fx_1_median = np.median(fx_group_2)
    fx_2_median = np.median(fx_group_3)

    fx_geomean = geomean(np.abs(fx))
    fx_05_geomean = geomean(np.abs(fx_group_1))
    fx_1_geomean = geomean(np.abs(fx_group_2))
    fx_2_geomean = geomean(np.abs(fx_group_3))

    # prepare 4 different combinations
    #
    # We use two aggregation methods:
    #    - median  : Most robust method that should theoretically work under all circumstances (any a,b,c-values),
    #                  but might be slightly less accurate because it doesn't truly average out multiple points.
    #    - geomean : Only expected to be accurate if b_true=0 (so no discontinuity).  But could be more accurate
    #                  under slightly noisy conditions, since it averages out data points in each group.
    #
    # We use two methods for generating fx_1:
    #    - local  : Only use the middle group of data points.
    #    - global : Use all data points.  Might be more accurate because more data points are taken into account,
    #                 but might be more subject to skewing effects under very noisy conditions.
    fx_05_values = [fx_05_median, fx_05_median, fx_05_geomean, fx_05_geomean]
    fx_1_values = [fx_1_median, fx_median, fx_1_geomean, fx_geomean]
    fx_2_values = [fx_2_median, fx_2_median, fx_2_geomean, fx_2_geomean]
    method_names = ["median_local", "median_global", "geomean_local", "geomean_global"]

    # try different combinations and select most optimal one
    for i in range(4):
        a_est, b_est, c_est = fit_curve_exact_three_points(
            fx_05=fx_05_values[i],
            fx_1=fx_1_values[i],
            fx_2=fx_2_values[i],
            range_a=range_a,
            range_b=range_b,
            range_c=range_c,
        )
        current_cost = fitting_cost(x, fx, a_est, b_est, c_est, reg)

        if debug_flag:
            print(f"Method '{method_names[i]}'  --> cost=", current_cost)

        if current_cost < optimal_cost:
            a_opt, b_opt, c_opt = a_est, b_est, c_est
            optimal_cost = current_cost
            optimal_method = method_names[i]

    # --- return all results ------------------------------
    return a_opt, b_opt, c_opt, optimal_cost, optimal_method
