"""Defines data structures for machine learning model runs.

This module provides Pydantic models to represent machine learning models
and the predictions generated when these models are applied to audio clips
within the soundevent package.
"""

import warnings
from typing import Any, Optional

from pydantic import BaseModel, computed_field, model_validator

from soundevent.data.prediction_sets import PredictionSet

__all__ = [
    "ModelRun",
    "Model",
    "ModelInfo",
]


class ModelInfo(BaseModel):
    """Contains descriptive metadata about a machine learning model.

    Attributes
    ----------
    name : str
        The unique name identifying the model.
    description : Optional[str]
        A brief description of the model's purpose, architecture, or training data.
    """

    name: str
    description: Optional[str] = None


class Model(BaseModel):
    """Represents a specific version of a machine learning model.

    This class combines the general information about a model with its
    specific version identifier.

    Attributes
    ----------
    info : ModelInfo
        Descriptive metadata about the model.
    version : str
        The specific version identifier for this instance of the model (e.g., "1.0.0", "v2-beta").
    """

    info: ModelInfo
    version: str


class ModelRun(PredictionSet):
    """Represents the set of predictions generated by a model run.

    A model run encapsulates the output of applying a specific machine
    learning model to a collection of audio clips. It inherits prediction
    data management from `PredictionSet` and adds information about the
    model used.

    Attributes
    ----------
    model : Model
        The specific model version used to generate the predictions in this run.
    """

    model: Model

    @computed_field
    @property
    def version(self) -> Optional[str]:
        """Provides backward-compatible access to 'version'. Deprecated."""
        warnings.warn(
            "Accessing 'version' directly on ModelRun is deprecated. "
            "Use 'model_run.model.version' instead.",
            DeprecationWarning,
            stacklevel=2,
        )
        return self.model.version

    @version.setter
    def version(self, value: str) -> None:
        warnings.warn(
            "Setting 'version' directly on ModelRun is deprecated. "
            "Set 'model_run.model.version' instead.",
            DeprecationWarning,
            stacklevel=2,
        )
        self.model.version = value

    @model_validator(mode="before")
    @classmethod
    def _fill_in_missing_model_info(cls, data: Any) -> Any:
        """Transform old flat input data to the new nested structure."""
        if not isinstance(data, dict):
            return data

        if "model" in data:
            return data

        old_keys = {"version"}
        if old_keys.issubset(data.keys()):
            warnings.warn(
                "Initializing ModelRun with ('version',) is deprecated. "
                "Provide data in the nested format: "
                "{'model': {'model_info': {'name':..., 'description':...}, 'version':...}}",
                DeprecationWarning,
                stacklevel=3,
            )

            return {
                **data,
                "model": {
                    "info": {
                        "name": data.get("name"),
                        "description": data.get("description"),
                    },
                    "version": data.get("version"),
                },
            }

        return data
