"""Prediction Sets.

This module defines the `PredictionSet` class, which is used to group
predictions made on a collection of audio clips. In bioacoustic analysis, it's
common to generate predictions (e.g., from a machine learning model or a manual
annotation process) for multiple clips. A `PredictionSet` serves as a
container to organize these related predictions.

Organizing predictions into sets allows for easier management, analysis, and
comparison. For instance, a `PredictionSet` might represent all the predictions
generated by a specific model version on a particular dataset, or a batch of
predictions reviewed by a specific annotator.
"""

import datetime
from typing import List, Optional
from uuid import UUID, uuid4

from pydantic import BaseModel, Field

from soundevent.data.clip_predictions import ClipPrediction

__all__ = ["PredictionSet"]


class PredictionSet(BaseModel):
    """A collection of predictions for multiple audio clips.

    The `PredictionSet` class groups together `ClipPrediction` objects,
    typically representing the output of an analysis process (like a model
    inference run) applied to a set of audio clips. This allows managing these
    predictions as a single unit.

    Attributes
    ----------
    uuid
        A unique identifier for the prediction set, automatically generated.
    clip_predictions
        A list of `ClipPrediction` objects, where each object contains the
        predictions associated with a specific audio clip within this set.
    name
        An optional name for the prediction set (e.g., "Model V2 - Test Set").
    description
        An optional detailed description of the prediction set, providing
        context about its origin or purpose.
    created_on
        The date and time when the `PredictionSet` object was created.
    """

    uuid: UUID = Field(default_factory=uuid4)
    clip_predictions: List[ClipPrediction] = Field(default_factory=list)
    name: Optional[str] = None
    description: Optional[str] = None

    created_on: datetime.datetime = Field(
        default_factory=datetime.datetime.now
    )
