"""
Base formatter for all Devix report generators.

This module provides the abstract base class that all formatters inherit from,
defining the common interface and shared utilities for generating reports
in different formats (markdown, text, HTML, etc.).
"""

import logging
from abc import ABC, abstractmethod
from datetime import datetime
from pathlib import Path
from typing import Any, Dict, List, Optional, Union


class BaseFormatter(ABC):
    """Abstract base class for all report formatters in Devix."""
    
    def __init__(self, project_path: str = "../", logger: Optional[logging.Logger] = None):
        """
        Initialize the base formatter.
        
        Args:
            project_path: Path to the project being analyzed
            logger: Optional logger instance
        """
        self.project_path = Path(project_path).resolve()
        self.project_name = self.project_path.name
        self.logger = logger or self._setup_logger()
        self.generation_time = datetime.now()
        
    def _setup_logger(self) -> logging.Logger:
        """Set up logger for the formatter."""
        logger = logging.getLogger(f"devix.reporting.{self.__class__.__name__}")
        if not logger.handlers:
            handler = logging.StreamHandler()
            formatter = logging.Formatter(
                '%(asctime)s - %(name)s - %(levelname)s - %(message)s'
            )
            handler.setFormatter(formatter)
            logger.addHandler(handler)
            logger.setLevel(logging.INFO)
        return logger
    
    @abstractmethod
    def format_report(self, analysis_results: Dict[str, Any]) -> str:
        """
        Format the analysis results into a report string.
        
        Args:
            analysis_results: Dictionary containing all analysis data
            
        Returns:
            Formatted report as string
        """
        pass
    
    @abstractmethod
    def get_file_extension(self) -> str:
        """Return the file extension for this format (e.g., '.md', '.txt')."""
        pass
    
    def generate_header(self) -> str:
        """Generate a standard header for reports."""
        return self._format_header()
    
    def generate_footer(self) -> str:
        """Generate a standard footer for reports."""
        return self._format_footer()
    
    def generate_summary(self, analysis_results: Dict[str, Any]) -> str:
        """Generate a summary section from analysis results."""
        return self._format_summary(analysis_results)
    
    def generate_metrics_section(self, analysis_results: Dict[str, Any]) -> str:
        """Generate metrics section from analysis results."""
        return self._format_metrics(analysis_results)
    
    def save_report(self, content: str, output_path: Optional[Path] = None) -> Path:
        """
        Save the formatted report to a file.
        
        Args:
            content: The formatted report content
            output_path: Optional custom output path
            
        Returns:
            Path to the saved file
        """
        if output_path is None:
            timestamp = self.generation_time.strftime("%Y%m%d_%H%M%S")
            filename = f"devix_report_{timestamp}{self.get_file_extension()}"
            output_path = self.project_path / filename
        
        try:
            output_path.parent.mkdir(parents=True, exist_ok=True)
            with open(output_path, 'w', encoding='utf-8') as f:
                f.write(content)
            
            self.logger.info(f"Report saved to: {output_path}")
            return output_path
            
        except Exception as e:
            self.logger.error(f"Failed to save report: {e}")
            raise
    
    def _format_header(self) -> str:
        """Default header formatting - can be overridden by subclasses."""
        timestamp = self.generation_time.strftime("%Y-%m-%d %H:%M:%S")
        return f"Devix Analysis Report\nProject: {self.project_name}\nGenerated: {timestamp}\n"
    
    def _format_footer(self) -> str:
        """Default footer formatting - can be overridden by subclasses."""
        return f"\nReport generated by Devix - Modular Code Analysis Tool"
    
    def _format_summary(self, analysis_results: Dict[str, Any]) -> str:
        """Default summary formatting - can be overridden by subclasses."""
        summary_lines = ["ANALYSIS SUMMARY"]
        
        # Extract key metrics from different analyzers
        total_issues = 0
        analyzers_run = []
        
        for analyzer_name, results in analysis_results.items():
            if isinstance(results, dict) and 'issues' in results:
                issues_count = len(results.get('issues', []))
                total_issues += issues_count
                analyzers_run.append(f"{analyzer_name}: {issues_count} issues")
        
        summary_lines.extend([
            f"Total Issues Found: {total_issues}",
            f"Analyzers Run: {len(analyzers_run)}",
            ""
        ])
        
        if analyzers_run:
            summary_lines.extend(analyzers_run)
        
        return "\n".join(summary_lines)
    
    def _format_metrics(self, analysis_results: Dict[str, Any]) -> str:
        """Default metrics formatting - can be overridden by subclasses."""
        metrics_lines = ["METRICS"]
        
        for analyzer_name, results in analysis_results.items():
            if isinstance(results, dict) and 'metrics' in results:
                metrics = results['metrics']
                if metrics:
                    metrics_lines.append(f"\n{analyzer_name.upper()} METRICS:")
                    for key, value in metrics.items():
                        if isinstance(value, (int, float)):
                            metrics_lines.append(f"  {key}: {value}")
                        else:
                            metrics_lines.append(f"  {key}: {str(value)}")
        
        return "\n".join(metrics_lines) if len(metrics_lines) > 1 else ""
    
    def _extract_issues_by_severity(self, analysis_results: Dict[str, Any]) -> Dict[str, List[str]]:
        """Extract and categorize issues by severity level."""
        severity_issues = {
            'critical': [],
            'high': [],
            'medium': [],
            'low': [],
            'info': []
        }
        
        for analyzer_name, results in analysis_results.items():
            if isinstance(results, dict) and 'issues' in results:
                issues = results.get('issues', [])
                for issue in issues:
                    # Determine severity based on issue content or explicit severity
                    severity = self._determine_issue_severity(issue, analyzer_name)
                    if severity in severity_issues:
                        severity_issues[severity].append(f"[{analyzer_name}] {issue}")
        
        return severity_issues
    
    def _determine_issue_severity(self, issue: str, analyzer_name: str) -> str:
        """Determine issue severity based on content and analyzer type."""
        issue_lower = issue.lower()
        
        # Security issues are generally high priority
        if analyzer_name == 'security':
            if any(word in issue_lower for word in ['critical', 'high', 'severe']):
                return 'critical'
            elif any(word in issue_lower for word in ['medium', 'moderate']):
                return 'medium'
            return 'high'
        
        # Performance issues vary
        elif analyzer_name == 'performance':
            if any(word in issue_lower for word in ['critical', 'severe', 'blocking']):
                return 'high'
            elif any(word in issue_lower for word in ['slow', 'inefficient']):
                return 'medium'
            return 'low'
        
        # Quality issues are generally medium priority
        elif analyzer_name == 'quality':
            if any(word in issue_lower for word in ['error', 'critical', 'fatal']):
                return 'high'
            elif any(word in issue_lower for word in ['warning', 'style']):
                return 'low'
            return 'medium'
        
        # Test issues
        elif analyzer_name == 'test':
            if any(word in issue_lower for word in ['failed', 'error', 'broken']):
                return 'high'
            elif any(word in issue_lower for word in ['missing', 'coverage']):
                return 'medium'
            return 'low'
        
        # Default severity
        return 'medium'
    
    def _calculate_project_health_score(self, analysis_results: Dict[str, Any]) -> float:
        """Calculate an overall project health score (0-100)."""
        try:
            total_weight = 0
            weighted_score = 0
            
            # Define weights for different analyzers
            analyzer_weights = {
                'security': 0.3,
                'quality': 0.25,
                'test': 0.25,
                'performance': 0.2
            }
            
            for analyzer_name, results in analysis_results.items():
                if isinstance(results, dict) and analyzer_name in analyzer_weights:
                    weight = analyzer_weights[analyzer_name]
                    total_weight += weight
                    
                    # Calculate score based on issues found
                    issues_count = len(results.get('issues', []))
                    
                    # Score decreases with more issues
                    if issues_count == 0:
                        analyzer_score = 100
                    elif issues_count <= 5:
                        analyzer_score = 90 - (issues_count * 10)
                    elif issues_count <= 15:
                        analyzer_score = 50 - ((issues_count - 5) * 3)
                    else:
                        analyzer_score = max(0, 20 - (issues_count - 15))
                    
                    weighted_score += analyzer_score * weight
            
            if total_weight > 0:
                return round(weighted_score / total_weight, 1)
            else:
                return 50.0  # Default neutral score
                
        except Exception as e:
            self.logger.debug(f"Error calculating health score: {e}")
            return 50.0
    
    def _format_file_statistics(self, analysis_results: Dict[str, Any]) -> str:
        """Format file statistics from analysis results."""
        stats_lines = ["FILE STATISTICS"]
        
        # Extract file counts from different analyzers
        total_files = set()
        analyzed_files = set()
        skipped_files = set()
        
        for analyzer_name, results in analysis_results.items():
            if isinstance(results, dict):
                if 'analyzed_files' in results:
                    analyzed_files.update(results['analyzed_files'])
                if 'total_files' in results:
                    total_files.update(results['total_files'])
                if 'skipped_files' in results:
                    skipped_files.update(results['skipped_files'])
        
        stats_lines.extend([
            f"Total Files in Project: {len(total_files)}",
            f"Files Analyzed: {len(analyzed_files)}",
            f"Files Skipped: {len(skipped_files)}",
        ])
        
        return "\n".join(stats_lines)
