from __future__ import annotations

from pathlib import Path
from typing import Set
from pydantic import BaseModel

import fastworkflow

from fastworkflow.command_routing import RoutingDefinition
from fastworkflow.train.generate_synthetic import generate_diverse_utterances
class Signature:
    class Output(BaseModel):
        valid_command_names: list[str]

    # Constants from plain_utterances.json
    plain_utterances = [
        "show me all the utterances",
        "what can you do?",
        "what are my options?",
        "what are my choices?",
        "what are my capabilities?",
        "what can i do?",
        "what can i use?",
        "what are my tools?",
        "now what?",
        "list commands",
        "list utterances"
    ]

    @staticmethod
    def generate_utterances(workflow: fastworkflow.Workflow, command_name: str) -> list[str]:
        return [
            command_name.split('/')[-1].lower().replace('_', ' ')
        ] + generate_diverse_utterances(Signature.plain_utterances, command_name)


class ResponseGenerator:
    def _process_command(
        self, workflow: fastworkflow.Workflow
    ) -> Signature.Output:
        """
        Provides helpful information about this type of work-item.
        If the workitem_path is not provided, it provides information about the current work-item.

        :param input: The input parameters for the function.
        """
        app_workflow = workflow.context["app_workflow"]
        subject_crd = fastworkflow.RoutingRegistry.get_definition(
            app_workflow.folderpath)
        
        crd = fastworkflow.RoutingRegistry.get_definition(
            workflow.folderpath)
        cme_command_names = crd.get_command_names('IntentDetection')

        fully_qualified_command_names = (
            set(cme_command_names) | 
            set(subject_crd.get_command_names(app_workflow.current_command_context_name))
        ) - {'wildcard'}

        valid_command_names = [
            fully_qualified_command_name.split('/')[-1] 
            for fully_qualified_command_name in fully_qualified_command_names
        ]

        return Signature.Output(valid_command_names=sorted(valid_command_names))

    def __call__(
        self,
        workflow: fastworkflow.Workflow,
        command: str,
    ) -> fastworkflow.CommandOutput:
        output = self._process_command(workflow)

        app_workflow = workflow.context["app_workflow"]
        context_name = (
            'global' if app_workflow.current_command_context_name == '*'
            else app_workflow.current_command_context_name
        )

        response = "\n".join([f"{cmd}" for cmd in output.valid_command_names])
        response = (
            f"Commands available in the current context ({context_name}):\n"
            f"{response}\n"
        )

        return fastworkflow.CommandOutput(
            workflow_id=workflow.id,
            command_responses=[
                fastworkflow.CommandResponse(
                    response=response,
                )
            ]
        ) 
