"""
Component models for the Konigle SDK.

This module provides models for component resources including creation,
update, and resource representations.
"""

from typing import Any, ClassVar, Dict, Optional, Set

from pydantic import BaseModel, Field

from konigle.models.base import CreateModel, TimestampedResource, UpdateModel


class BaseComponent(BaseModel):
    """
    Base class for component models with common editable fields.

    Contains fields that can be set during creation/update.
    """

    name: str = Field(
        ...,
        max_length=255,
        title="Name",
        description="Name given by the user for display and search purposes",
    )
    """Name given by the user for display and search purposes."""

    description: str = Field(
        default="",
        title="Description",
        description="Description of the component",
    )
    """Description of the component."""

    version: str = Field(
        default="1.0.0",
        max_length=10,
        title="Version",
        description="Version of the component",
    )
    """Version of the component."""

    template_html: str = Field(
        default="",
        title="Template HTML",
        description="Django template for the component",
    )
    """Django template for custom components."""


class Component(BaseComponent, TimestampedResource):
    """
    Component resource model.

    Represents a reusable component for building landing pages and templates.
    """

    # Metadata for field loading behavior
    _detail_only_fields: ClassVar[Set[str]] = {
        "template_html",
    }
    _foreign_key_fields: ClassVar[Set[str]] = set()

    type: str = Field(
        default="fluid",
        max_length=30,
        title="Type",
        description="Type of component: fluid, dynamic, or widget",
    )
    """Type of component: fluid, dynamic, or widget."""

    # Not validating. We will send the valid context for the custom HTML
    # component during the create time
    context: Dict[str, Any] = Field(
        default_factory=dict,
        title="Context",
        description="Component node context for fluid components",
    )
    """Component node context for fluid components."""

    thumbnail: Optional[str] = Field(
        None,
        title="Thumbnail",
        description="URL of the component thumbnail image. Generated by the "
        "system",
    )
    """URL of the component thumbnail image. Generated by the system."""

    thumbnail_width: Optional[int] = Field(
        None,
        title="Thumbnail Width",
        description="Width of the thumbnail image in pixels",
    )
    """Width of the thumbnail image in pixels."""

    thumbnail_height: Optional[int] = Field(
        None,
        title="Thumbnail Height",
        description="Height of the thumbnail image in pixels",
    )
    """Height of the thumbnail image in pixels."""

    def __str__(self) -> str:
        return f"Component(id = {self.id}, name = {self.name})"


class ComponentCreate(BaseComponent, CreateModel):
    """
    Model for creating a new component.

    Contains all required and optional fields for component creation.
    """

    pass


class ComponentUpdate(UpdateModel):
    """
    Model for updating an existing component.

    All fields are optional for partial updates.
    """

    name: Optional[str] = Field(
        None,
        max_length=255,
        title="Name",
        description="Name given by the user for display and search purposes",
    )
    """Name given by the user for display and search purposes."""

    description: Optional[str] = Field(
        None,
        title="Description",
        description="Description of the component",
    )
    """Description of the component."""

    version: Optional[str] = Field(
        None,
        max_length=10,
        title="Version",
        description="Version of the component",
    )
    """Version of the component."""

    template_html: Optional[str] = Field(
        None,
        title="Template HTML",
        description="Django template for custom components",
    )
    """Django template for custom components."""


__all__ = [
    "Component",
    "ComponentCreate",
    "ComponentUpdate",
]
